/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>
#include <flAudioMixer/AudioMixer.h>
#include <flAudioMixer/flAudioInputPin.h>
#include <flAudioMixer/flAudioBuffer.h>
#include <math.h>
#include <limits.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue

//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
    &MEDIATYPE_Audio,		// MajorType
    &MEDIASUBTYPE_PCM		// MinorType
};
//------------------------------------------------------------------------------
const AMOVIESETUP_PIN sudPins [] =
{
	{
		L"Input",			// String pin name
		FALSE,				// Is it rendered
		FALSE,				// Is it an output
		FALSE,				// Allowed none
		FALSE,				// Allowed many
		&CLSID_NULL,		// Connects to filter
		L"Output",			// Connects to pin
		1,					// Number of types
		&sudPinTypes		// The pin details
	},
	{
		L"Output",			// String pin name
		FALSE,				// Is it rendered
		TRUE,				// Is it an output
		FALSE,				// Allowed none
		FALSE,				// Allowed many
		&CLSID_NULL,		// Connects to filter
		L"Input",			// Connects to pin
		1,					// Number of types
		&sudPinTypes		// The pin details
	},
	{
		L"Input Audio",		// Pins string name
		FALSE,				// Is it rendered
		FALSE,				// Is it an output
		FALSE,				// Are we allowed none
		FALSE,				// And allowed many
		&CLSID_NULL,		// Connects to filter
		NULL,				// Connects to pin
		1,					// Number of types
		&sudPinTypes		// Pin information
	}
};
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudAudioMixer =
{
    &CLSID_AudioMixer,		// Filter CLSID
    L"NIME Audio Mixer",	// String name
    MERIT_DO_NOT_USE,		// Filter merit
    3,						// Number of pins
    sudPins					// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[1] = {
    { L"AudioMixer"
    , &CLSID_AudioMixer
    , CAudioMixer::CreateInstance
    , NULL
    , &sudAudioMixer }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CAudioMixer::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
	CAudioMixer *pNewObject = new CAudioMixer(NAME("AudioMixer"), punk, phr);
	if (pNewObject == NULL)
	{
		*phr = E_OUTOFMEMORY;
	}
	return pNewObject;
}
//------------------------------------------------------------------------------
CAudioMixer::CAudioMixer(TCHAR *tszName, LPUNKNOWN punk, HRESULT *phr) :
CTransInPlaceFilter(tszName, punk, CLSID_AudioMixer, phr),
CPersistStream(punk, phr)
{
	_runFlag		= false;
	_mixingTime		= 0;
	_sampleTime		= 0;

	_audioInputPin	= NULL;

	_audioBuffer	= new flAudioBuffer();
	_audioBuffer->setBufferSize(44100 * 2 * 2 * 10); // 44.1(kHz) 16(bit) 2(ch) -> 10(sec)

	_bufferSize		= 0;
	_buffer			= NULL;
}
//------------------------------------------------------------------------------
CAudioMixer::~CAudioMixer()
{
	delete _audioInputPin;
	delete _audioBuffer;
	delete _buffer;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CAudioMixer::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
	CheckPointer(ppv, E_POINTER);

	if (riid == IID_IAudioMixer)
	{
		return GetInterface((IAudioMixer *)this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
		return GetInterface((ISpecifyPropertyPages *)this, ppv);
	}
	else
	{
		return CTransInPlaceFilter::NonDelegatingQueryInterface(riid, ppv);
	}
}
//------------------------------------------------------------------------------
// CTransInPlaceFilter Overrides
int
CAudioMixer::GetPinCount()
{
	return 3;
}
//------------------------------------------------------------------------------
CBasePin*
CAudioMixer::GetPin(int n)
{
	if (m_pInput == NULL || m_pOutput == NULL || _audioInputPin == NULL) 
	{
		HRESULT hr = S_OK;

		m_pInput = new CTransInPlaceInputPin(NAME("AudioMixer input pin"),
											this, &hr, L"Input");

		if (FAILED(hr) || m_pInput == NULL) 
			goto PINCREATEERROR;

		m_pOutput = new CTransInPlaceOutputPin(NAME("AudioMixer output pin"),
											this,	&hr, L"Output");

		if (FAILED(hr) || m_pOutput == NULL) 
			goto PINCREATEERROR;

		_audioInputPin = new CAudioInputPin(NAME("Audio input pin"),
											this, &hr, L"AudioIn");

		if (FAILED(hr) || _audioInputPin == NULL)
			goto PINCREATEERROR;
	}

	switch(n) 
	{
	case 0:
		return m_pInput;
	case 1:
		return m_pOutput;
	case 2:
		return _audioInputPin;
	}

	return NULL;

PINCREATEERROR:
	delete m_pInput;		m_pInput		= NULL;
	delete m_pOutput;		m_pOutput		= NULL;
	delete _audioInputPin;	_audioInputPin	= NULL;
	return NULL;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CAudioMixer::FindPin(LPCWSTR Id, IPin **ppPin)
{
	CheckPointer(ppPin, E_POINTER);
	ValidateReadWritePtr(ppPin, sizeof(IPin *));

	if (!lstrcmpW(Id, L"In"))
	{
		*ppPin = GetPin(0);
	}
	else if (!lstrcmpW(Id, L"Out"))
	{
		*ppPin = GetPin(1);
	}
	else if (!lstrcmpW(Id, L"AudioIn"))
	{
		*ppPin = GetPin(2);
	}
	else
	{
		*ppPin = NULL;
		return VFW_E_NOT_FOUND;
	}

	HRESULT hr = NOERROR;

	if(*ppPin)
	{
		(*ppPin)->AddRef();
	}
	else
	{
		hr = E_OUTOFMEMORY;
	}

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CAudioMixer::Stop()
{
	_runFlag = false;
	return CTransInPlaceFilter::Stop();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CAudioMixer::Run(REFERENCE_TIME tStart)
{
	_runFlag = true;

	_audioBuffer->clear();
	return CTransInPlaceFilter::Run(tStart);
}
//------------------------------------------------------------------------------
HRESULT
CAudioMixer::CheckInputType(const CMediaType* mtIn)
{
	if (_audioInputPin->IsConnected() == FALSE)
	{
		if (IsEqualGUID(mtIn->majortype, MEDIATYPE_Audio) &&
			IsEqualGUID(mtIn->subtype, MEDIASUBTYPE_PCM) &&
			IsEqualGUID(mtIn->formattype, FORMAT_WaveFormatEx))
			return S_OK;
		else
			return VFW_E_TYPE_NOT_ACCEPTED;
	}

	WAVEFORMATEX *formatEx1 = (WAVEFORMATEX *)mtIn->Format();
	WAVEFORMATEX *formatEx2 = (WAVEFORMATEX *)_audioInputPin->CurrentMediaType().Format();

	if (formatEx1 == NULL || formatEx2 == NULL)
		return VFW_E_TYPE_NOT_ACCEPTED;

	if (formatEx1->wFormatTag == WAVE_FORMAT_PCM &&
		formatEx1->wFormatTag == formatEx2->wFormatTag &&
		formatEx1->nChannels == formatEx2->nChannels &&
		formatEx1->nSamplesPerSec == formatEx2->nSamplesPerSec &&
		formatEx1->wBitsPerSample == formatEx2->wBitsPerSample &&
		formatEx1->wFormatTag == formatEx2->wFormatTag)
		return S_OK;

	return VFW_E_TYPE_NOT_ACCEPTED;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
STDMETHODIMP
CAudioMixer::GetClassID(CLSID *pClsid)
{
	*pClsid = CLSID_AudioMixer;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CAudioMixer::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	flUInt bufferSize = _audioBuffer->getBufferSize();
	WRITEOUT(bufferSize);
	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CAudioMixer::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	flUInt bufferSize;
	READIN(bufferSize);
	_audioBuffer->setBufferSize(bufferSize);
	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CAudioMixer::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CAudioMixer::SizeMax()
{
	return 0;
}
//------------------------------------------------------------------------------
// IAudioMixer
STDMETHODIMP
CAudioMixer::get_BufferSize(flUInt *size)
{
	CAutoLock lock(&_filterLock);
	*size = _audioBuffer->getBufferSize();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CAudioMixer::put_BufferSize(flUInt size)
{
	CAutoLock lock(&_filterLock);
	_audioBuffer->setBufferSize(size);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CAudioMixer::is_Mixing(flBool *flag)
{
	*flag = (timeGetTime() - _mixingTime < _sampleTime + 100);
	return NOERROR;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages
STDMETHODIMP
CAudioMixer::GetPages(CAUUID *pPages)
{
	pPages->cElems = 0;
	pPages->pElems = NULL;
	return NOERROR;
}
//------------------------------------------------------------------------------
// CTransformInPlaceFilter Overrides
HRESULT
CAudioMixer::Transform(IMediaSample *pSample)
{
	flULong	dataLength;
	flByte*	data;
	dataLength = pSample->GetActualDataLength();
	pSample->GetPointer(&data);

	if (_bufferSize < dataLength)
	{
		delete _buffer;
		_buffer = new flByte[dataLength];
		_bufferSize = dataLength;
	}

	{
		CAutoLock lock(&_filterLock);
		if (!_audioBuffer->getData(_buffer, dataLength))
			return NOERROR;
	}

	WAVEFORMATEX *format = (WAVEFORMATEX *)m_pInput->CurrentMediaType().Format();

	flUShort blockAlign		= format->nBlockAlign;
	flUShort channels		= format->nChannels;
	flUShort bitsPerSample	= format->wBitsPerSample;
	flUInt samplePerSec		= format->nSamplesPerSec;

	flByte* ptr1 = data;
	flByte* ptr2 = _buffer;

	switch(bitsPerSample)
	{
	case 8:
		while(ptr1 < data + dataLength)
		{
			flInt v1 = (flInt)*ptr1;
			flInt v2 = (flInt)*ptr2;

			v1 -= 127;
			v2 -= 127;

			flInt v = v1 + v2;
			v += 127;

			if (255 < v)	v = 255;
			else if (v < 0)	v = 0;

			*ptr1 = (flByte)v;

			ptr1 ++;
			ptr2 ++;
		}
		break;
	case 16:
		while(ptr1 < data + dataLength)
		{
			flInt v1 = (flInt)(*((flShort *)ptr1));
			flInt v2 = (flInt)(*((flShort *)ptr2));

			flInt  v = v1 + v2;

			if (SHRT_MAX < v)		v = SHRT_MAX;
			else if (v < SHRT_MIN)	v = SHRT_MIN;

			*((flShort *)ptr1) = (flShort)v;

			ptr1 += 2;
			ptr2 += 2;
		}
		break;
	}

	_mixingTime = timeGetTime();
	_sampleTime = flULong(flFloat(samplePerSec) /
						flFloat(dataLength) /
						flFloat(blockAlign) * 1000.0f);

	return NOERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
HRESULT
CAudioMixer::ReceiveAudio(IMediaSample *pSource, CAudioInputPin *ppin)
{
	// Get Media Sample data
	flULong	dataLength;
	flByte*	data;
	dataLength = pSource->GetActualDataLength();
	pSource->GetPointer(&data);

	do
	{
		{
			CAutoLock lock(&_filterLock);
			if (_audioBuffer->setData(data, dataLength))
				break;
		}
		Sleep(1);

	} while(_runFlag);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CAudioMixer::CheckAudioInputType(const CMediaType* mtIn)
{
	if (m_pInput->IsConnected() == FALSE)
	{
		if (IsEqualGUID(mtIn->majortype, MEDIATYPE_Audio) &&
			IsEqualGUID(mtIn->subtype, MEDIASUBTYPE_PCM) &&
			IsEqualGUID(mtIn->formattype, FORMAT_WaveFormatEx))
			return S_OK;
		else
			return VFW_E_TYPE_NOT_ACCEPTED;
	}

	WAVEFORMATEX *formatEx1 = (WAVEFORMATEX *)mtIn->Format();
	WAVEFORMATEX *formatEx2 = (WAVEFORMATEX *)m_pInput->CurrentMediaType().Format();

	if (formatEx1 == NULL || formatEx2 == NULL)
		return VFW_E_TYPE_NOT_ACCEPTED;

	if (formatEx1->wFormatTag == WAVE_FORMAT_PCM &&
		formatEx1->wFormatTag == formatEx2->wFormatTag &&
		formatEx1->nChannels == formatEx2->nChannels &&
		formatEx1->nSamplesPerSec == formatEx2->nSamplesPerSec &&
		formatEx1->wBitsPerSample == formatEx2->wBitsPerSample &&
		formatEx1->wFormatTag == formatEx2->wFormatTag)
		return S_OK;

	return VFW_E_TYPE_NOT_ACCEPTED;
}
//------------------------------------------------------------------------------
