//------------------------------------------------------------------------------
#ifndef	__flHash_inl__
#define	__flHash_inl__
//------------------------------------------------------------------------------
#ifdef	_DEBUG
#include <flBase/flNotify.h>
#endif
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flHashMap<K, V, HF, CF>::flHashMap(flUInt tableSize) :
_tableSize(tableSize), _keyTable(NULL), _valueTable(NULL)
{
	if (_tableSize == 0)
		_tableSize = DEFAULT_TABLESIZE;
	_keyTable	= new flArray<K>[_tableSize];
	_valueTable	= new flArray<V>[_tableSize];
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flHashMap<K, V, HF, CF>::flHashMap(const flHashMap& hashMap)
{
	_tableSize	= hashMap._tableSize;
	_keyTable	= new flArray<K>[_tableSize];
	_valueTable	= new flArray<V>[_tableSize];

	for (flUInt i = 0; i < _tableSize; ++i)
	{
		_keyTable[i]	= hashMap._keyTable[i];
		_valueTable[i]	= hashMap._valueTable[i];
	}
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flHashMap<K, V, HF, CF>::~flHashMap()
{
	delete [] _keyTable;
	delete [] _valueTable;
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
void
flHashMap<K, V, HF, CF>::copy(const flHashMap& hashMap)
{
	if (this == &hashMap)
		return;

	clear();
	delete [] _keyTable;
	delete [] _valueTable;

	_tableSize	= hashMap._tableSize;
	_keyTable	= new flArray<K>[_tableSize];
	_valueTable	= new flArray<V>[_tableSize];

	for (flUInt i = 0; i < _tableSize; ++i)
	{
		_keyTable[i].copy(hashMap._keyTable[i]);
		_valueTable[i].copy(hashMap._valueTable[i]);
	}
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flBool
flHashMap<K, V, HF, CF>::equal(const flHashMap& hashMap) const
{
	if (this == &hashMap)
		return true;

	if (_tableSize != hashMap._tableSize)
		return false;

	for (flUInt i = 0; i < _tableSize; ++i)
	{
		if (_keyTable[i] != hashMap._keyTable[i])
			return false;
		if (_valueTable[i] != hashMap._valueTable[i])
			return false;
	}

	return true;
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
void
flHashMap<K, V, HF, CF>::clear()
{
	for (flUInt i = 0; i < _tableSize; ++i)
	{
		_keyTable[i].clear();
		_valueTable[i].clear();
	}
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flUInt
flHashMap<K, V, HF, CF>::getSize() const
{
	flUInt size = 0;
	for (flUInt i = 0; i < _tableSize; ++i)
		size += _keyTable[i].getSize();
	return size;
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
inline flUInt
flHashMap<K, V, HF, CF>::getTableSize() const
{
	return _tableSize;
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flBool
flHashMap<K, V, HF, CF>::resizeTable(flUInt newTableSize)
{
	assert(newTableSize > 0);

	flUInt		oldTableSize	= _tableSize;
	flArray<K>*	oldKeyTable		= _keyTable;
	flArray<V>*	oldValueTable	= _valueTable;

	_tableSize	= newTableSize;
	_keyTable	= new flArray<K>[_tableSize];
	_valueTable	= new flArray<V>[_tableSize];

	for (flUInt i = 0, j; i < oldTableSize; ++i)
	{
		flArray<K>& karray = oldKeyTable[i];
		flArray<V>& varray = oldValueTable[i];

		for (j = 0; j < karray.getSize(); ++j)
			insert(karray[j], varray[j]);
		
		karray.clear();
		varray.clear();
	}

	delete [] oldKeyTable;
	delete [] oldValueTable;

	return true;
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
void
flHashMap<K, V, HF, CF>::insert(const K& key, const V& value)
{
	HF hashFunc;
	flUInt hashValue = hashFunc(key) % _tableSize;
	_keyTable[hashValue].add(key);
	_valueTable[hashValue].add(value);
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flBool
flHashMap<K, V, HF, CF>::get(const K& key, V& value) const
{
	HF hashFunc;
	CF compFunc;
	flUInt hashValue = hashFunc(key) % _tableSize;
	const flArray<K>& karray = _keyTable[hashValue];
	const flArray<V>& varray = _valueTable[hashValue];
	for (flUInt i = 0; i < karray.getSize(); ++i)
	{
		if (compFunc(key, karray[i]))
		{
			value = varray[i];
			return true;
		}
	}
	return false;
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flBool
flHashMap<K, V, HF, CF>::find(const K& key) const
{
	HF hashFunc;
	CF compFunc;
	flUInt hashValue = hashFunc(key) % _tableSize;
	const flArray<K>& karray = _keyTable[hashValue];
	for (flUInt i = 0; i < karray.getSize(); ++i)
	{
		if (compFunc(key, karray[i]))
			return true;
	}
	return false;
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
void
flHashMap<K, V, HF, CF>::remove(const K& key)
{
	HF hashFunc;
	CF compFunc;
	flUInt hashValue = hashFunc(key) % _tableSize;
	flArray<K>& karray = _keyTable[hashValue];
	flArray<V>& varray = _valueTable[hashValue];
	for (flUInt i = 0; i < karray.getSize(); ++i)
	{
		if (compFunc(key, karray[i]))
		{
			karray.removeIndex(i);
			varray.removeIndex(i);
			break;
		}
	}
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
void
flHashMap<K, V, HF, CF>::replace(const K& key, const V& newValue)
{
	HF hashFunc;
	CF compFunc;
	flUInt hashValue = hashFunc(key) % _tableSize;
	flArray<K>& karray = _keyTable[hashValue];
	flArray<V>& varray = _valueTable[hashValue];
	for (flUInt i = 0; i < karray.getSize(); ++i)
	{
		if (compFunc(key, karray[i]))
		{
			varray[i] = newValue;
			break;
		}
	}
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
const K&
flHashMap<K, V, HF, CF>::getKey(flUInt index) const
{
	assert(index < getSize());

	for (flUInt i = 0; i < _tableSize; ++i)
	{
		if (index < _keyTable[i].getSize())
			break;
		index -= _keyTable[i].getSize();
	}
	assert(i < _tableSize);

	return _keyTable[i][index];
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
const V&
flHashMap<K, V, HF, CF>::getValue(flUInt index) const
{
	assert(index < getSize());

	for (flUInt i = 0; i < _tableSize; ++i)
	{
		if (index < _valueTable[i].getSize())
			break;
		index -= _valueTable[i].getSize();
	}
	assert(i < _tableSize);

	return _valueTable[i][index];
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flBool
flHashMap<K, V, HF, CF>::operator ==(const flHashMap& hashMap) const
{
	return equal(hashMap);
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flBool
flHashMap<K, V, HF, CF>::operator !=(const flHashMap& hashMap) const
{
	return !equal(hashMap);
}
//------------------------------------------------------------------------------
template<class K, class V, class HF, class CF>
flHashMap<K, V, HF, CF>&
flHashMap<K, V, HF, CF>::operator =(const flHashMap& hashMap)
{
	copy(hashMap);
	return *this;
}
//------------------------------------------------------------------------------
#ifdef	_DEBUG
template<class K, class V, class HF, class CF>
void
flHashMap<K, V, HF, CF>::postInformation() const
{
	for (flUInt i = 0; i < _tableSize; ++i)
	{
		flNotify::post(flNotify::DEBUG, "table:%d size:%d",
			i, _keyTable[i].getSize());
	}
}
#endif
//------------------------------------------------------------------------------
#endif	// !__flHash_inl__
