/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "StdAfx.h"
#include <flBase/flImage.h>
//------------------------------------------------------------------------------
flImage::flImage() :
_width(0),
_height(0),
_components(3),
_image(NULL)
{
}
//------------------------------------------------------------------------------
flImage::flImage(const flImage& p) :
_width(0),
_height(0),
_components(3),
_image(NULL)
{
	*this = p;
}
//------------------------------------------------------------------------------
flImage::flImage(flUInt width,
				 flUInt height,
				 flUInt components,
				 const flByte* image) :
_width(width),
_height(height),
_components(components),
_image(NULL)
{
	flUInt size = getSize();
	if (size)
	{
		_image = new flByte[size];
		memcpy(_image, image, size);
	}
}
//------------------------------------------------------------------------------
flImage::~flImage()
{
	clear();
}
//------------------------------------------------------------------------------
void
flImage::copy(const flImage& p)
{
	if (this == &p)
		return;

	set(p._width, p._height, p._components, p._image);
}
//------------------------------------------------------------------------------
flBool
flImage::equal(const flImage& p) const
{
	if (_width != p._width)
		return false;
	if (_height != p._height)
		return false;
	if (_components != p._components)
		return false;

	flUInt	size = getSize();
	return (size > 0) ? memcmp(_image, p._image, size) == 0 : false;
}
//------------------------------------------------------------------------------
void
flImage::set(flUInt width,
			 flUInt height,
			 flUInt components,
			 const flByte* image)
{
	clear();

	_width		= width;
	_height		= height;
	_components	= components;

	flUInt	size = getSize();
	if (size > 0)
	{
		_image = new flByte[size];
		memcpy(_image, image, size);
	}
}
//------------------------------------------------------------------------------
flBool
flImage::isValid() const
{
	return _width > 0 && _height > 0 && _image != NULL;
}
//------------------------------------------------------------------------------
flUInt
flImage::getSize() const
{
	return _width * _height * _components;
}
//------------------------------------------------------------------------------
flBool
flImage::isTransparent() const
{
	return (_components == 2) || (_components == 4);
}
//------------------------------------------------------------------------------
void
flImage::clear()
{
	delete [] _image;

	_width		= 0;
	_height		= 0;
	_components	= 3;
	_image		= NULL;
}
//--------------------------------------------------------------------------
void
flImage::swapColor()
{
	if (_components == 3)
	{
		flUInt numDatas = _width * _height;
		flByte* ptr = _image;
		for(flUInt i = 0; i < numDatas; i++)
		{
			flByte tmp;
			tmp = *ptr;
			*ptr = *(ptr + 2);
			*(ptr + 2) = tmp;

			ptr += 3;
		}
	}
}
//------------------------------------------------------------------------------
void
flImage::resize(flUInt width, flUInt height, AlignEnum align)
{
	if (!isValid())
		return;

	if (width == 0 || height == 0)
		return;

	if (width == _width && height == _height)
		return;

	flUInt newimagesize = width * height * _components;
	flByte*	newimage = new flByte[newimagesize];

	// Process Resize
	flInt orgX, orgY;

	if ((align & A_LEFT) != 0)
		orgX = 0;
	else if ((align & A_RIGHT) != 0)
		orgX = flInt(width) - flInt(_width);
	else
		orgX = (flInt(_width) - flInt(width)) / 2;

	if ((align & A_TOP) != 0)
		orgY = 0;
	else if ((align & A_BOTTOM) != 0)
		orgY = flInt(height) - flInt(_height);
	else
		orgY = (flInt(_height) - flInt(height)) / 2;

	memset(newimage, 0, newimagesize);

	flByte* srcPtr;
	flByte* dstPtr;

	flInt sx, sy;
	flInt dx, dy;

	for(sx = 0, dx = orgX; sx < (flInt)width; sx++, dx++)
	{
		for(sy = 0, dy = orgY; sy < (flInt)height; sy++, dy++)
		{
			if (dx < 0 || dy < 0 || _width <= (flUInt)dx || _height <= (flUInt)dy)
				continue;

			srcPtr = _image + (dx + dy * _width) * _components;
			dstPtr = newimage + (sx + sy * width) * _components;
            
			memcpy(dstPtr, srcPtr, _components);
		}
	}
	///

	delete [] _image;
	_width	= width;
	_height	= height;
	_image	= newimage;
}
//------------------------------------------------------------------------------
static void
scaleImage(flUInt widthin,
		   flUInt heightin,
		   flUInt components,
		   const flByte* datain,
		   flUInt widthout,
		   flUInt heightout,
		   flByte* dataout)
{
	// see Mesa 3-D graphics library mipmap.c
	assert(datain != NULL && dataout != NULL);
	assert(widthin > 0 && heightin > 0 && components > 0 &&
		widthout > 0 && heightout > 0);

	if (widthin == widthout && heightin == heightout)
	{
		memcpy(dataout, datain, widthin * heightin * components);
		return;
	}

	flUInt		i, j, k;
	flFloat*	tempin;
	flFloat*	tempout;
	flFloat		sx, sy;
	flUInt		rowstride = components * widthin;

	// Allocate storage for intermediate images
	tempin	= new flFloat[widthin  * heightin  * components * sizeof(flFloat)];
	tempout	= new flFloat[widthout * heightout * components * sizeof(flFloat)];

	// Unpack the pixel data and convert to floating point
	k = 0;
	for (i = 0; i < heightin; ++i)
	{
		flByte*	ubptr = (flByte*)datain + i * rowstride;
		for (j = 0; j < widthin * components; ++j)
			tempin[k++] = (flFloat)*ubptr++;
	}

	// Scale the image!
	if (widthout > 1)
		sx = flFloat(widthin - 1) / flFloat(widthout - 1);
	else
		sx = flFloat(widthin - 1);
	if (heightout > 1)
		sy = flFloat(heightin - 1) / flFloat(heightout - 1);
	else
		sy = flFloat(heightin - 1);

	if (sx < 1 && sy < 1)
	{
		// magnify both width and height:  use weighted sample of 4 pixels
		flUInt		i0, i1, j0, j1;
		flFloat		alpha, beta;
		flFloat*	src00;
		flFloat*	src01;
		flFloat*	src10;
		flFloat*	src11;
		flFloat		s1, s2;
		flFloat*	dst;

		for (i = 0; i < heightout; ++i)
		{
			i0 = flUInt(i * sy);
			i1 = i0 + 1;
			if (i1 >= heightin)
				i1 = heightin - 1;
			alpha = i * sy - i0;
			for (j = 0; j < widthout; ++j)
			{
				j0 = flUInt(j * sx);
				j1 = j0 + 1;
				if (j1 >= widthin)
					j1 = widthin - 1;
				beta = j * sx - j0;

				// compute weighted average of pixels in rect (i0,j0)-(i1,j1)
				src00 = tempin + (i0 * widthin + j0) * components;
				src01 = tempin + (i0 * widthin + j1) * components;
				src10 = tempin + (i1 * widthin + j0) * components;
				src11 = tempin + (i1 * widthin + j1) * components;

				dst = tempout + (i * widthout + j) * components;

				for (k = 0; k < components; ++k)
				{
					s1 = *src00++ * (1 - beta) + *src01++ * beta;
					s2 = *src10++ * (1 - beta) + *src11++ * beta;
					*dst++ = s1 * (1 - alpha) + s2 * alpha;
				}
			}
		}
	}
	else
	{
		// shrink width and/or height:  use an unweighted box filter
		flUInt		i0, i1;
		flUInt		j0, j1;
		flUInt		ii, jj;
		flFloat		sum;
		flFloat*	dst;

		for (i = 0; i < heightout; ++i)
		{
			i0 = flUInt(i * sy);
			i1 = i0 + 1;
			if (i1 >= heightin)
				i1 = heightin - 1;
			for (j = 0; j < widthout; ++j)
			{
				j0 = flUInt(j * sx);
				j1 = j0 + 1;
				if (j1 >= widthin)
					j1 = widthin - 1;

				dst = tempout + (i * widthout + j) * components;

				// compute average of pixels in the rectangle (i0,j0)-(i1,j1)
				for (k = 0; k < components; ++k)
				{
					sum = 0;
					for (ii = i0; ii <= i1; ++ii)
					{
						for (jj = j0; jj <= j1; ++jj)
							sum += *(tempin + (ii * widthin + jj) * components + k);
					}
					sum /= (j1 - j0 + 1) * (i1 - i0 + 1);
					*dst++ = sum;
				}
			}
		}
	}

	// Return output image
	k = 0;
	rowstride = widthout * components;
	for (i = 0; i < heightout; ++i)
	{
		flByte*	ubptr = (flByte*)dataout + i * rowstride;
		for (j = 0; j < widthout * components; ++j)
		{
			if (tempout[k] > 255)		tempout[k] = 255;
			else if (tempout[k] < 0)	tempout[k] = 0;
			*ubptr++ = (flByte)tempout[k++];
		}
	}

	// free temporary image storage
	delete [] tempin;
	delete [] tempout;
}
//------------------------------------------------------------------------------
void
flImage::stretch(flUInt width, flUInt height)
{
	if (!isValid())
		return;

	if (width == 0 || height == 0)
		return;

	if (width == _width && height == _height)
		return;

	flByte*	newimage = new flByte[width * height * _components];
	scaleImage(_width, _height, _components, _image,
		width, height, newimage);

	delete [] _image;
	_width	= width;
	_height	= height;
	_image	= newimage;
}
//--------------------------------------------------------------------------
flUInt
flImage::hash() const
{
	flUInt total = 0, shift = 0;
	if (_image != NULL)
	{
		for (flByte* s = _image; *s; s+=4)
		{
			total = total ^ ((*s) << shift);
			shift += 5;
			if (shift > 24)
				shift -= 24;
		}
	}
	return total;
}
//------------------------------------------------------------------------------
