/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "StdAfx.h"
#include <flBase/flImageIO.h>
#define	XMD_H
#undef	FAR
extern "C"
{
#include <jpeglib.h>
#include <cdjpeg.h>
#include <png.h>
}
#include <imglib.h>
//------------------------------------------------------------------------------
const flInt		JPEG_QUALITY = 90;
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
flImageIO::flImageIO()
{
}
//------------------------------------------------------------------------------
flImageIO::~flImageIO()
{
}
//------------------------------------------------------------------------------
const flImage
flImageIO::load(const flString& filename) const
{
	flString fn = filename;
	fn.toLower();

	if (strstr(fn.c_str(), ".jpg") ||
		strstr(fn.c_str(), ".jfif") ||
		strstr(fn.c_str(), ".jpeg"))
		return loadJPEG(filename);

	else if (strstr(fn.c_str(), ".bmp") ||
		strstr(fn.c_str(), ".dib"))
		return loadBMP(filename);

	else if (strstr(fn.c_str(), ".ppm"))
		return loadPPM(filename);

	else if (strstr(fn.c_str(), ".tga"))
		return loadTGA(filename);

	else if (strstr(fn.c_str(), ".png"))
		return loadPNG(filename);

	else if (strstr(fn.c_str(), ".tif") ||
		strstr(fn.c_str(), ".tiff"))
		return loadTIFF(filename);

	else if (strstr(fn.c_str(), ".rgb") ||
		strstr(fn.c_str(), ".rgba") ||
		strstr(fn.c_str(), ".sgi"))
		return loadRGB(filename);

	return flImage();
}
//------------------------------------------------------------------------------
static void
flJpegErrorFunc(j_common_ptr cinfo)
{
	(*cinfo->err->output_message)(cinfo);
	jpeg_destroy(cinfo);
	throw 0;
}
//------------------------------------------------------------------------------
const flImage
flImageIO::loadJPEG(const flString& filename) const
{
	struct jpeg_decompress_struct	cinfo;
	struct jpeg_error_mgr			jerr;
	JSAMPARRAY						buffer;
	flInt							width;
	flInt							height;
	flInt							components;
	flInt							row_stride;
	flByte*							data = NULL;
	flByte*							src;
	flByte*							dst;
	flInt							i, j;
	flImage							image;
	FILE*							fp = NULL;

	memset(&cinfo, 0, sizeof(jpeg_decompress_struct));

	try
	{
		fp = fopen(filename.c_str(), "rb");
		if (fp == NULL)
			throw 0;

		cinfo.err		= jpeg_std_error(&jerr);
		jerr.error_exit	= flJpegErrorFunc;

		jpeg_create_decompress(&cinfo);

		jpeg_stdio_src(&cinfo, fp);
		jpeg_read_header(&cinfo, TRUE);
		jpeg_start_decompress(&cinfo);

		width		= cinfo.image_width;
		height		= cinfo.image_height;
		components	= cinfo.output_components;
		row_stride	= width * components;

		assert(width * height > 0 && row_stride > 0);

		buffer = (*cinfo.mem->alloc_sarray)
			((j_common_ptr)&cinfo, JPOOL_IMAGE, row_stride, 1);
		if (buffer == NULL)
			throw 0;

		data	= new flByte[width * height * components];

		for (i = 0; cinfo.output_scanline < (flUInt)height; ++i)
		{
			jpeg_read_scanlines(&cinfo, buffer, 1);

			src = buffer[0];
			dst = &data[(height-i-1) * row_stride];
			for (j = 0; j < row_stride; j++)
				*dst++ = *src++;
		}

		jpeg_finish_decompress(&cinfo);
		jpeg_destroy_decompress(&cinfo);

		image.set(width, height, components, data);
		delete [] data;

		fclose(fp);
	}
	catch (...)
	{
		image.clear();
		delete [] data;
		jpeg_destroy_decompress(&cinfo);
		if (fp != NULL)
			fclose(fp);
	}

	return image;
}
//------------------------------------------------------------------------------
typedef	cjpeg_source_ptr	(*_init_read_func)(jpeg_compress_struct*);
//------------------------------------------------------------------------------
static const flImage
_load(const flString& filename, flInt check, _init_read_func func)
{
	struct jpeg_compress_struct	cinfo;
	struct jpeg_error_mgr		jerr;
	cjpeg_source_ptr			src_mgr = NULL;
	flInt						width;
	flInt						height;
	flInt						components;
	flInt						row_stride;
	flByte*						data = NULL;
	flByte*						src;
	flByte*						dst;
	flInt						i, j;
	flImage						image;
	FILE*						fp = NULL;

	memset(&cinfo, 0, sizeof(jpeg_compress_struct));

	try
	{
		fp = fopen(filename.c_str(), "rb");
		if (fp == NULL)
			throw 0;
		flInt	c;
		if (((c = getc(fp)) == EOF) || ungetc(c, fp) == EOF || c != check)
			throw 0;

		cinfo.err		= jpeg_std_error(&jerr);
		jerr.error_exit	= flJpegErrorFunc;

		jpeg_create_compress(&cinfo);

		src_mgr				= func(&cinfo);
		src_mgr->input_file	= fp;
		(*src_mgr->start_input)(&cinfo, src_mgr);

		jpeg_set_defaults(&cinfo);
		jpeg_stdio_dest(&cinfo, NULL);
		jpeg_start_compress(&cinfo, TRUE);

		width		= cinfo.image_width;
		height		= cinfo.image_height;
		components	= cinfo.input_components;
		row_stride	= width * components;

		assert(width * height > 0 && row_stride > 0);

		data = new flByte[width * height * components];

		for (i = 0; i < height; ++i)
		{
			(*src_mgr->get_pixel_rows)(&cinfo, src_mgr);
			src	= src_mgr->buffer[0];
			dst	= &data[(height-i-1) * row_stride];
			for (j = 0; j < row_stride; ++j)
				*dst++ = *src++;
		}

		(*src_mgr->finish_input)(&cinfo, src_mgr);
		jpeg_destroy_compress(&cinfo);

		image.set(width, height, components, data);
		delete [] data;

		fclose(fp);
	}
	catch (...)
	{
		image.clear();
		delete [] data;
		jpeg_destroy_compress(&cinfo);
		if (fp != NULL)
			fclose(fp);
	}

	return image;
}
//------------------------------------------------------------------------------
const flImage
flImageIO::loadBMP(const flString& filename) const
{
	return _load(filename, 'B', jinit_read_bmp);
}
//------------------------------------------------------------------------------
const flImage
flImageIO::loadPPM(const flString& filename) const
{
	return _load(filename, 'P', jinit_read_ppm);
}
//------------------------------------------------------------------------------
const flImage
flImageIO::loadTGA(const flString& filename) const
{
	return _load(filename, 0x00, jinit_read_targa);
}
//------------------------------------------------------------------------------
static void __cdecl
flPngErrorFunc(png_structp Png,png_const_charp message)
{
	throw 0;
}
//------------------------------------------------------------------------------
static void __cdecl
flPngWarningFunc(png_structp Png,png_const_charp message)
{
    throw 0;
}
//------------------------------------------------------------------------------
const flImage
flImageIO::loadPNG(const flString& filename) const
{
	png_structp		png_ptr = NULL;
	png_infop		info_ptr = NULL;
	png_uint_32		width = 0;
	png_uint_32		height = 0;
	flInt			bit_depth;
	flInt			color_type;
	flInt			interlace_type;
	png_bytepp		row_pointers = NULL;
	png_uint_32		i, j;
	flInt			intent;
	flDouble		screen_gamma = 2.2;
	flDouble		image_gamma;

	const flUInt	SIG_BYTES = 8;
	flByte			sig[SIG_BYTES];

	flUInt			components;
	flByte*			data = NULL;
	flByte*			src;
	flByte*			dst;
	flImage			image;

	FILE*			fp = NULL;

	try
	{
		fp = fopen(filename.c_str(), "rb");
		if (fp == NULL)
			throw 0;

		// check
		if (fread(sig, 1, SIG_BYTES, fp) != SIG_BYTES)
			throw 0;
		//fseek(fp, 0, SEEK_SET);
		if (!png_check_sig(sig, SIG_BYTES))
			throw 0;

		// setup
		png_ptr	= png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL,
			flPngErrorFunc, flPngWarningFunc);
		if (png_ptr == NULL)
			throw 0;

		info_ptr = png_create_info_struct(png_ptr);
		if (info_ptr == NULL)
			throw 0;

		png_init_io(png_ptr, fp);
		png_set_sig_bytes(png_ptr, SIG_BYTES);

		// read header
		png_read_info(png_ptr, info_ptr);                       
		png_get_IHDR(png_ptr, info_ptr, &width, &height,
			&bit_depth, &color_type, &interlace_type,
			NULL, NULL);

		// read image
		if (bit_depth < 8)
			png_set_packing(png_ptr);
		else if (bit_depth == 16)
			png_set_strip_16(png_ptr);

		switch (color_type)
		{
		case PNG_COLOR_TYPE_GRAY:
			if (bit_depth < 8)
				png_set_gray_1_2_4_to_8(png_ptr);
			if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
			{
				png_set_tRNS_to_alpha(png_ptr);
				components = 2;
			}
			else
			{
				components = 1;
			}
			break;
		case PNG_COLOR_TYPE_GRAY_ALPHA:
			components = 2;
			break;
		case PNG_COLOR_TYPE_RGB:
			if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
			{
				png_set_tRNS_to_alpha(png_ptr);
				components = 4;
			}
			else
			{
				components = 3;
			}
			break;
		case PNG_COLOR_TYPE_RGB_ALPHA:
			components = 4;
			break;
		case PNG_COLOR_TYPE_PALETTE:
			if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
			{
				png_set_tRNS_to_alpha(png_ptr);
				components = 4;
			}
			else
			{
				png_set_palette_to_rgb(png_ptr);
				components = 3;
			}
			break;
		default:
			throw 0;
		}
		png_read_update_info(png_ptr, info_ptr);

		row_pointers = new png_bytep[height];
		for (i = 0; i < height; ++i)
			row_pointers[i] = new png_byte[png_get_rowbytes(png_ptr, info_ptr)];
		png_read_image(png_ptr, row_pointers);

		if (png_get_sRGB(png_ptr, info_ptr, &intent))
		{
			png_set_gamma(png_ptr, screen_gamma, 0.45455);
		}
		else
		{
			if (png_get_gAMA(png_ptr, info_ptr, &image_gamma))
				png_set_gamma(png_ptr, screen_gamma, image_gamma);
			else
				png_set_gamma(png_ptr, screen_gamma, 0.45455);
		}

		data = new flByte[width * height * components];
		dst = data;
		src = &row_pointers[0][0];

		for (i = 0; i < height; ++i)
		{
			src = row_pointers[height-i-1];
			for (j = 0; j < width; ++j)
			{
				for (flUInt k = 0; k < components; ++k)
					*dst++ = *src++;
			}
		}

		// create flimage
		image.set(width, height, components, data);

		delete [] data;
		for (i = 0; i < height; ++i)
			delete [] row_pointers[i];
		delete [] row_pointers;

		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);

		fclose(fp);
	}
	catch (...)
	{
		image.clear();

		delete [] data;
		for (i = 0; i < height; ++i)
			delete [] row_pointers[i];
		delete [] row_pointers;

		png_destroy_read_struct(&png_ptr, &info_ptr, NULL);

		if (fp != NULL)
			fclose(fp);
	}

	return image;
}
//------------------------------------------------------------------------------
const flImage
flImageIO::loadTIFF(const flString& filename) const
{
	flUInt	width;
	flUInt	height;
	flUInt	components;
	flByte*	data = NULL;
	flImage	image;
	FILE*	fp = NULL;

	try
	{
		fp = fopen(filename.c_str(), "rb");
		if (fp == NULL)
			throw 0;

		if (!read_tiff(fp, width, height, components, data))
			throw 0;

		image.set(width, height, components, data);
		delete [] data;
		fclose(fp);
	}
	catch (...)
	{
		image.clear();
		delete [] data;
		if (fp != NULL)
			fclose(fp);
	}

	return image;
}
//------------------------------------------------------------------------------
const flImage
flImageIO::loadRGB(const flString& filename) const
{
	flUInt	width;
	flUInt	height;
	flUInt	components;
	flByte*	data = NULL;
	flImage	image;
	FILE*	fp = NULL;

	try
	{
		fp = fopen(filename.c_str(), "rb");
		if (fp == NULL)
			throw 0;

		if (!read_rgb(fp, width, height, components, data))
			throw 0;

		image.set(width, height, components, data);
		delete [] data;
		fclose(fp);
	}
	catch (...)
	{
		image.clear();
		delete [] data;
		if (fp != NULL)
			fclose(fp);
	}

	return image;
}
//------------------------------------------------------------------------------
flBool
flImageIO::save(const flString& filename, const flImage& image) const
{
	if (!image.isValid())
		return false;

	flString fn = filename;
	fn.toLower();

	if (strstr(fn.c_str(), ".jpg") ||
		strstr(fn.c_str(), ".jfif") ||
		strstr(fn.c_str(), ".jpeg"))
		return saveJPEG(filename, image);

	else if (strstr(fn.c_str(), ".bmp") ||
		strstr(fn.c_str(), ".dib"))
		return saveBMP(filename, image);

	else if (strstr(fn.c_str(), ".png"))
		return savePNG(filename, image);

	return false;
}
//------------------------------------------------------------------------------
flBool
flImageIO::saveJPEG(const flString& filename, const flImage& image) const
{
	if (!image.isValid())
		return false;

	struct jpeg_compress_struct	cinfo;
	struct jpeg_error_mgr		jerr;
	flUInt						height = 0;
	flUInt						width = 0;
	flInt						components;
	flInt						format;
	JSAMPROW					jsampRow;
	flInt						i;
	flByte*						data = NULL;
	FILE*						fp = NULL;

	try
	{
		fp = fopen(filename.c_str(), "wb");
		if (fp == NULL)
			throw 0;

		width		= image.width();
		height		= image.height();
		components	= image.components();

		switch (components)
		{
		case 1:
			data = new flByte[width * height];
			memcpy(data, image.image(), width * height);
			format		= JCS_GRAYSCALE;
			break;
		case 2:
			data = new flByte[width * height];
			for (flUInt i = 0, j = 0; i < width * height * 2; i+=2, ++j)
				data[j] = image.image()[i];
			components	= 1;
			format		= JCS_GRAYSCALE;
			break;
		case 3:
			data = new flByte[width * height * 3];
			memcpy(data, image.image(), width * height * 3);
			format		= JCS_RGB;
			break;
		case 4:
			data = new flByte[width * height * 3];
			for (flUInt i = 0, j = 0; i < width * height * 4; i+=4, j+=3)
			{
				data[j  ] = image.image()[i  ];
				data[j+1] = image.image()[i+1];
				data[j+2] = image.image()[i+2];
			}
			components	= 3;
			format		= JCS_RGB;
			break;
		}

		cinfo.err				= jpeg_std_error(&jerr);
		jerr.error_exit			= flJpegErrorFunc;

		jpeg_create_compress(&cinfo);
		jpeg_stdio_dest(&cinfo, fp);

		cinfo.image_width		= width;
		cinfo.image_height		= height;
		cinfo.input_components	= components;
		cinfo.in_color_space	= (J_COLOR_SPACE)format;

		jpeg_set_defaults(&cinfo);
		jpeg_set_quality(&cinfo, JPEG_QUALITY, FALSE);
		jpeg_start_compress(&cinfo, TRUE);

		for (i = 0; cinfo.next_scanline < cinfo.image_height; ++i)
		{
			jsampRow = (JSAMPROW)&data[(height-i-1)*width*components];
			jpeg_write_scanlines(&cinfo, (JSAMPARRAY)&jsampRow , 1);
		}

		delete [] data;
		jpeg_finish_compress(&cinfo);
		jpeg_destroy_compress(&cinfo);
		fclose(fp);
	}
	catch (...)
	{
		delete [] data;
		jpeg_destroy_compress(&cinfo);
		if (fp != NULL)
			fclose(fp);
		return false;
	}

	return true;
}
//------------------------------------------------------------------------------
flBool
flImageIO::saveBMP(const flString& filename, const flImage& image) const
{
	if (!image.isValid())
		return false;

	BITMAPFILEHEADER	header;
	BITMAPINFO			info;

	flUInt				width;
	flUInt				height;
	flUInt				components;
	flUInt				pixelWidth;
	flByte*				data = NULL;
	flByte*				src;
	flByte*				dst;
	flByte				r, g, b;
	flUInt				i, j;
	FILE*				fp = NULL;

	try
	{
		fp = fopen(filename.c_str(), "wb");
		if (fp == NULL)
			throw 0;

		width		= image.width();
		height		= image.height();
		components	= image.components();
		pixelWidth	= ((width * 3) + 3) & ~3;

		data = new flByte[pixelWidth * height];
		memset(data, 0, pixelWidth * height);

		for (i = 0; i < height; ++i)
		{
			src = (flByte*)&image.image()[i*width*components];
			dst = &data[i*pixelWidth];
			for (j = 0; j < width; ++j)
			{
				switch (components)
				{
				case 1:
					r = *src;
					g = *src;
					b = *src++;
					break;
				case 2:
					r = *src;
					g = *src;
					b = *src++;
					src++;
					break;
				case 3:
					r = *src++;
					g = *src++;
					b = *src++;
					break;
				case 4:
					r = *src++;
					g = *src++;
					b = *src++;
					src++;
					break;
				}
				*dst++ = b;
				*dst++ = g;
				*dst++ = r;
			}
		}

		info.bmiHeader.biSize			= sizeof(BITMAPINFOHEADER);
		info.bmiHeader.biWidth			= width;
		info.bmiHeader.biHeight			= height;
		info.bmiHeader.biPlanes			= 1;
		info.bmiHeader.biBitCount		= 24;
		info.bmiHeader.biCompression	= BI_RGB;
		info.bmiHeader.biSizeImage		= pixelWidth * height;
		info.bmiHeader.biXPelsPerMeter	= 2952; // 75 DPI
		info.bmiHeader.biYPelsPerMeter	= 2952;
		info.bmiHeader.biClrUsed		= 0;
		info.bmiHeader.biClrImportant	= 0;

		header.bfType      = 'MB';
		header.bfSize      = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER) +
			info.bmiHeader.biSizeImage;
		header.bfReserved1 = 0;
		header.bfReserved2 = 0;
		header.bfOffBits   = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER);

		if (fwrite(&header, 1, sizeof(BITMAPFILEHEADER), fp) < sizeof(BITMAPFILEHEADER))
			throw 0;

		if (fwrite(&info, 1, sizeof(BITMAPINFOHEADER), fp) < sizeof(BITMAPINFOHEADER))
			throw 0;

		if (fwrite(data, 1, info.bmiHeader.biSizeImage, fp) < info.bmiHeader.biSizeImage)
			throw 0;

		delete [] data;
		fclose(fp);
	}
	catch (...)
	{
		delete [] data;
		if (fp != NULL)
			fclose(fp);
		return false;
	}

	return true;
}
//------------------------------------------------------------------------------
flBool
flImageIO::savePNG(const flString& filename, const flImage& image) const
{
	if (!image.isValid())
		return false;

	png_structp		png_ptr = NULL;
	png_infop		info_ptr = NULL;
	FILE			*fp;

	time_t			gmt;
	png_time		mod_time;
	const flInt		NUM_TEXTS = 5;
	png_text		text_ptr[NUM_TEXTS];
	png_bytepp		row_pointers = NULL;

	flUInt			height = 0;
	flUInt			width = 0;
	flUInt			components;
	flInt			formats[] =
	{
		PNG_COLOR_TYPE_GRAY,
		PNG_COLOR_TYPE_GRAY_ALPHA,
		PNG_COLOR_TYPE_RGB,
		PNG_COLOR_TYPE_RGB_ALPHA
	};
	flUInt			i;

	try
	{
		fp = fopen(filename.c_str(), "wb");
		if (fp == NULL)
			throw 0;

		width = image.width();
		height = image.height();
		components = image.components();

		row_pointers = new png_bytep[height];
		for (i = 0; i < height; ++i)
		{
			row_pointers[i] = new png_byte[width * components];
			memcpy(row_pointers[i],
				&image.image()[(height-i-1)*width*components],
				width * components);
		}

		png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL,
			flPngErrorFunc, flPngWarningFunc);
		if (png_ptr == NULL)
			throw 0;

		info_ptr = png_create_info_struct(png_ptr);
		if (info_ptr == NULL)
			throw 0;

		png_init_io(png_ptr, fp);
		png_set_filter(png_ptr, 0, PNG_ALL_FILTERS);
		png_set_compression_level(png_ptr, Z_BEST_COMPRESSION);
		png_set_IHDR(png_ptr, info_ptr, width, height,
			8, formats[components-1], PNG_INTERLACE_NONE,
			PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

		time(&gmt);
		png_convert_from_time_t(&mod_time, gmt);
		png_set_tIME(png_ptr, info_ptr, &mod_time);

		text_ptr[0].key = "Title";
		text_ptr[0].text = (png_charp)filename.c_str();
		text_ptr[0].compression = PNG_TEXT_COMPRESSION_NONE;
		text_ptr[1].key = "Author";
		text_ptr[1].text = "";
		text_ptr[1].compression = PNG_TEXT_COMPRESSION_NONE;
		text_ptr[2].key = "Description";
		text_ptr[2].text = "";
		text_ptr[2].compression = PNG_TEXT_COMPRESSION_NONE;
		text_ptr[3].key = "Copyright";
		text_ptr[3].text = "";
		text_ptr[3].compression = PNG_TEXT_COMPRESSION_NONE;
		text_ptr[4].key = "Creation Time";
		text_ptr[4].text = png_convert_to_rfc1123(png_ptr, &mod_time);
		text_ptr[4].compression = PNG_TEXT_COMPRESSION_NONE;

		png_set_text(png_ptr, info_ptr, text_ptr, NUM_TEXTS);

		// write header
		png_write_info(png_ptr, info_ptr);

		// PNG write image
		png_write_image(png_ptr, row_pointers);

		png_write_end(png_ptr, info_ptr);
		png_destroy_write_struct(&png_ptr, &info_ptr);

		for (i = 0; i < height; ++i)
			delete [] row_pointers[i];
		delete [] row_pointers;

		fclose(fp);
	}
	catch (...)
	{
		for (i = 0; i < height; ++i)
			delete [] row_pointers[i];
		delete [] row_pointers;

		png_destroy_write_struct(&png_ptr, &info_ptr);

		if (fp != NULL)
			fclose(fp);
		return false;
	}

	return true;
}
//------------------------------------------------------------------------------
