/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flDigitalClock/DigitalClock.h>
#include <flDigitalClock/DigitalClockProp.h>
#include <flDigitalClock/DigitalClockStream.h>
#include <flFilterCommon/flTextDraw.h>
#include <flFilterCommon/flFillColor.h>
#include <flBase/flSystemTime.h>
#include <time.h>
#include <sys/timeb.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudOpPinTypes =
{
    &MEDIATYPE_Video,			// MajorType
    &MEDIASUBTYPE_NULL			// MinorType
};
//------------------------------------------------------------------------------
const AMOVIESETUP_PIN sudOpPin =
{
	L"Out",						// The Pins name
	FALSE,						// Is rendered
	TRUE,						// Is an output pin
	FALSE,						// Allowed none
	FALSE,						// Allowed many
	&CLSID_NULL,				// Connects to filter
	NULL,						// Connects to pin
	1,							// Number of types
	&sudOpPinTypes				// Pin details
};
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudDigitalClock =
{
    &CLSID_DigitalClock,		// Filter CLSID
    L"NIME Digital Clock",		// String name
    MERIT_DO_NOT_USE,			// Filter merit
    1,							// Number of pins
    &sudOpPin					// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[2] = {
    { L"DigitalClock"
    , &CLSID_DigitalClock
    , CDigitalClock::CreateInstance
    , NULL
    , &sudDigitalClock }
  ,
    { L"DigitalClockProp"
    , &CLSID_DigitalClockPropertyPage
    , CDigitalClockProp::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CDigitalClock::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
    CDigitalClock *pNewObject = new CDigitalClock(NAME("DigitalClock"), punk, phr);
    if (pNewObject == NULL)
	{
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CDigitalClock::CDigitalClock(TCHAR *tszName, LPUNKNOWN pUnk, HRESULT *phr) :
CFLSource(tszName, pUnk, CLSID_DigitalClock, phr),
CPersistStream(pUnk, phr)
{
	ASSERT(tszName != NULL);
	ASSERT(phr != NULL);

	CAutoLock cAutoLock(&_stateLock);

	_streams = new CFLSourceStream*[1];
	if (_streams == NULL)
	{
		*phr = E_OUTOFMEMORY;
		return ;
	}

	_streams[0] = new CDigitalClockStream(phr, this, L"Out");
	if (_streams[0] == NULL)
	{
		*phr = E_OUTOFMEMORY;
		return;
	}
	_digitalClockStream = (CDigitalClockStream *)_streams[0];

	// Property
	_outVideoResolution		= DCVR_320x240;
	_frameRate				= 30;
	_fontSize				= DCFS_AUTO;
	_fontColor[0]			= 0x00;
	_fontColor[1]			= 0x00;
	_fontColor[2]			= 0x00;
	_backgroundColor[0]		= 0x96;
	_backgroundColor[1]		= 0xb4;
	_backgroundColor[2]		= 0x96;
	_headerString			= "";
	_footerString			= "";

	// Status
	_isRunning				= false;

	// Shape Draws
	_time1TextDraw			= new flTextDraw();
	_time2TextDraw			= new flTextDraw();
	_headerTextDraw			= new flTextDraw();
	_footerTextDraw			= new flTextDraw();
	_backgroundDraw			= new flFillColor();
	_time1TextDraw->setTextAlign(flTextDraw::TDTA_CENTER);
	_time2TextDraw->setTextAlign(flTextDraw::TDTA_CENTER);
	_headerTextDraw->setTextAlign(flTextDraw::TDTA_CENTER);
	_footerTextDraw->setTextAlign(flTextDraw::TDTA_CENTER);
	_time1TextDraw->setTextBorderOn(false);
	_time2TextDraw->setTextBorderOn(false);
	_headerTextDraw->setTextBorderOn(false);
	_footerTextDraw->setTextBorderOn(false);

	setup();
}
//------------------------------------------------------------------------------
CDigitalClock::~CDigitalClock()
{
	delete _backgroundDraw;
	delete _footerTextDraw;
	delete _headerTextDraw;
	delete _time2TextDraw;
	delete _time1TextDraw;
}
//------------------------------------------------------------------------------
// Reveals IDigitalClock & ISpecifyPropertyPages
STDMETHODIMP
CDigitalClock::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IDigitalClock)
	{
        return GetInterface((IDigitalClock *) this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    }
	else
	{
		return CFLSource::NonDelegatingQueryInterface(riid, ppv);
    }
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// IMediaFilter
STDMETHODIMP
CDigitalClock::Run(REFERENCE_TIME startTime)
{
	HRESULT hr = CFLSource::Run(startTime);
	if (FAILED(hr))
		return hr;

	CAutoLock autoLock(&_filterLock);

	_isRunning			= true;

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::Pause()
{
    return CFLSource::Pause();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::Stop()
{
	{
		CAutoLock autoLock(&_filterLock);
		_isRunning			= false;
	}

	HRESULT hr = CFLSource::Stop();
	if (FAILED(hr))
		return hr;

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::GetState(DWORD dwMSecs, FILTER_STATE *State)
{
	HRESULT hr = CFLSource::GetState(dwMSecs, State);

	if (m_State == State_Paused)
		return VFW_S_CANT_CUE;
	else
		return hr;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_DigitalClock;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CDigitalClock::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	flUInt length;
	const flChar *str;

	WRITEOUT(_outVideoResolution);
	WRITEOUT(_frameRate);
	WRITEOUT(_fontSize);
	WRITENOUT(_fontColor, 3);
	WRITENOUT(_backgroundColor, 3);
	length = _headerString.length();
	WRITEOUT(length);
	if (length != 0)
	{
		str = _headerString.c_str();
		WRITENOUT(str, length);
	}
	length = _footerString.length();
	WRITEOUT(length);
	if (length != 0)
	{
		str = _footerString.c_str();
		WRITENOUT(str, length);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CDigitalClock::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	flUInt length;
	flChar str[DC_MAXIMUM_HEADERFOOTER_LENGTH];

	READIN(_outVideoResolution);
	READIN(_frameRate);
	READIN(_fontSize);
	READNIN(_fontColor, 3);
	READNIN(_backgroundColor, 3);
	READIN(length);
	if (length != 0)
	{
		READNIN(str, length);
		str[length] = 0;
		_headerString = flString(str);
	}
	READIN(length);
	if (length != 0)
	{
		READNIN(str, length);
		str[length] = 0;
		_footerString = flString(str);
	}

	setup();

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CDigitalClock::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CDigitalClock::SizeMax()
{
	return	1000;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages method
STDMETHODIMP
CDigitalClock::GetPages(CAUUID *pPages)
{
	pPages->cElems = 1;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 1);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_DigitalClockPropertyPage;
	return NOERROR;
}
//------------------------------------------------------------------------------
// IDigitalClock
STDMETHODIMP
CDigitalClock::get_OutputVideoResolution(flInt *videoResolution)
{
	CAutoLock autoLock(&_filterLock);
	*videoResolution = _outVideoResolution;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::put_OutputVideoResolution(flInt videoResolution)
{
	if (m_State != State_Stopped)
		return NOERROR;

	CAutoLock autoLock(&_filterLock);
	_outVideoResolution = videoResolution;

	ReconnectPin((IPin *)_streams[0], NULL);

	setup();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::get_FrameRate(flInt *frameRate)
{
	CAutoLock autoLock(&_filterLock);
	*frameRate = _frameRate;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::put_FrameRate(flInt frameRate)
{
	CAutoLock autoLock(&_filterLock);
	_frameRate = frameRate;
	if (_frameRate == 0)
		_frameRate = 1;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::get_FontSize(flInt* fontSize)
{
	CAutoLock autoLock(&_filterLock);
	*fontSize = _fontSize;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::put_FontSize(flInt fontSize)
{
	CAutoLock autoLock(&_filterLock);
	_fontSize = fontSize;

	setup();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::get_FontColor(flByte *color)
{
	CAutoLock autoLock(&_filterLock);
	memcpy(color, _fontColor, sizeof(_fontColor));
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::put_FontColor(const flByte *color)
{
	CAutoLock autoLock(&_filterLock);
	memcpy(_fontColor, color, sizeof(_fontColor));

	setup();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::get_BackgroundColor(flByte *color)
{
	CAutoLock autoLock(&_filterLock);
	memcpy(color, _backgroundColor, sizeof(_backgroundColor));
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::put_BackgroundColor(const flByte *color)
{
	CAutoLock autoLock(&_filterLock);
	memcpy(_backgroundColor, color, sizeof(_backgroundColor));

	setup();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::get_HeaderString(flChar *string)
{
	CAutoLock autoLock(&_filterLock);

	if (string == NULL)
		return NOERROR;

	if (_headerString.length() == 0)
		string[0] = 0;
	else
		strcpy(string, _headerString.c_str());
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::put_HeaderString(const flChar *string)
{
	CAutoLock autoLock(&_filterLock);

	if (string == NULL)
		return NOERROR;

	_headerString = flString(string);

	setup();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::get_FooterString(flChar *string)
{
	CAutoLock autoLock(&_filterLock);

	if (string == NULL)
		return NOERROR;

	if (_footerString.length() == 0)
		string[0] = 0;
	else
		strcpy(string, _footerString.c_str());
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClock::put_FooterString(const flChar *string)
{
	CAutoLock autoLock(&_filterLock);

	if (string == NULL)
		return NOERROR;

	_footerString = flString(string);

	setup();

	return NOERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
void
CDigitalClock::setup()
{
	const flUInt FontSize [] = { 8, 12, 16, 20, 24, 28, 36, 44, 60, 76, 100, 124 };

	const flUInt VideoWidth []		= {  128,  176,  352,  160,  320,  360,  640,  720,  800, 1024, 1280, 1600,	1280, 1920,  852,  852 };
	const flUInt VideoHeight[]		= {   96,  144,  288,  120,  240,  240,  480,  480,  600,  768, 1024, 1200,	 720, 1080,  480,  480 };

	const flUInt Time1FontSize []	= {   20,   26,   48,   26,   48,   48,   80,   80,   86,  108,  124,  148,  124,  160,   86,   86 };
	const flUInt Time2FontSize []	= {   20,   26,   48,   26,   48,   48,   80,   80,   86,  108,  124,  148,  124,  160,   86,   86 };
	const flUInt HeaderFontSize []	= {   12,   14,   24,   14,   24,   24,   40,   40,   40,   48,   48,   52,   48,   48,   40,   40 };
	const flUInt FooterFontSize []	= {   12,   14,   24,   14,   24,   24,   40,   40,   40,   48,   48,   52,   48,   48,   40,   40 };

	const flInt Time1Top []			= {   24,   36,   72,   30,   60,   60,  120,  120,  150,  192,  256,  300,  200,  270,  120,  120 };
	const flInt Time2Top []			= {   48,   72,  144,   60,  120,  120,  240,  240,  300,  384,  512,  600,  400,  540,  240,  240 };
	const flInt HeaderTop []		= {    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0 };
	const flInt FooterBottom []		= {   96,  144,  288,  120,  240,  240,  480,  480,  600,  768, 1024, 1200,  720, 1080,  480,  480 };


	flInt index = _outVideoResolution;


	// Video Size
	_time1TextDraw->setVideoSize(VideoWidth[index], VideoHeight[index]);
	_time2TextDraw->setVideoSize(VideoWidth[index], VideoHeight[index]);
	_headerTextDraw->setVideoSize(VideoWidth[index], VideoHeight[index]);
	_footerTextDraw->setVideoSize(VideoWidth[index], VideoHeight[index]);
	_backgroundDraw->setVideoSize(VideoWidth[index], VideoHeight[index]);

	// Text Position
	_time1TextDraw->setPosition(0, Time1Top[index]);
	_time2TextDraw->setPosition(0, Time2Top[index]);
	_headerTextDraw->setPosition(0, HeaderTop[index]);
	_footerTextDraw->setPosition(0, FooterBottom[index] - FooterFontSize[index] - 5);

	// Text Area
	// Text Font Size
	if (_fontSize == DCFS_AUTO)
	{
		_time1TextDraw->setTextArea(VideoWidth[index], Time1FontSize[index] + 5);
		_time2TextDraw->setTextArea(VideoWidth[index], Time2FontSize[index] + 5);
		_time1TextDraw->setTextFontSize(Time1FontSize[index]);
		_time2TextDraw->setTextFontSize(Time2FontSize[index]);
	}
	else
	{
		_time1TextDraw->setTextArea(VideoWidth[index], FontSize[_fontSize] + 5);
		_time2TextDraw->setTextArea(VideoWidth[index], FontSize[_fontSize] + 5);
		_time1TextDraw->setTextFontSize(FontSize[_fontSize]);
		_time2TextDraw->setTextFontSize(FontSize[_fontSize]);
	}
	_headerTextDraw->setTextArea(VideoWidth[index], HeaderFontSize[index] + 5);
	_footerTextDraw->setTextArea(VideoWidth[index], FooterFontSize[index] + 5);
	_headerTextDraw->setTextFontSize(HeaderFontSize[index]);
	_footerTextDraw->setTextFontSize(FooterFontSize[index]);

	// Text
	_headerTextDraw->setText(_headerString);
	_footerTextDraw->setText(_footerString);

	// Text Font Color
	_time1TextDraw->setColor(_fontColor);
	_time2TextDraw->setColor(_fontColor);
	_headerTextDraw->setColor(_fontColor);
	_footerTextDraw->setColor(_fontColor);

	// Background Color
	_backgroundDraw->setColor(_backgroundColor);
}
//------------------------------------------------------------------------------
void
CDigitalClock::draw(flByte* data, flUInt dataLength)
{
	CAutoLock lock(&_filterLock);

	_backgroundDraw->draw(data);

	flChar str[128];

	_timeb timeb;
	_ftime(&timeb);

	struct tm *newtime;
	newtime = localtime(&timeb.time);

	// Set Year/Month/Day
	sprintf(str, "%d/%d/%d",
		(newtime->tm_year + 1900),
		(newtime->tm_mon + 1),
		newtime->tm_mday);
	_time1TextDraw->setText(flString(str));

	// Set Hour:Minute:Second.Millisecond
	sprintf(str, "%d:%02d:%02d.%03d",
		newtime->tm_hour,
		newtime->tm_min,
		newtime->tm_sec,
		timeb.millitm);
	_time2TextDraw->setText(flString(str));

	// Draw Text
	_time1TextDraw->draw(data);
	_time2TextDraw->draw(data);
	_headerTextDraw->draw(data);
	_footerTextDraw->draw(data);
}
//------------------------------------------------------------------------------
