/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN

#include <windows.h>
#include <streams.h>
#include <initguid.h>
#include <olectl.h>
#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif

#include <flTypes/flUIDs.h>
#include <flTypes/IDigitalClock.h>
#include <flDigitalClock/DigitalClock.h>
#include <flDigitalClock/DigitalClockStream.h>
//------------------------------------------------------------------------------
CDigitalClockStream::CDigitalClockStream
					(HRESULT *phr, CDigitalClock *pParent, LPCWSTR pPinName) :
CFLSourceStream(NAME("DigitalClock Stream"), phr, pParent, pPinName)
{
	_digitalClock	= pParent;
	_streamTime		= 0;
}
//------------------------------------------------------------------------------
CDigitalClockStream::~CDigitalClockStream()
{}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
HRESULT
CDigitalClockStream::FillBuffer(IMediaSample *pms)
{
	// Stream Time
	REFERENCE_TIME interval = (10000000 / _digitalClock->_frameRate);
    REFERENCE_TIME timeStart = _streamTime;
    REFERENCE_TIME timeEnd  = _streamTime + interval;
	pms->SetTime(&timeStart, &timeEnd);
	_streamTime = timeEnd;

	// Media Time
	pms->SetMediaTime(NULL, NULL);

	// Preroll
	pms->SetPreroll(FALSE);

	// Discontinuity
	pms->SetDiscontinuity(FALSE);

	// SyncPoint
    pms->SetSyncPoint(TRUE);

	// Data
	BYTE* sampleData;
	flLong sampleDataLen;
	pms->GetPointer(&sampleData);
	sampleDataLen = pms->GetActualDataLength();

	_digitalClock->draw(sampleData, sampleDataLen);

	return S_OK;
}
//------------------------------------------------------------------------------
void
CDigitalClockStream::AdjustRate(flInt64 deliverProcTime)
{
	if (_digitalClock->_frameRate == 0)
		return ;

	flInt64 interval = 10000000 / (flInt64)_digitalClock->_frameRate;
	flInt64 adjustTime = interval - (deliverProcTime + 1);

	if (30000 < adjustTime)
		Sleep((flULong)((adjustTime - 30000) / 10000));
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
HRESULT
CDigitalClockStream::DecideBufferSize(IMemAllocator *pAlloc,
                            ALLOCATOR_PROPERTIES *pProperties)
{
    CAutoLock cAutoLock(_source->pStateLock());
    ASSERT(pAlloc);
    ASSERT(pProperties);
    HRESULT hr = NOERROR;

	flULong sampleSize = m_mt.GetSampleSize();
	pProperties->cBuffers = 1;
	pProperties->cbBuffer = sampleSize;

    ASSERT(pProperties->cbBuffer);

    ALLOCATOR_PROPERTIES Actual;
    hr = pAlloc->SetProperties(pProperties, &Actual);
    if (FAILED(hr))
        return hr;

    if (Actual.cbBuffer < pProperties->cbBuffer)
        return E_FAIL;

    ASSERT(Actual.cBuffers == 1);
    return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CDigitalClockStream::SetMediaType(const CMediaType *pmt)
{
    CAutoLock cAutoLock(_source->pStateLock());
	
    // Pass the call up to my base class
    HRESULT hr = CFLSourceStream::SetMediaType(pmt);
	
	return hr;
}
//------------------------------------------------------------------------------
HRESULT
CDigitalClockStream::GetMediaType(CMediaType *pmt)
{
	CAutoLock cAutoLock(_source->pStateLock());
	CheckPointer(pmt, E_POINTER);

	flUInt w, h;

	switch(_digitalClock->_outVideoResolution)
	{
	case IDigitalClock::DCVR_SQCIF:		w = 128;	h = 96;		break;
	case IDigitalClock::DCVR_QCIF:		w = 176;	h = 144;	break;
	case IDigitalClock::DCVR_CIF:		w = 352;	h = 288;	break;
	case IDigitalClock::DCVR_160x120:	w = 160;	h = 120;	break;
	case IDigitalClock::DCVR_320x240:	w = 320;	h = 240;	break;
	case IDigitalClock::DCVR_360x240:	w = 360;	h = 240;	break;
	case IDigitalClock::DCVR_640x480:	w = 640;	h = 480;	break;
	case IDigitalClock::DCVR_720x480:	w = 720;	h = 480;	break;
	case IDigitalClock::DCVR_800x600:	w = 800;	h = 600;	break;
	case IDigitalClock::DCVR_1024x768:	w = 1024;	h = 768;	break;
	case IDigitalClock::DCVR_1280x1024:	w = 1280;	h = 1024;	break;
	case IDigitalClock::DCVR_1600x1200:	w = 1600;	h = 1200;	break;
	case IDigitalClock::DCVR_HDTV720P:	w = 1280;	h = 720;	break;
	case IDigitalClock::DCVR_HDTV1080I:	w = 1920;	h = 1080;	break;
	case IDigitalClock::DCVR_SDTV480P:	w = 852;	h = 480;	break;
	case IDigitalClock::DCVR_SDTV480I:	w = 852;	h = 480;	break;
	}

	VIDEOINFOHEADER *pvi = (VIDEOINFOHEADER *)pmt->AllocFormatBuffer(sizeof(VIDEOINFOHEADER));
	if (pvi == 0) 
		return(E_OUTOFMEMORY);

	ZeroMemory(pvi, pmt->cbFormat);   
	pvi->AvgTimePerFrame = (10000000 / _digitalClock->_frameRate);

	pvi->bmiHeader.biSize			= sizeof(BITMAPINFOHEADER);
	pvi->bmiHeader.biWidth			= w;
	pvi->bmiHeader.biHeight			= h;
	pvi->bmiHeader.biPlanes			= 1;
	pvi->bmiHeader.biBitCount		= 24;
	pvi->bmiHeader.biCompression	= BI_RGB;
	pvi->bmiHeader.biXPelsPerMeter	= 0;
	pvi->bmiHeader.biYPelsPerMeter	= 0;
	pvi->bmiHeader.biClrUsed		= 0;
	pvi->bmiHeader.biClrImportant	= 0;
	pvi->bmiHeader.biSizeImage		= GetBitmapSize(&pvi->bmiHeader);

	SetRectEmpty(&(pvi->rcSource));
	SetRectEmpty(&(pvi->rcTarget));

	pmt->SetType(&MEDIATYPE_Video);
	pmt->SetFormatType(&FORMAT_VideoInfo);
	pmt->SetTemporalCompression(FALSE);

	const GUID SubTypeGUID = GetBitmapSubtype(&pvi->bmiHeader);
	pmt->SetSubtype(&SubTypeGUID);
	pmt->SetSampleSize(pvi->bmiHeader.biSizeImage);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CDigitalClockStream::OnThreadCreate()
{
	_streamTime = 0;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDigitalClockStream::Notify(IBaseFilter * pSender, Quality q)
{
	return S_OK;
}
//------------------------------------------------------------------------------
