/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>
#include <flDrawTools/DrawCaption.h>
#include <flDrawTools/DrawCaptionProp.h>
#include <flBase/flNotify.h>
#include <math.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
CUnknown * WINAPI
CDrawCaption::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
	CDrawCaption *pNewObject = new CDrawCaption(NAME("DrawCaption"), punk, phr);
	if (pNewObject == NULL)
	{
		*phr = E_OUTOFMEMORY;
	}
	return pNewObject;
}
//------------------------------------------------------------------------------
CDrawCaption::CDrawCaption(TCHAR *tszName, LPUNKNOWN punk, HRESULT *phr) :
CTransInPlaceFilter(tszName, punk, CLSID_DrawCaption, phr),
CPersistStream(punk, phr)

{
	_captionOn			= true;
	_caption[0]			= 0;
	_fontSize			= 16;

	_textDraw			= new flTextDraw();
	_textDraw->setTextFontSize(_fontSize);
	_textDraw->setTextAlign(flTextDraw::TDTA_CENTER);
}
//------------------------------------------------------------------------------
CDrawCaption::~CDrawCaption()
{
	delete _textDraw;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawCaption::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
	CheckPointer(ppv, E_POINTER);

	if (riid == IID_IDrawCaption)
	{
		return GetInterface((IDrawCaption *)this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
		return GetInterface((ISpecifyPropertyPages *)this, ppv);
	}
	else
	{
		return CTransInPlaceFilter::NonDelegatingQueryInterface(riid, ppv);
	}
}
//------------------------------------------------------------------------------
HRESULT
CDrawCaption::CheckInputType(const CMediaType* mtIn)
{
	if (IsEqualGUID(mtIn->majortype, MEDIATYPE_Video) &&
		IsEqualGUID(mtIn->subtype, MEDIASUBTYPE_RGB24) &&
		IsEqualGUID(mtIn->formattype, FORMAT_VideoInfo))
		return S_OK;

	return VFW_E_TYPE_NOT_ACCEPTED;
}
//------------------------------------------------------------------------------
HRESULT
CDrawCaption::SetMediaType(PIN_DIRECTION direction, const CMediaType *pmt)
{
	VIDEOINFO *vinfo = (VIDEOINFO *)pmt->pbFormat;
	_textDraw->setVideoSize(vinfo->bmiHeader.biWidth, vinfo->bmiHeader.biHeight);
	_textDraw->setTextArea(vinfo->bmiHeader.biWidth, _fontSize + 2);
	_textDraw->setPosition(0, vinfo->bmiHeader.biHeight - (_fontSize + 2));
	return NOERROR;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
STDMETHODIMP
CDrawCaption::GetClassID(CLSID *pClsid)
{
	*pClsid = CLSID_DrawCaption;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CDrawCaption::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	WRITEOUT(_captionOn);
	WRITENOUT(_caption, DC_MAXIMUM_CAPTION_LENGTH);
	WRITEOUT(_fontSize);
	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CDrawCaption::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	READIN(_captionOn);
	READNIN(_caption, DC_MAXIMUM_CAPTION_LENGTH);
	READIN(_fontSize);

	_textDraw->setText(flString(_caption));
	_textDraw->setTextFontSize(_fontSize);

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CDrawCaption::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CDrawCaption::SizeMax()
{
	return sizeof(flBool) * 1 +
		   sizeof(flChar) * DC_MAXIMUM_CAPTION_LENGTH;
}
//------------------------------------------------------------------------------
// IDrawCaption
STDMETHODIMP
CDrawCaption::get_CaptionOn(flBool *captionOn)
{
	CAutoLock autoLock(&_filterLock);
	*captionOn = _captionOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawCaption::put_CaptionOn(flBool captionOn)
{
	CAutoLock autoLock(&_filterLock);
	_captionOn = captionOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawCaption::get_Caption(flChar* caption)
{
	CAutoLock autoLock(&_filterLock);
	strcpy(caption, _caption);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawCaption::put_Caption(const flChar* caption)
{
	CAutoLock autoLock(&_filterLock);
	if (caption == NULL)
		return NOERROR;
	if (DC_MAXIMUM_CAPTION_LENGTH <= strlen(caption) + 1)
	{
		strncpy(_caption, caption, DC_MAXIMUM_CAPTION_LENGTH);
		_caption[DC_MAXIMUM_CAPTION_LENGTH - 1] = 0;
	}
	else
	{
		strcpy(_caption, caption);
	}

	_textDraw->setText(flString(_caption));

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawCaption::get_FontSize(flUInt *fontSize)
{
	CAutoLock autoLock(&_filterLock);
	*fontSize = _fontSize;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawCaption::put_FontSize(flUInt fontSize)
{
	CAutoLock autoLock(&_filterLock);
	_fontSize = fontSize;

	_textDraw->setTextFontSize(fontSize);

	flUInt width, height;
	_textDraw->getVideoSize(width, height);
	if (width == 0 || height == 0)
		return NOERROR;

	_textDraw->setTextArea(width, _fontSize + 2);
	_textDraw->setPosition(0, height - (_fontSize + 2));

	return NOERROR;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages
STDMETHODIMP
CDrawCaption::GetPages(CAUUID *pPages)
{
	pPages->cElems = 1;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 1);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_DrawCaptionPropertyPage;
	return NOERROR;
}
//------------------------------------------------------------------------------
// CTransformInPlaceFilter Overrides
HRESULT
CDrawCaption::Transform(IMediaSample *pSample)
{
	CAutoLock autoLock(&_filterLock);

	if (!_captionOn)
		return NOERROR;

	BYTE *data;
	pSample->GetPointer(&data);

	_textDraw->draw(data);

	return NOERROR;
}
//------------------------------------------------------------------------------
