/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>
#include <flDrawTools/DrawFrame.h>
#include <flDrawTools/DrawFrameProp.h>
#include <flBase/flNotify.h>
#include <math.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
CUnknown * WINAPI
CDrawFrame::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
	CDrawFrame *pNewObject = new CDrawFrame(NAME("DrawFrame"), punk, phr);
	if (pNewObject == NULL)
	{
		*phr = E_OUTOFMEMORY;
	}
	return pNewObject;
}
//------------------------------------------------------------------------------
CDrawFrame::CDrawFrame(TCHAR *tszName, LPUNKNOWN punk, HRESULT *phr) :
CTransInPlaceFilter(tszName, punk, CLSID_DrawFrame, phr),
CPersistStream(punk, phr)

{
	_frameOn			= true;
	_frameType			= DFFT_RECTANGLE;
	_pointerInDisable	= false;
	ZeroMemory((void *)_position, sizeof(flInt) * 3);
	_size[0]			= 100;
	_size[1]			= 100;
	_color[0]			= 0xff;
	_color[1]			= 0xff;
	_color[2]			= 0xff;
	_thickness			= 1;

	_pointerInputPin	= NULL;
}
//------------------------------------------------------------------------------
CDrawFrame::~CDrawFrame()
{
	delete _pointerInputPin;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
	CheckPointer(ppv, E_POINTER);

	if (riid == IID_IDrawFrame)
	{
		return GetInterface((IDrawFrame *)this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
		return GetInterface((ISpecifyPropertyPages *)this, ppv);
	}
	else
	{
		return CTransInPlaceFilter::NonDelegatingQueryInterface(riid, ppv);
	}
}
//------------------------------------------------------------------------------
// CTransInPlaceFilter Overrides
int
CDrawFrame::GetPinCount()
{
	return 3;
}
//------------------------------------------------------------------------------
CBasePin*
CDrawFrame::GetPin(int n)
{
	if (m_pInput == NULL || m_pOutput == NULL || _pointerInputPin == NULL) 
	{
		HRESULT hr = S_OK;

		m_pInput = new CTransInPlaceInputPin(NAME("DrawFrame input pin"),
											this, &hr, L"Input");

		if (FAILED(hr) || m_pInput == NULL) 
			goto PINCREATEERROR;

		m_pOutput = new CTransInPlaceOutputPin(NAME("DrawFrame output pin"),
											this,	&hr, L"Output");

		if (FAILED(hr) || m_pOutput == NULL) 
			goto PINCREATEERROR;

		_pointerInputPin = new CPointerInputPin(NAME("Pointer input pin"),
											this, &hr, L"PointerIn");

		if (FAILED(hr) || _pointerInputPin == NULL)
			goto PINCREATEERROR;
	}

	switch(n) 
	{
	case 0:
		return m_pInput;
	case 1:
		return m_pOutput;
	case 2:
		return _pointerInputPin;
	}

	return NULL;

PINCREATEERROR:
	delete m_pInput;			m_pInput		= NULL;
	delete m_pOutput;			m_pOutput		= NULL;
	delete _pointerInputPin;	_pointerInputPin= NULL;
	return NULL;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::FindPin(LPCWSTR Id, IPin **ppPin)
{
	CheckPointer(ppPin, E_POINTER);
	ValidateReadWritePtr(ppPin, sizeof(IPin *));

	if (!lstrcmpW(Id, L"In"))
	{
		*ppPin = GetPin(0);
	}
	else if (!lstrcmpW(Id, L"Out"))
	{
		*ppPin = GetPin(1);
	}
	else if (!lstrcmpW(Id, L"PointerIn"))
	{
		*ppPin = GetPin(2);
	}
	else
	{
		*ppPin = NULL;
		return VFW_E_NOT_FOUND;
	}

	HRESULT hr = NOERROR;

	if(*ppPin)
	{
		(*ppPin)->AddRef();
	}
	else
	{
		hr = E_OUTOFMEMORY;
	}

	return hr;
}
//------------------------------------------------------------------------------
HRESULT
CDrawFrame::CheckInputType(const CMediaType* mtIn)
{
	if (IsEqualGUID(mtIn->majortype, MEDIATYPE_Video) &&
		IsEqualGUID(mtIn->subtype, MEDIASUBTYPE_RGB24) &&
		IsEqualGUID(mtIn->formattype, FORMAT_VideoInfo))
		return S_OK;

	return VFW_E_TYPE_NOT_ACCEPTED;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
STDMETHODIMP
CDrawFrame::GetClassID(CLSID *pClsid)
{
	*pClsid = CLSID_DrawFrame;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CDrawFrame::WriteToStream(IStream *pStream)
{
	HRESULT hr;

	WRITEOUT(_frameOn);
	WRITEOUT(_frameType);
	WRITEOUT(_pointerInDisable);
	WRITENOUT(_position, sizeof(flInt) * 2);
	WRITENOUT(_size, sizeof(flUInt) * 2);
	WRITENOUT(_color, sizeof(flByte) * 3);
	WRITEOUT(_thickness);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CDrawFrame::ReadFromStream(IStream *pStream)
{
	HRESULT hr;

	READIN(_frameOn);
	READIN(_frameType);
	READIN(_pointerInDisable);
	READNIN(_position, sizeof(flInt) * 2);
	READNIN(_size, sizeof(flUInt) * 2);
	READNIN(_color, sizeof(flByte) * 3);
	READIN(_thickness);

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CDrawFrame::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CDrawFrame::SizeMax()
{
	return sizeof(flBool) * 2 +
			sizeof(flInt) * 3 + 
			sizeof(flUInt) * 3 +
			sizeof(flByte) * 3;
}
//------------------------------------------------------------------------------
// IDrawFrame
STDMETHODIMP
CDrawFrame::get_FrameOn(flBool *frameOn)
{
	CAutoLock autoLock(&_filterLock);
	*frameOn = _frameOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::put_FrameOn(flBool frameOn)
{
	CAutoLock autoLock(&_filterLock);
	_frameOn = frameOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::get_FrameType(flInt *frameType)
{
	CAutoLock autoLock(&_filterLock);
	*frameType = _frameType;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::put_FrameType(flInt frameType)
{
	CAutoLock autoLock(&_filterLock);
	_frameType = frameType;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::get_PointerInDisable(flBool* flag)
{
	CAutoLock autoLock(&_filterLock);
	*flag = _pointerInDisable;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::put_PointerInDisable(flBool flag)
{
	CAutoLock autoLock(&_filterLock);
	_pointerInDisable = flag;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::get_Position(flInt* x, flInt* y)
{
	CAutoLock autoLock(&_filterLock);
	*x = _position[0];
	*y = _position[1];
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::put_Position(flInt x, flInt y)
{
	CAutoLock autoLock(&_filterLock);
	_position[0] = x;
	_position[1] = y;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::get_Size(flUInt* width, flUInt* height)
{
	CAutoLock autoLock(&_filterLock);
	*width = _size[0];
	*height= _size[1];
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::put_Size(flUInt width, flUInt height)
{
	CAutoLock autoLock(&_filterLock);
	_size[0] = width;
	_size[1] = height;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::get_Color(flByte* r, flByte* g, flByte* b)
{
	CAutoLock autoLock(&_filterLock);
	*r = _color[2];
	*g = _color[1];
	*b = _color[0];
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::put_Color(flByte r, flByte g, flByte b)
{
	CAutoLock autoLock(&_filterLock);
	_color[2] = r;
	_color[1] = g;
	_color[0] = b;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::get_Thickness(flUInt* thickness)
{
	CAutoLock autoLock(&_filterLock);
	*thickness = _thickness;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CDrawFrame::put_Thickness(flUInt thickness)
{
	CAutoLock autoLock(&_filterLock);
	_thickness = thickness;
	if (5 < _thickness)
		_thickness = 5;
	return NOERROR;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages
STDMETHODIMP
CDrawFrame::GetPages(CAUUID *pPages)
{
	pPages->cElems = 1;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 1);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_DrawFramePropertyPage;
	return NOERROR;
}
//------------------------------------------------------------------------------
// CTransformInPlaceFilter Overrides
HRESULT
CDrawFrame::Transform(IMediaSample *pSample)
{
	CAutoLock autoLock(&_filterLock);

	if (!_frameOn)
		return NOERROR;

    flByte *buffer;
	flInt size;
	flInt sw, sh;
	flInt x1, x2, y1, y2;

    size = pSample->GetActualDataLength();
    pSample->GetPointer(&buffer);

	// Get Image Size
	CMediaType imt = m_pInput->CurrentMediaType();
	VIDEOINFO* ivinfo = (VIDEOINFO *)imt.Format();
	sw = ivinfo->bmiHeader.biWidth;
	sh = ivinfo->bmiHeader.biHeight;

	// Get Region
	if (_pointerInDisable || _pointerInputPin->IsConnected() == FALSE)
	{
		x1 = _position[0];
		y1 = sh - _size[1] - _position[1];
	} else
	{
		flFloat cx, cy;
		_pointerInfo.getNormalizedPointerPosition(cx, cy);
		x1 = (flInt)(flFloat(sw) * cx);
		y1 = sh - _size[1] - (flInt)(flFloat(sh) * cy);
	}
	x2 = x1 + _size[0] - 1;
	y2 = y1 + _size[1] - 1;

	switch(_frameType)
	{
	default:
	case DFFT_RECTANGLE:
		{
			flInt xx1 = x1;
			flInt xx2 = x2;

			if (xx1 < 0)	xx1 = 0;
			if (xx2 < 0)	xx2 = 0;
			if (sw <= xx1)	xx1 = sw - 1;
			if (sw <= xx2)	xx2 = sw - 1;

			// Horiz Top
			if (xx1 != xx2)
			{
				for(flInt j = y1; j < y1 + (flInt)_thickness; j++)
				{
					if (0 <= j && j < sh)
					{
						for(flInt i = xx1; i <= xx2; i++)
							CopyMemory(buffer + (i + j * sw) * 3, _color, sizeof(flByte) * 3);
					}
				}

				// Horiz Bottom
				for(flInt j = y2 - (flInt)_thickness + 1; j < y2+ 1; j++)
				{
					if (0 <= j && j < sh)
					{
						for(flInt i = xx1; i <= xx2; i++)
							CopyMemory(buffer + (i + j * sw) * 3, _color, sizeof(flByte) * 3);
					}
				}
			}

			if (y1 < 0)		y1 = 0;
			if (y2 < 0)		y2 = 0;
			if (sh <= y1)	y1 = sh - 1;
			if (sh <= y2)	y2 = sh - 1;

			// Vert Left
			if (y1 != y2)
			{
				for(flInt j = x1; j < x1 + (flInt)_thickness; j++)
				{
					if (0 <= j && j < sw)
					{
						for(flInt i = y1; i <= y2; i++)
							CopyMemory(buffer + (j + i * sw) * 3, _color, sizeof(flByte) * 3);
					}
				}

				// Vert Right
				for(flInt j = x2 - (flInt)_thickness + 1; j < x2 + 1; j++)
				{
					if (0 <= j && j < sw)
					{
						for(flInt i = y1; i <= y2; i++)
							CopyMemory(buffer + (j + i * sw) * 3, _color, sizeof(flByte) * 3);
					}
				}
			}
		}
		break;
	case DFFT_OVAL:
		{
			flInt fx1 = x1;
			flInt fy1 = y1;
			flInt fw = x2 - x1;
			flInt fh = y2 - y1;

			flInt xa, xb, yy;
			flInt oxa, oxb;
			flFloat th;

			for(flInt tk = 0; tk < (flInt)_thickness; tk++)
			{
				for(flInt y = 0; y <= fh; y++)
				{
					th = flFloat(y) / flFloat(fh) * 2.0f - 1.0f;

					xa = flInt(flFloat(fw) * (1.0f - sqrtf(1.0f - th * th)) * 0.5f);
					xb = -xa + fw;

					xa += fx1;
					xb += fx1;
					yy = y + fy1;

					if (y == 0)
					{
						oxa = xa;
						oxb = xb;
					}

#define DRAW_POINT								\
if (0 <= x && x < sw && 0 <= yy && yy < sh)		\
	CopyMemory(buffer + (x + yy * sw) * 3,		\
			_color, sizeof(flByte) * 3);

					// Left Arc
					if (oxa < xa)
					{
						for(flInt x = oxa; x <= xa; x++)
							DRAW_POINT;
					}
					else
					{
						for(flInt x = xa; x <= oxa; x++)
							DRAW_POINT;
					}

					// Right Arc
					if (oxb < xb)
					{
						for(flInt x = oxb; x <= xb; x++)
							DRAW_POINT;
					}
					else
					{
						for(flInt x = xb; x <= oxb; x++)
							DRAW_POINT;
					}
#undef DRAW_POINT
					oxa = xa;
					oxb = xb;
				}

				fx1 ++;
				fy1 ++;

				fw -= 2;
				fh -= 2;
			}
		}
		break;
	}

	return NOERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
HRESULT
CDrawFrame::ReceivePointer(IMediaSample *pSource, CPointerInputPin *ppin)
{
	// Get Media Sample data
	flLong dataLength;
	BYTE *data;
	dataLength = pSource->GetActualDataLength();
	pSource->GetPointer(&data);

	// Setup New PointerInfo
	CAutoLock autoLock(&_filterLock);
	_pointerInfo.setChanged(data, dataLength);

	return NO_ERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CPointerInputPin::CPointerInputPin(TCHAR *pObjectName,
						CDrawFrame *pDrawFrame, HRESULT * phr, LPCWSTR pName) :
CBaseInputPin(pObjectName, pDrawFrame, &pDrawFrame->m_csFilter, phr, pName)
{
	DbgLog((LOG_TRACE,2,TEXT("CPointerInputPin::CPointerInputPin")));
	_drawFrame  = pDrawFrame;
}
//------------------------------------------------------------------------------
HRESULT
CPointerInputPin::CheckMediaType(const CMediaType* pmt)
{
	if (*pmt->Type() == MEDIATYPE_Stream &&
		*pmt->Subtype() == MEDIASUBTYPE_PointerInfo)
		return NOERROR;
	else
		return S_FALSE;
}
//------------------------------------------------------------------------------
// IMemInputPin
HRESULT
CPointerInputPin::Receive(IMediaSample * pSample)
{
	HRESULT hr;
	CAutoLock lck(&_drawFrame->m_csReceive);
	ASSERT(pSample);

	hr = CBaseInputPin::Receive(pSample);
	if (S_OK == hr)
		hr = _drawFrame->ReceivePointer(pSample, this);

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInputPin::BeginFlush(void)
{
	CAutoLock lck(&_drawFrame->m_csFilter);
	HRESULT hr = CBaseInputPin::BeginFlush();
	if (FAILED(hr))
		return hr;

	return S_OK;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInputPin::EndFlush(void)
{
	CAutoLock lck(&_drawFrame->m_csFilter);
	return CBaseInputPin::EndFlush();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInputPin::EndOfStream(void)
{
	CAutoLock lck(&_drawFrame->m_csReceive);
	HRESULT hr = CheckStreaming();
	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInputPin::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate)
{
	return CBasePin::NewSegment(tStart, tStop, dRate);
}
//------------------------------------------------------------------------------
HRESULT
CPointerInputPin::CheckStreaming()
{
	ASSERT(IsConnected());

	if(m_bFlushing)
		return S_FALSE;

	if(IsStopped())
		return VFW_E_WRONG_STATE;

	if(m_bRunTimeError)
		return VFW_E_RUNTIME_ERROR;

	return S_OK;
}
//------------------------------------------------------------------------------
