/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "Stdafx.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <streams.h>

#include <flFilterCommon/flBufferStream.h>
#include <flFilterCommon/flBufferInputPin.h>
#include <flFilterCommon/flBuffer.h>
#include <flBase/flSystemTime.h>
//------------------------------------------------------------------------------
CFLBufferStream::CFLBufferStream(TCHAR *pObjectName,
								 HRESULT *phr,
								 CFLBuffer *filter,
								 LPCWSTR pPinName)
: CBaseOutputPin(pObjectName, filter, &filter->_csFilter, phr, pPinName)
{
	_position		= NULL;
	_bufferFilter	= filter;

	_absTime		= new flSystemTime();
}
//------------------------------------------------------------------------------
CFLBufferStream::~CFLBufferStream(void)
{
	delete _absTime;

	if (_position != NULL)
		_position->Release();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CFLBufferStream::QueryId(LPWSTR *Id)
{
    return AMGetWideString(L"Out", Id);
}
//------------------------------------------------------------------------------
STDMETHODIMP
CFLBufferStream::NonDelegatingQueryInterface(REFIID riid, void **ppv)
{
	CheckPointer(ppv, E_POINTER);
	ValidateReadWritePtr(ppv, sizeof(PVOID));
	*ppv = NULL;

	if (riid == IID_IMediaPosition || riid == IID_IMediaSeeking)
	{
		ASSERT(_bufferFilter->_bufferInputPin != NULL);

		if (_position == NULL)
		{
			HRESULT hr = CreatePosPassThru(
								GetOwner(),
								FALSE,
								(IPin *)_bufferFilter->_bufferInputPin,
								&_position);
			if (FAILED(hr))
				return hr;
		}
		return _position->QueryInterface(riid, ppv);
	} else {
		return CBaseOutputPin::NonDelegatingQueryInterface(riid, ppv);
	}
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::Active(void)
{
	CAutoLock lock(&_bufferFilter->_csFilter);

	HRESULT hr;

	if (_bufferFilter->IsActive())
		return S_FALSE;

	if (!IsConnected())
		return NOERROR;

	hr = CBaseOutputPin::Active();
	if (FAILED(hr))
		return hr;

	ASSERT(!ThreadExists());

	if (!Create())
		return E_FAIL;

	hr = Init();
	if (FAILED(hr))
		return hr;

	return Pause();
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::Inactive(void)
{
	CAutoLock lock(&_bufferFilter->_csFilter);

	HRESULT hr;

	if (!IsConnected())
		return NOERROR;

	hr = CBaseOutputPin::Inactive();
	if (FAILED(hr))
		return hr;

	if (ThreadExists())
	{
		hr = Stop();

		if (FAILED(hr))
			return hr;

		hr = Exit();
		if (FAILED(hr))
			return hr;

		Close();
	}

	// hr = CBaseOutputPin::Inactive();  // call this first to Decommit the allocator
	//if (FAILED(hr)) {
	//	return hr;
	//}

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CFLBufferStream::ThreadProc(void)
{
	Command com;

	do {
		com = GetRequest();
		if (com != CMD_INIT)
		{
			DbgLog((LOG_ERROR, 1, TEXT("Thread expected init command")));
			Reply((DWORD) E_UNEXPECTED);
		}
	} while (com != CMD_INIT);

	DbgLog((LOG_TRACE, 1, TEXT("CFLBufferStream worker thread initializing")));

	Reply(NOERROR);

	Command cmd;
	do {
		cmd = GetRequest();

		switch(cmd)
		{
		case CMD_EXIT:
			Reply(NOERROR);
			break;

		case CMD_RUN:
			DbgLog((LOG_ERROR, 1, TEXT("CMD_RUN received before a CMD_PAUSE???")));

		case CMD_PAUSE:
			Reply(NOERROR);
			DoBufferProcessingLoop();
			break;

		case CMD_STOP:
			Reply(NOERROR);
			break;

		default:
			DbgLog((LOG_ERROR, 1, TEXT("Unknown command %d received!"), cmd));
			Reply((DWORD) E_NOTIMPL);
			break;
		}
	} while (cmd != CMD_EXIT);

	HRESULT hr = _bufferFilter->OnEndDelivery();
	if (FAILED(hr))
	{
		DbgLog((LOG_ERROR, 1, TEXT("CFLBufferStream::OnThreadDestroy failed. Exiting thread.")));
		return 1;
	}

	DbgLog((LOG_TRACE, 1, TEXT("CFLBufferStream worker thread exiting")));
	return 0;
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::DoBufferProcessingLoop(void)
{
	Command com;

	_bufferFilter->OnStartDelivery();

	do {
		while(!CheckRequest(&com))
		{
			IMediaSample *pSample;

			HRESULT hr = _bufferFilter->InitializeOutputSample(&pSample);
			if (FAILED(hr))
			{
				Sleep(1);
				continue;
			}

			// o^C~Op̎v
			_absTime->update();
			flInt64 deliverBegin = _absTime->getTime();

			hr = _bufferFilter->Deliver(pSample);
			if (hr == S_OK)
			{
				hr = Deliver(pSample);
				pSample->Release();

				if(hr != S_OK)
				{
					DbgLog((LOG_TRACE, 2, TEXT("Deliver() returned %08x; stopping"), hr));
					return S_OK;
				}

				// o^C~O̒
				_absTime->update();
				_bufferFilter->AdjustDeliverRate(_absTime->getTime() - deliverBegin);
			}
			else
			{
				pSample->Release();
			}
		}

		if (com == CMD_RUN || com == CMD_PAUSE)
		{
			Reply(NOERROR);
		}
		else if (com != CMD_STOP)
		{
			Reply((DWORD) E_UNEXPECTED);
			DbgLog((LOG_ERROR, 1, TEXT("Unexpected command!!!")));
		}

	} while (com != CMD_STOP);

	return S_FALSE;
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::CheckConnect(IPin *pPin)
{
	ASSERT(_bufferFilter->_bufferInputPin != NULL);
	if (_bufferFilter->_bufferInputPin->IsConnected() == FALSE)
		return E_UNEXPECTED;

	HRESULT hr = _bufferFilter->CheckConnect(PINDIR_OUTPUT, pPin);
	if (FAILED(hr))
		return hr;

	return CBaseOutputPin::CheckConnect(pPin);
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::BreakConnect()
{
	ASSERT(IsStopped());
	_bufferFilter->BreakConnect(PINDIR_OUTPUT);
	return CBaseOutputPin::BreakConnect();
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::CompleteConnect(IPin *pReceivePin)
{
    HRESULT hr = _bufferFilter->CompleteConnect(PINDIR_OUTPUT, pReceivePin);
    if (FAILED(hr))
        return hr;

    return CBaseOutputPin::CompleteConnect(pReceivePin);
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::SetMediaType(const CMediaType *pmtOut)
{
	ASSERT(_bufferFilter->_bufferInputPin != NULL);
	ASSERT(_bufferFilter->_bufferInputPin->CurrentMediaType().IsValid());

	HRESULT hr = NOERROR;

	hr = CBasePin::SetMediaType(pmtOut);
	if (FAILED(hr))
		return hr;

	return _bufferFilter->SetMediaType(PINDIR_OUTPUT, pmtOut);
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::GetMediaType(int iPosition, CMediaType *pMediaType)
{
	ASSERT(_bufferFilter->_bufferInputPin != NULL);

	CAutoLock lock(&_bufferFilter->_csFilter);

	if (_bufferFilter->_bufferInputPin->IsConnected())
		return _bufferFilter->GetMediaType(iPosition, pMediaType);
	else
		return VFW_S_NO_MORE_ITEMS;
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::CheckMediaType(const CMediaType *pMediaType)
{
	ASSERT(_bufferFilter->_bufferInputPin != NULL);

	CAutoLock lock(&_bufferFilter->_csFilter);

	if (_bufferFilter->_bufferInputPin->IsConnected() == FALSE)
		return E_INVALIDARG;

	return _bufferFilter->CheckTransform(
					&_bufferFilter->_bufferInputPin->CurrentMediaType(),
					pMediaType);
}
//------------------------------------------------------------------------------
HRESULT
CFLBufferStream::DecideBufferSize(IMemAllocator * pAllocator,
								  ALLOCATOR_PROPERTIES * pPropInputRequest)
{
    return _bufferFilter->DecideBufferSize(pAllocator, pPropInputRequest);
}
//------------------------------------------------------------------------------
STDMETHODIMP
CFLBufferStream::Notify(IBaseFilter * pSender, Quality q)
{
	UNREFERENCED_PARAMETER(pSender);
	ValidateReadPtr(pSender, sizeof(IBaseFilter));

	HRESULT hr = _bufferFilter->AlterQuality(q);
	if (hr != S_FALSE)
		return hr;

	ASSERT(_bufferFilter->_bufferInputPin != NULL);

	return _bufferFilter->_bufferInputPin->PassNotify(q);
}
//------------------------------------------------------------------------------
CMediaType&
CFLBufferStream::CurrentMediaType()
{
	return m_mt;
}
//------------------------------------------------------------------------------
