/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "Stdafx.h"
#include <flFilterCommon/flLocusDraw.h>
#include <math.h>
#include <limits.h>
#ifdef _DEBUG
#include <flBase/flNotify.h>
#endif
//------------------------------------------------------------------------------
flLocusDraw::flLocusDraw(flUInt videoWidth, flUInt videoHeight) :
flShapeDraw(videoWidth, videoHeight)
{
	_points.setGrowSize(128);
	_color[0]			= 0xff;
	_color[1]			= 0xff;
	_color[2]			= 0xff;
	_edgingOn			= false;
	_brokenLineOn		= false;
	_lineWidth			= 1;
}
//------------------------------------------------------------------------------
flLocusDraw::flLocusDraw() :
flShapeDraw()
{
	_points.setGrowSize(128);
	_color[0]			= 0xff;
	_color[1]			= 0xff;
	_color[2]			= 0xff;
	_edgingOn			= false;
	_brokenLineOn		= false;
	_lineWidth			= 1;
}
//------------------------------------------------------------------------------
flLocusDraw::~flLocusDraw()
{
}
//------------------------------------------------------------------------------
void
flLocusDraw::draw(flByte* data)
{
	flUInt size = _points.getSize();
	if (size == 0)
		return ;

	flShort x, y;
	flByte color[3], colorInv[3];
	flUInt breakCount;
	flBool fill;

	flShort rad = (flShort)_lineWidth / 2;
	flUInt breakTiming = 10 * ((_lineWidth / 4) + 1);

	color[2] = _color[0];
	color[1] = _color[1];
	color[0] = _color[2];

	colorInv[0] = 0xff - color[0];
	colorInv[1] = 0xff - color[1];
	colorInv[2] = 0xff - color[2];

	// Locus Edging
	if (_edgingOn)
	{
		breakCount = 0;
		fill = true;

		flShort brad, erad;

		if (1 < _lineWidth)
		{
			brad = rad + 1;
			erad = rad + 1;
		}
		else
		{
			brad = 1;
			erad = 2;
		}

		for(flUInt i = 0; i < size; i++)
		{
			x = _points[i].x;
			y = _points[i].y;

			if (_brokenLineOn)
				breakCount++;

			if (_brokenLineOn && ((breakCount / breakTiming) % 2) != 0)
			{
				fill = true;
				continue;
			}

			if (x == SHRT_MAX || y == SHRT_MAX)
			{
				fill = true;
				continue;
			}

			rectangle(data, x-brad, y-brad, x+erad, y+erad, colorInv, fill);
			fill = false;
		}
	}

	// Locus
	breakCount = 0;
	fill = true;

	for(flUInt i = 0; i < size; i++)
	{
		x = _points[i].x;
		y = _points[i].y;

		if (_brokenLineOn)
			breakCount++;

		if (_brokenLineOn && ((breakCount / breakTiming) % 2) != 0)
		{
			fill = true;
			continue;
		}

		if (1 < _lineWidth)
		{
			if (x == SHRT_MAX || y == SHRT_MAX)
			{
				fill = true;
				continue;
			}

			rectangle(data, x-rad, y-rad, x+rad, y+rad, color, fill);
			fill = false;
		}
		else
		{
			if (x < 0 || _videoWidth <= (flUInt)x ||
				y < 0 || _videoHeight <= (flUInt)y)
				continue;

            CopyMemory(data + x * 3 + y * _videoWidth * 3, color, 3);
		}
	}
}
//------------------------------------------------------------------------------
void
flLocusDraw::addPoint(flShort x, flShort y)
{
	flUInt size = _points.getSize();
	flPoint newPoint(x, -y + _videoHeight);

	if (size == 0)
	{
		if (newPoint.x != SHRT_MAX)
			_points.add(newPoint);
		return ;
	}

	const flPoint& prevPoint = _points[size - 1];

	if (prevPoint != newPoint)
	{
		if (newPoint.x == SHRT_MAX || prevPoint.x == SHRT_MAX)
		{
			_points.add(newPoint);
			return ;
		}

		flInt wx, wy;
		flFloat dx, dy, length, xx, yy;

		wx = newPoint.x - prevPoint.x;
		wy = newPoint.y - prevPoint.y;

		length = sqrtf(flFloat(wx * wx + wy * wy));
		dx = flFloat(wx) / length;
		dy = flFloat(wy) / length;

		xx = flFloat(prevPoint.x);
		yy = flFloat(prevPoint.y);

		flPoint interPoint(SHRT_MAX, SHRT_MAX);

		for(flInt i = 0; i < length; i++)
		{
			xx += dx;
			yy += dy;

			if (xx < 0.0f || _videoWidth <= (flUInt)xx ||
				yy < 0.0f || _videoHeight <= (flUInt)yy)
				continue;
			interPoint.x = flShort(xx);
			interPoint.y = flShort(yy);
			_points.add(interPoint);
		}

		if (interPoint != newPoint)
			_points.add(newPoint);
	}
}
//------------------------------------------------------------------------------
void
flLocusDraw::addDiscontinuousPoint()
{
	addPoint(SHRT_MAX, SHRT_MAX);
}
//------------------------------------------------------------------------------
void
flLocusDraw::clearPoints()
{
	_points.setSize(0);
}
//------------------------------------------------------------------------------
const flByte*
flLocusDraw::getColor() const
{
	return _color;
}
//------------------------------------------------------------------------------
void
flLocusDraw::setColor(const flByte* color)
{
	memcpy(_color, color, sizeof(_color));
}
//------------------------------------------------------------------------------
void
flLocusDraw::setEdgingOn(flBool flag)
{
	_edgingOn = flag;
}
//------------------------------------------------------------------------------
flBool
flLocusDraw::getEdgingOn() const
{
	return _edgingOn;
}
//------------------------------------------------------------------------------
void
flLocusDraw::setBrokenLineOn(flBool flag)
{
	_brokenLineOn = flag;
}
//------------------------------------------------------------------------------
flBool
flLocusDraw::getBrokenLineOn() const
{
	return _brokenLineOn;
}
//------------------------------------------------------------------------------
void
flLocusDraw::setLineWidth(flUInt width)
{
	_lineWidth = width;
}
//------------------------------------------------------------------------------
flUInt
flLocusDraw::getLineWidth() const
{
	return _lineWidth;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
void
flLocusDraw::rectangle(flByte* data, flShort x1, flShort y1, flShort x2, flShort y2,
						const flByte* color, flBool fill)
{
	flShort xx, yy;

	flShort vw = (flShort)_videoWidth;
	flShort vh = (flShort)_videoHeight;

	if (fill)
	{
		for(xx = x1; xx < x2; xx++)
		{
			for(yy = y1; yy < y2; yy++)
			{
				if (xx < 0 || vw <= xx ||
					yy < 0 || vh <= yy)
					continue;

				CopyMemory(data + xx * 3 + yy * vw * 3, color, 3);
			}
		}
	}
	else
	{
		yy = y1;
		if (0 <= yy && yy < vh)
		{
			for(xx = x1; xx < x2; xx++)
				if (0 <= xx && xx < vw)
					CopyMemory(data + xx * 3 + yy * vw * 3, color, 3);
		}

		yy = y2 - 1;
		if (0 <= yy && yy < vh)
		{
			for(xx = x1; xx < x2; xx++)
				if (0 <= xx && xx < vw)
					CopyMemory(data + xx * 3 + yy * vw * 3, color, 3);
		}

		xx = x1;
		if (0 <= xx && xx < vw)
		{
			for(yy = y1; yy < y2; yy++)
				if (0 <= yy && yy < vh)
					CopyMemory(data + xx * 3 + yy * vw * 3, color, 3);
		}

		xx = x2 - 1;
		if (0 <= xx && xx < vw)
		{
			for(yy = y1; yy < y2; yy++)
				if (0 <= yy && yy < vh)
					CopyMemory(data + xx * 3 + yy * vw * 3, color, 3);
		}
	}
}
//------------------------------------------------------------------------------
