/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "stdafx.h"
#include <windows.h>
#include <streams.h>
#include <initguid.h>
#include <olectl.h>
#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif
#include <assert.h>
#include <flFilterCommon/flMediaType.h>
//------------------------------------------------------------------------------
flMediaType::flMediaType() :
CMediaType()
{
	setDefaultMediaType();
	configMediaType();
}
//------------------------------------------------------------------------------
flMediaType::~flMediaType()
{
}
//------------------------------------------------------------------------------
void
flMediaType::setMediaType(MediaTypeEnum mediaType)
{
	_mediaType = mediaType;
	configMediaType();
}
//------------------------------------------------------------------------------
flMediaType::MediaTypeEnum
flMediaType::getMediaType() const
{
	return _mediaType;
}
//------------------------------------------------------------------------------
void
flMediaType::setVideoFormat(VideoFormatEnum videoFormat)
{
	_videoFormat = videoFormat;
	configMediaType();
}
//------------------------------------------------------------------------------
flMediaType::VideoFormatEnum
flMediaType::getVideoFormat() const
{
	return _videoFormat;
}
//------------------------------------------------------------------------------
void
flMediaType::setVideoResolution(VideoResolutionEnum videoResolution)
{
	_videoResolution = videoResolution;
	configMediaType();
}
//------------------------------------------------------------------------------
flMediaType::VideoResolutionEnum
flMediaType::getVideoResolution() const
{
	return _videoResolution;
}
//------------------------------------------------------------------------------
void
flMediaType::setAudioFormat(AudioFormatEnum audioFormat)
{
	_audioFormat = audioFormat;
	configMediaType();
}
//------------------------------------------------------------------------------
flMediaType::AudioFormatEnum
flMediaType::getAudioFormat() const
{
	return _audioFormat;
}
//------------------------------------------------------------------------------
void
flMediaType::setAudioFrequency(AudioFrequencyEnum audioFrequency)
{
	_audioFrequency = audioFrequency;
	configMediaType();
}
//------------------------------------------------------------------------------
flMediaType::AudioFrequencyEnum
flMediaType::getAudioFrequency() const
{
	return _audioFrequency;
}
//------------------------------------------------------------------------------
void
flMediaType::setAudioStereo(AudioStereoEnum audioStereo)
{
	_audioStereo = audioStereo;
	configMediaType();
}
//------------------------------------------------------------------------------
flMediaType::AudioStereoEnum
flMediaType::getAudioStereo() const
{
	return _audioStereo;
}
//------------------------------------------------------------------------------
void
flMediaType::setAudioBitCount(AudioBitCountEnum audioBitCount)
{
	_audioBitCount = audioBitCount;
	configMediaType();
}
//------------------------------------------------------------------------------
flMediaType::AudioBitCountEnum
flMediaType::getAudioBitCount() const
{
	return _audioBitCount;
}
//------------------------------------------------------------------------------
flMediaType&
flMediaType::operator=(const AM_MEDIA_TYPE& mt)
{
    Set(mt);
    return *this;
}
//------------------------------------------------------------------------------
flMediaType&
flMediaType::operator=(const CMediaType& mt)
{
    *this = (AM_MEDIA_TYPE &)mt;
    return *this;
}
//------------------------------------------------------------------------------
BOOL
flMediaType::operator==(const flMediaType& mt) const
{
	if (_mediaType != mt._mediaType)
		return FALSE;

	if (_mediaType == MT_VIDEO)
	{
		return (_videoFormat == mt._videoFormat &&
				_videoResolution == mt._videoResolution);
	}
	else if (_mediaType == MT_AUDIO)
	{
		return (_audioFormat == mt._audioFormat &&
				_audioFrequency == mt._audioFrequency &&
				_audioStereo == mt._audioStereo &&
				_audioBitCount == mt._audioBitCount);
	}
	else
		return TRUE;
}
//------------------------------------------------------------------------------
BOOL
flMediaType::operator!=(const flMediaType& mt) const
{
	if (*this == mt)
	{
		return FALSE;
	}
	return TRUE;
}
//------------------------------------------------------------------------------
HRESULT
flMediaType::Set(const AM_MEDIA_TYPE& mt)
{
	HRESULT hr = CMediaType::Set(mt);
	if (hr == S_OK)
		analyzeMediaType();
	return hr;
}
//------------------------------------------------------------------------------
void
flMediaType::setDefaultMediaType()
{
	_mediaType			= MT_VIDEO;
	_videoFormat		= VF_DVSD;
	_videoResolution	= VR_720x480;
	_audioFormat		= AF_MP3;
	_audioFrequency		= AQ_48KHz;
	_audioStereo		= AS_STEREO;
	_audioBitCount		= AB_16Bit;
}
//------------------------------------------------------------------------------
void
flMediaType::configMediaType()
{
	switch(_mediaType)
	{
	case MT_VIDEO:
		_configVideoMediaType();
		break;
	case MT_AUDIO:
		_configAudioMediaType();
		break;
	case MT_POINTER:
		_configPointerMediaType();
		break;
	case MT_UNKNOWN:
		_configUnknownMediaType();
		break;
	}
}
//------------------------------------------------------------------------------
void
flMediaType::_configVideoMediaType()
{
	VIDEOINFO *pvi = (VIDEOINFO *)AllocFormatBuffer(sizeof(VIDEOINFO));
	ZeroMemory(pvi, sizeof(VIDEOINFO));

	SetRectEmpty(&(pvi->rcSource));
	SetRectEmpty(&(pvi->rcTarget));

	// Video Format
	switch(_videoFormat)
	{
	case VF_RGB555:
		pvi->bmiHeader.biBitCount			= 16;
		pvi->bmiHeader.biCompression		= BI_BITFIELDS;

		bFixedSizeSamples					= TRUE;
		bTemporalCompression				= FALSE;
		break;

	case VF_RGB24:
		pvi->bmiHeader.biBitCount			= 24;
		pvi->bmiHeader.biCompression		= BI_RGB;

		bFixedSizeSamples					= TRUE;
		bTemporalCompression				= FALSE;
		break;

	case VF_MPEG4_2:
		pvi->bmiHeader.biBitCount			= 24;
		pvi->bmiHeader.biCompression		= 0x3234504d;

		bFixedSizeSamples					= FALSE;
		bTemporalCompression				= TRUE;
		break;

	case VF_MPEG4_3:
		pvi->bmiHeader.biBitCount			= 24;
		pvi->bmiHeader.biCompression		= 0x3334504d;

		bFixedSizeSamples					= FALSE;
		bTemporalCompression				= TRUE;
		break;

	case VF_DVSD:
		pvi->bmiHeader.biBitCount			= 24;
		pvi->bmiHeader.biCompression		= 0x64737664;

		bFixedSizeSamples					= TRUE;
		bTemporalCompression				= FALSE;
		break;

	case VF_H261:
		pvi->bmiHeader.biBitCount			= 24;
		pvi->bmiHeader.biCompression		= 0x3136326d;

		bFixedSizeSamples					= FALSE;
		bTemporalCompression				= TRUE;
		break;

	case VF_H263:
		pvi->bmiHeader.biBitCount			= 24;
		pvi->bmiHeader.biCompression		= 0x3336326d;

		bFixedSizeSamples					= FALSE;
		bTemporalCompression				= TRUE;
		break;
	case VF_MJPEG:
		pvi->bmiHeader.biBitCount			= 24;
		pvi->bmiHeader.biCompression		= 0x47504a4d;

		bFixedSizeSamples					= TRUE;
		bTemporalCompression				= FALSE;

		break;
	}

	// Video Resolution
	switch(_videoResolution)
	{
	case VR_SQCIF:
		pvi->bmiHeader.biWidth		= 128;
		pvi->bmiHeader.biHeight		= 96;
		break;
	case VR_QCIF:
		pvi->bmiHeader.biWidth		= 176;
		pvi->bmiHeader.biHeight		= 144;
		break;
	case VR_CIF:
		pvi->bmiHeader.biWidth		= 352;
		pvi->bmiHeader.biHeight		= 288;
		break;
	case VR_160x120:
		pvi->bmiHeader.biWidth		= 160;
		pvi->bmiHeader.biHeight		= 120;
		break;
	case VR_180x120:
		pvi->bmiHeader.biWidth		= 180;
		pvi->bmiHeader.biHeight		= 120;
		break;
	case VR_320x240:
		pvi->bmiHeader.biWidth		= 320;
		pvi->bmiHeader.biHeight		= 240;
		break;
	case VR_360x240:
		pvi->bmiHeader.biWidth		= 360;
		pvi->bmiHeader.biHeight		= 240;
		break;
	case VR_640x480:
		pvi->bmiHeader.biWidth		= 640;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_720x480:
		pvi->bmiHeader.biWidth		= 720;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_800x600:
		pvi->bmiHeader.biWidth		= 800;
		pvi->bmiHeader.biHeight		= 600;
		break;
	case VR_1024x768:
		pvi->bmiHeader.biWidth		= 1024;
		pvi->bmiHeader.biHeight		= 768;
		break;
	case VR_1280x1024:
		pvi->bmiHeader.biWidth		= 1280;
		pvi->bmiHeader.biHeight		= 1024;
		break;
	case VR_1600x1200:
		pvi->bmiHeader.biWidth		= 1600;
		pvi->bmiHeader.biHeight		= 1200;
		break;
	case VR_HDTV720P:
		pvi->bmiHeader.biWidth		= 1280;
		pvi->bmiHeader.biHeight		= 720;
		break;
	case VR_HDTV1080I:
		pvi->bmiHeader.biWidth		= 1920;
		pvi->bmiHeader.biHeight		= 1080;
		break;
	case VR_SDTV480P:
		pvi->bmiHeader.biWidth		= 852;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_SDTV480I:
		pvi->bmiHeader.biWidth		= 852;
		pvi->bmiHeader.biHeight		= 480;
		break;
	}

    pvi->bmiHeader.biSize			= sizeof(BITMAPINFOHEADER);
	pvi->bmiHeader.biSizeImage		= GetBitmapSize(&pvi->bmiHeader);
	pvi->bmiHeader.biPlanes			= 1;
	pvi->bmiHeader.biClrImportant	= 0;	
	pvi->bmiHeader.biClrUsed		= 0;

	majortype	= MEDIATYPE_Video;
	subtype		= GetBitmapSubtype(&pvi->bmiHeader);
	lSampleSize	= pvi->bmiHeader.biSizeImage;
	formattype	= FORMAT_VideoInfo;

}
//------------------------------------------------------------------------------
#include <mmreg.h>
void
flMediaType::_configAudioMediaType()
{
	WAVEFORMATEXTENSIBLE *pwf = (WAVEFORMATEXTENSIBLE *)AllocFormatBuffer(sizeof(WAVEFORMATEXTENSIBLE));
	ZeroMemory(pwf, sizeof(WAVEFORMATEXTENSIBLE));

	// Audio Frequency
	switch(_audioFrequency)
	{
	case AQ_8KHz:
		pwf->Format.nSamplesPerSec	= 8000; 
		break;
	case AQ_11KHz:
		pwf->Format.nSamplesPerSec	= 11025; 
		break;
	case AQ_16KHz:
		pwf->Format.nSamplesPerSec	= 16000; 
		break;
	case AQ_22KHz:
		pwf->Format.nSamplesPerSec	= 22050; 
		break;
	case AQ_44_1KHz:
		pwf->Format.nSamplesPerSec	= 44100; 
		break;
	case AQ_48KHz:
		pwf->Format.nSamplesPerSec	= 48000; 
		break;
	}

	// Audio Stereo
	switch(_audioStereo)
	{
	case AS_STEREO:
		pwf->Format.nChannels		= 2;
		break;
	case AS_MONO:
		pwf->Format.nChannels		= 1;
		break;
	}

	// Audio BitCount
	switch(_audioBitCount)
	{
	case AB_0Bit:
		pwf->Format.wBitsPerSample	= 0;
		break;
	case AB_8Bit:
		pwf->Format.wBitsPerSample	= 8;
		break;
	case AB_16Bit:
		pwf->Format.wBitsPerSample	= 16;
		break;
	}

	// Audio Format
	switch(_audioFormat)
	{
	case AF_MP3:
		bFixedSizeSamples				= TRUE;
		bTemporalCompression			= FALSE;
		lSampleSize						= 100000;
		subtype							= MEDIASUBTYPE_NULL;
		
		pwf->Format.wFormatTag			= WAVE_FORMAT_MPEGLAYER3;
		pwf->Format.nBlockAlign			= 2;
		pwf->Format.nAvgBytesPerSec		= 7000;
		pwf->Format.cbSize				= 12;
		pwf->Samples.wValidBitsPerSample= 1;
		pwf->Samples.wSamplesPerBlock	= 1;
		pwf->Samples.wReserved			= 1;
		//pwf->dwChannelMask			= 2;
		break;

	case AF_PCM:
		bFixedSizeSamples				= TRUE;
		bTemporalCompression			= FALSE;
		lSampleSize						= 100000;
		subtype							= MEDIASUBTYPE_PCM;

		pwf->Format.wFormatTag			= WAVE_FORMAT_PCM;
		pwf->Format.nBlockAlign			= pwf->Format.wBitsPerSample / 8 * pwf->Format.nChannels;
		pwf->Format.nAvgBytesPerSec		= pwf->Format.nSamplesPerSec * pwf->Format.nBlockAlign;
		pwf->Format.cbSize				= 0;
		pwf->Samples.wValidBitsPerSample= 0;
		pwf->Samples.wSamplesPerBlock	= 0;
		pwf->Samples.wReserved			= 0;
		//pwf->dwChannelMask			= 2;
		break;

	case AF_MPEG1:
		bFixedSizeSamples				= TRUE;
		bTemporalCompression			= FALSE;
		lSampleSize						= 100000;
		subtype							= MEDIASUBTYPE_MPEG1AudioPayload;

		pwf->Format.wFormatTag			= WAVE_FORMAT_MPEG;
		pwf->Format.nBlockAlign			= 1;
		pwf->Format.nAvgBytesPerSec		= 8000;
		pwf->Format.cbSize				= 22;
		pwf->Samples.wValidBitsPerSample= 2;
		pwf->Samples.wSamplesPerBlock	= 2;
		pwf->Samples.wReserved			= 2;
		break;
		
	}

	majortype	= MEDIATYPE_Audio;
	formattype	= FORMAT_WaveFormatEx;

}
//------------------------------------------------------------------------------
#include <flTypes/PointerInfo.h>
void
flMediaType::_configPointerMediaType()
{
	ResetFormatBuffer();

	bFixedSizeSamples		= FALSE;
	bTemporalCompression	= FALSE;
	majortype				= MEDIATYPE_Stream;
	subtype					= MEDIASUBTYPE_PointerInfo;
	lSampleSize				= 0;
	formattype				= FORMAT_None;
}
//------------------------------------------------------------------------------
void
flMediaType::_configUnknownMediaType()
{
	ResetFormatBuffer();

	bFixedSizeSamples		= TRUE;
	bTemporalCompression	= FALSE;
	majortype				= MEDIATYPE_NULL;
	subtype					= MEDIASUBTYPE_None;
	lSampleSize				= 0;
	formattype				= FORMAT_None;
}
//------------------------------------------------------------------------------
void
flMediaType::analyzeMediaType()
{
	if (IsEqualGUID(majortype, MEDIATYPE_Video))
		_analyzeVideoMediaType();
	else if (IsEqualGUID(majortype, MEDIATYPE_Audio))
		_analyzeAudioMediaType();
	else if (IsEqualGUID(majortype, MEDIATYPE_Stream))
		_analyzePointerMediaType();
	else
		_mediaType = MT_UNKNOWN;
}
//------------------------------------------------------------------------------
void
flMediaType::_analyzeVideoMediaType()
{
	_mediaType = MT_VIDEO;

	VIDEOINFO *pvi = (VIDEOINFO *)Format();

	// Video Format
	_videoFormat = VF_UNKNOWN;
	switch(pvi->bmiHeader.biBitCount)
	{
	case 16:
		// RGB555
		if (pvi->bmiHeader.biCompression == BI_BITFIELDS &&
			bFixedSizeSamples == TRUE &&
			bTemporalCompression == FALSE)
			_videoFormat = VF_RGB555;
		break;
	case 24:
		switch(pvi->bmiHeader.biCompression)
		{
		// RGB24
		case BI_RGB:
			if (bFixedSizeSamples == TRUE &&
				bTemporalCompression == FALSE)
				_videoFormat = VF_RGB24;
			break;
		// MPEG4_2
		case 0x3234504d:
			if (bFixedSizeSamples == FALSE &&
				bTemporalCompression == TRUE)
				_videoFormat = VF_MPEG4_2;
			break;
		// MPEG4_3
		case 0x3334504d:
			if (bFixedSizeSamples == FALSE &&
				bTemporalCompression == TRUE)
				_videoFormat = VF_MPEG4_3;
			break;
		// dvsd
		case 0x64737664:
			if (bFixedSizeSamples == TRUE &&
				bTemporalCompression == FALSE)
				_videoFormat = VF_DVSD;
			break;
		// H.261
		case 0x3136326d:
			if (bFixedSizeSamples == FALSE &&
				bTemporalCompression == TRUE)
				_videoFormat = VF_H261;
			break;
		// H.263
		case 0x3336326d:
			if (bFixedSizeSamples == FALSE &&
				bTemporalCompression == TRUE)
				_videoFormat = VF_H263;
			break;
		// MJPEG
		case 0x47504a4d:
			if (bFixedSizeSamples == TRUE &&
				bTemporalCompression == FALSE)
				_videoFormat = VF_MJPEG;
			break;
		}
		break;
	}

	// Video Resolution
	_videoResolution = VR_UNKNOWN;
	if (pvi->bmiHeader.biWidth == 128 && pvi->bmiHeader.biHeight == 96)
		_videoResolution = VR_SQCIF;
	else if (pvi->bmiHeader.biWidth == 176 && pvi->bmiHeader.biHeight == 144)
		_videoResolution = VR_QCIF;
	else if (pvi->bmiHeader.biWidth == 352 && pvi->bmiHeader.biHeight == 288)
		_videoResolution = VR_CIF;
	else if (pvi->bmiHeader.biWidth == 160 && pvi->bmiHeader.biHeight == 120)
		_videoResolution = VR_160x120;
	else if (pvi->bmiHeader.biWidth == 180 && pvi->bmiHeader.biHeight == 120)
		_videoResolution = VR_180x120;
	else if (pvi->bmiHeader.biWidth == 320 && pvi->bmiHeader.biHeight == 240)
		_videoResolution = VR_320x240;
	else if (pvi->bmiHeader.biWidth == 360 && pvi->bmiHeader.biHeight == 240)
		_videoResolution = VR_360x240;
	else if (pvi->bmiHeader.biWidth == 640 && pvi->bmiHeader.biHeight == 480)
		_videoResolution = VR_640x480;
	else if (pvi->bmiHeader.biWidth == 720 && pvi->bmiHeader.biHeight == 480)
		_videoResolution = VR_720x480;
	else if (pvi->bmiHeader.biWidth == 800 && pvi->bmiHeader.biHeight == 600)
		_videoResolution = VR_800x600;
	else if (pvi->bmiHeader.biWidth == 1024 && pvi->bmiHeader.biHeight == 768)
		_videoResolution = VR_1024x768;
	else if (pvi->bmiHeader.biWidth == 1280 && pvi->bmiHeader.biHeight == 1024)
		_videoResolution = VR_1280x1024;
	else if (pvi->bmiHeader.biWidth == 1600 && pvi->bmiHeader.biHeight == 1200)
		_videoResolution = VR_1600x1200;
	else if (pvi->bmiHeader.biWidth == 1280 && pvi->bmiHeader.biHeight == 720)
		_videoResolution = VR_HDTV720P;
	else if (pvi->bmiHeader.biWidth == 1920 && pvi->bmiHeader.biHeight == 1080)
		_videoResolution = VR_HDTV1080I;
	else if (pvi->bmiHeader.biWidth == 852 && pvi->bmiHeader.biHeight == 480)
		_videoResolution = VR_SDTV480P;
	else if (pvi->bmiHeader.biWidth == 852 && pvi->bmiHeader.biHeight == 480)
		_videoResolution = VR_SDTV480I;
}
//------------------------------------------------------------------------------
void
flMediaType::_analyzeAudioMediaType()
{
	_mediaType = MT_AUDIO;

	WAVEFORMATEXTENSIBLE *pwf = (WAVEFORMATEXTENSIBLE *)Format();

	// Audio Frequency
	_audioFrequency = AQ_UNKNOWN;
	switch(pwf->Format.nSamplesPerSec)
	{
	case 8000:
		_audioFrequency = AQ_8KHz;
		break;
	case 11025:
		_audioFrequency = AQ_11KHz;
		break;
	case 16000:
		_audioFrequency = AQ_16KHz;
		break;
	case 22050:
		_audioFrequency = AQ_22KHz;
		break;
	case 44100:
		_audioFrequency = AQ_44_1KHz;
		break;
	case 48000:
		_audioFrequency = AQ_48KHz;
		break;
	}

	// Audio Stereo
	_audioStereo = AS_UNKNOWN;
	switch(pwf->Format.nChannels)
	{
	case 2:
		_audioStereo = AS_STEREO;
		break;
	case 1:
		_audioStereo = AS_MONO;
		break;
	}

	// Audio BitCount
	_audioBitCount = AB_UNKNOWN;
	switch(pwf->Format.wBitsPerSample)
	{
	case 0:
		_audioBitCount = AB_0Bit;
		break;
	case 8:
		_audioBitCount = AB_8Bit;
		break;
	case 16:
		_audioBitCount = AB_16Bit;
		break;
	}

	// Audio Format
	_audioFormat = AF_UNKNOWN;
	if (bTemporalCompression == FALSE)
	{
		if (bFixedSizeSamples == TRUE)
		{
			// PCM
			if (pwf->Format.wFormatTag == WAVE_FORMAT_PCM)
				_audioFormat = AF_PCM;

			// MPEG1
			else if (pwf->Format.wFormatTag == WAVE_FORMAT_MPEG)
				_audioFormat = AF_MPEG1;

			// MP3
			if (pwf->Format.wFormatTag == WAVE_FORMAT_MPEGLAYER3)
				_audioFormat = AF_MP3;
		}
	}
}
//------------------------------------------------------------------------------
void
flMediaType::_analyzePointerMediaType()
{
	if (IsEqualGUID(MEDIATYPE_Stream, majortype) == TRUE &&
		IsEqualGUID(MEDIASUBTYPE_PointerInfo, subtype) == TRUE &&
		IsEqualGUID(FORMAT_None, formattype) == TRUE)
		_mediaType = MT_POINTER;
}
//------------------------------------------------------------------------------
