/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "Stdafx.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <streams.h>

#include <flFilterCommon/flMultiBufferStream.h>
#include <flFilterCommon/flMultiBufferInputPin.h>
#include <flFilterCommon/flMultiBuffer.h>
#include <flBase/flSystemTime.h>
//------------------------------------------------------------------------------
CFLMultiBufferStream::CFLMultiBufferStream(TCHAR *pObjectName,
								 HRESULT *phr,
								 CFLMultiBuffer *filter,
								 LPCWSTR pPinName)
: CBaseOutputPin(pObjectName, filter, &filter->_csFilter, phr, pPinName)
{
	_bufferFilter	= filter;
	_absTime		= new flSystemTime();
}
//------------------------------------------------------------------------------
CFLMultiBufferStream::~CFLMultiBufferStream(void)
{
	delete _absTime;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CFLMultiBufferStream::QueryId(LPWSTR *Id)
{
    return AMGetWideString(L"Out", Id);
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::Active(void)
{
	CAutoLock lock(&_bufferFilter->_csFilter);

	HRESULT hr;

	if (_bufferFilter->IsActive())
		return S_FALSE;

	if (!IsConnected())
		return NOERROR;

	hr = CBaseOutputPin::Active();
	if (FAILED(hr))
		return hr;

	ASSERT(!ThreadExists());

	if (!Create())
		return E_FAIL;

	hr = Init();
	if (FAILED(hr))
		return hr;

	return Pause();
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::Inactive(void)
{
	CAutoLock lock(&_bufferFilter->_csFilter);

	HRESULT hr;

	if (!IsConnected())
		return NOERROR;

	hr = CBaseOutputPin::Inactive();
	if (FAILED(hr))
		return hr;

	if (ThreadExists())
	{
		hr = Stop();

		if (FAILED(hr))
			return hr;

		hr = Exit();
		if (FAILED(hr))
			return hr;

		Close();
	}

	// hr = CBaseOutputPin::Inactive();  // call this first to Decommit the allocator
	//if (FAILED(hr)) {
	//	return hr;
	//}

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CFLMultiBufferStream::ThreadProc(void)
{
	Command com;

	do {
		com = GetRequest();
		if (com != CMD_INIT)
		{
			DbgLog((LOG_ERROR, 1, TEXT("Thread expected init command")));
			Reply((DWORD) E_UNEXPECTED);
		}
	} while (com != CMD_INIT);

	DbgLog((LOG_TRACE, 1, TEXT("CFLMultiBufferStream worker thread initializing")));

	// Initialisation suceeded
	Reply(NOERROR);

	Command cmd;
	do {
		cmd = GetRequest();

		switch(cmd)
		{
		case CMD_EXIT:
			Reply(NOERROR);
			break;

		case CMD_RUN:
			DbgLog((LOG_ERROR, 1, TEXT("CMD_RUN received before a CMD_PAUSE???")));
			// !!! fall through???

		case CMD_PAUSE:
			Reply(NOERROR);
			DoBufferProcessingLoop();
			break;

		case CMD_STOP:
			Reply(NOERROR);
			break;

		default:
			DbgLog((LOG_ERROR, 1, TEXT("Unknown command %d received!"), cmd));
			Reply((DWORD) E_NOTIMPL);
			break;
		}
	} while (cmd != CMD_EXIT);

	HRESULT hr = _bufferFilter->OnEndDelivery();
	if (FAILED(hr))
	{
		DbgLog((LOG_ERROR, 1, TEXT("CFLMultiBufferStream::OnThreadDestroy failed. Exiting thread.")));
		return 1;
	}

	DbgLog((LOG_TRACE, 1, TEXT("CFLMultiBufferStream worker thread exiting")));
	return 0;
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::DoBufferProcessingLoop(void)
{
	Command com;

	_bufferFilter->OnStartDelivery();

	do {
		while(!CheckRequest(&com))
		{
			IMediaSample *pSample;

			HRESULT hr = _bufferFilter->InitializeOutputSample(&pSample);
			if (FAILED(hr))
			{
				Sleep(1);
				continue;
			}

			// o^C~Op̎v
			_absTime->update();
			flInt64 deliverBegin = _absTime->getTime();

			hr = _bufferFilter->Deliver(pSample);
			if (hr == S_OK)
			{
				hr = Deliver(pSample);
				pSample->Release();

				if(hr != S_OK)
				{
					DbgLog((LOG_TRACE, 2, TEXT("Deliver() returned %08x; stopping"), hr));
					return S_OK;
				}

				// o^C~O̒
				_absTime->update();
				_bufferFilter->AdjustDeliverRate(_absTime->getTime() - deliverBegin);
			}
			else
			{
				pSample->Release();
			}
		}

		if (com == CMD_RUN || com == CMD_PAUSE)
		{
			Reply(NOERROR);
		}
		else if (com != CMD_STOP)
		{
			Reply((DWORD) E_UNEXPECTED);
			DbgLog((LOG_ERROR, 1, TEXT("Unexpected command!!!")));
		}

	} while (com != CMD_STOP);

	return S_FALSE;
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::CheckConnect(IPin *pPin)
{
	HRESULT hr = _bufferFilter->CheckOutputConnect(pPin);
	if (FAILED(hr))
		return hr;

	return CBaseOutputPin::CheckConnect(pPin);
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::BreakConnect()
{
	ASSERT(IsStopped());
	_bufferFilter->BreakOutputConnect();
	return CBaseOutputPin::BreakConnect();
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::CompleteConnect(IPin *pReceivePin)
{
    HRESULT hr = _bufferFilter->CompleteOutputConnect(pReceivePin);
    if (FAILED(hr))
        return hr;

    return CBaseOutputPin::CompleteConnect(pReceivePin);
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::SetMediaType(const CMediaType *pmtOut)
{
	HRESULT hr = NOERROR;

	hr = CBasePin::SetMediaType(pmtOut);
	if (FAILED(hr))
		return hr;

	return _bufferFilter->SetOutputMediaType(pmtOut);
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::GetMediaType(int iPosition, CMediaType *pMediaType)
{
	CAutoLock lock(&_bufferFilter->_csFilter);
	return _bufferFilter->GetOutputMediaType(iPosition, pMediaType);
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::CheckMediaType(const CMediaType *pMediaType)
{
	CAutoLock lock(&_bufferFilter->_csFilter);
	return _bufferFilter->CheckOutputMediaType(pMediaType);
}
//------------------------------------------------------------------------------
HRESULT
CFLMultiBufferStream::DecideBufferSize(IMemAllocator * pAlloc,
									   ALLOCATOR_PROPERTIES * pProp)
{
    return _bufferFilter->DecideBufferSize(pAlloc, pProp);
}
//------------------------------------------------------------------------------
STDMETHODIMP
CFLMultiBufferStream::Notify(IBaseFilter * pSender, Quality q)
{
	return E_NOTIMPL;
}
//------------------------------------------------------------------------------
CMediaType&
CFLMultiBufferStream::CurrentMediaType()
{
	return m_mt;
}
//------------------------------------------------------------------------------
