/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "Stdafx.h"
#include <flFilterCommon/flPCMAudio.h>
#include <math.h>
#include <assert.h>

#ifdef _DEBUG
#include <flBase/flNotify.h>
#endif
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
flPCMAudioBuffer::flPCMAudioBuffer()
{
	Buffer			= NULL;
	BufferLength	= 0;

	DataLength		= 0;
	StartTime		= 0;
	EndTime			= 0;
}
//------------------------------------------------------------------------------
flPCMAudioBuffer::~flPCMAudioBuffer()
{
	delete Buffer;
}
//------------------------------------------------------------------------------
void
flPCMAudioBuffer::setData(const flByte* data, flUInt dataLength)
{
	if (BufferLength < dataLength)
	{
		delete Buffer;
		Buffer = new flByte[dataLength];
		BufferLength = dataLength;
	}

	DataLength = dataLength;
	CopyMemory(Buffer, data, DataLength);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
flPCMAudio::flPCMAudio()
{
	ZeroMemory(&_format, sizeof(WAVEFORMATEX));
	setBufferCount(10);
}
//------------------------------------------------------------------------------
flPCMAudio::~flPCMAudio()
{
}
//------------------------------------------------------------------------------
void
flPCMAudio::setFormat(const WAVEFORMATEX *format)
{
	assert(format != NULL);
	assert(format->wFormatTag == WAVE_FORMAT_PCM);
	
	CopyMemory(&_format, format, sizeof(WAVEFORMATEX));

#ifdef _DEBUG
/*
	flNotify::post(flNotify::L_INFO, "nAvgBytesPerSec: %d", _format.nAvgBytesPerSec);
	flNotify::post(flNotify::L_INFO, "nBlockAlign: %d", _format.nBlockAlign);
	flNotify::post(flNotify::L_INFO, "nChannels: %d", _format.nChannels);
	flNotify::post(flNotify::L_INFO, "nSamplesPerSec: %d", _format.nSamplesPerSec);
	flNotify::post(flNotify::L_INFO, "wBitsPerSample: %d", _format.wBitsPerSample);
	flNotify::post(flNotify::L_INFO, "wFormatTag: %d", _format.wFormatTag);
*/
#endif
}
//------------------------------------------------------------------------------
const WAVEFORMATEX*
flPCMAudio::getFormat() const
{
	return &_format;
}
//------------------------------------------------------------------------------
flUShort
flPCMAudio::getChannels() const
{
	return _format.nChannels;
}
//------------------------------------------------------------------------------
flUShort
flPCMAudio::getBitsPerSample() const
{
	return _format.wBitsPerSample;
}
//------------------------------------------------------------------------------
flUInt
flPCMAudio::getSamplesPerSec() const
{
	return _format.nSamplesPerSec;
}
//------------------------------------------------------------------------------
void
flPCMAudio::setBufferCount(flUInt count)
{
	if (count == 0)
		return ;
	_buffers.setSize(count);
	resetBuffer();
}
//------------------------------------------------------------------------------
flUInt
flPCMAudio::getBufferCount() const
{
	return _buffers.getSize();
}
//------------------------------------------------------------------------------
void
flPCMAudio::resetBuffer()
{
	for(flUInt i = 0; i < _buffers.getSize(); i++)
	{
		_buffers[i].DataLength	= 0;
		_buffers[i].StartTime	= -1;
		_buffers[i].EndTime		= -1;
	}
	_bufferIndex = 0;
}
//------------------------------------------------------------------------------
void
flPCMAudio::setData(const flByte* data, flUInt dataLength, flInt64 timeStamp)
{
	flInt64 startTime = timeStamp;
	flInt64	endTime = timeStamp + flInt64(dataLength / flDouble(_format.nAvgBytesPerSec) * 10000000.0);

	_buffers[_bufferIndex].setData(data, dataLength);
	_buffers[_bufferIndex].StartTime = startTime;
	_buffers[_bufferIndex].EndTime = endTime;

	_bufferIndex++;
	if (_buffers.getSize() <= _bufferIndex)
		_bufferIndex = 0;
}
//------------------------------------------------------------------------------
void
flPCMAudio::getVolume(VolumeInfo& volume, flInt64 startTime, flInt64 endTime) const
{
	// Constants
	const flFloat	MINVOLUME	= 1.0f / VI_DYNAMICRANGE;

	// Initial Volume
	volume.left = VI_SILENTVOLUME;
	volume.right = VI_SILENTVOLUME;

	flUShort blockAlign		= _format.nBlockAlign;
	flUShort channels		= _format.nChannels;
	flUShort bitsPerSample	= _format.wBitsPerSample;
	flUInt samplesPerSec	= _format.nSamplesPerSec;

	if ((bitsPerSample != 8 && bitsPerSample != 16) ||
		(channels != 1 && channels != 2))
		return ;

	flFloat left = 0.0f;
	flFloat right = 0.0f;

	for(flUInt i = 0; i < _buffers.getSize(); i++)
	{
		const flPCMAudioBuffer& buffer = _buffers[i];

		if (buffer.StartTime == -1 || buffer.EndTime == -1 ||
			endTime <= buffer.StartTime || buffer.EndTime <= startTime)
			continue;

		flDouble lStartTimeSec = (startTime - buffer.StartTime) / 10000000.0;
		flDouble lEndTimeSec = (endTime - buffer.StartTime) / 10000000.0;

		flByte* ptr = buffer.Buffer + flInt(samplesPerSec * lStartTimeSec) * blockAlign;
		flByte* ptrEnd = buffer.Buffer + flInt(samplesPerSec * lEndTimeSec) * blockAlign;

		if (ptr < buffer.Buffer)
			ptr = buffer.Buffer;
		if (buffer.Buffer + buffer.DataLength < ptrEnd)
			ptrEnd = buffer.Buffer + buffer.DataLength;
	
		// Get Maximum Value
		if (bitsPerSample == 8)
		{
			for(; ptr < ptrEnd; ptr += blockAlign)
			{
				left = flMax(fabsf(flFloat(*ptr) - 128.0f) / 128.0f, left);
				if (channels == 2)
					right = flMax(fabsf(flFloat(*(ptr + 1)) - 128.0f) / 128.0f, right);
			}
		}
		else
		{
			for(; ptr < ptrEnd; ptr += blockAlign)
			{
				left = flMax(fabsf((flFloat)*((flShort *)ptr)) / 32767.0f, left);
				if (channels == 2)
					right = flMax(fabsf((flFloat)*((flShort *)ptr + 1)) / 32767.0f, right);
			}
		}
	}

	// Compute Volume (Unit 0.01dB)
	if (MINVOLUME < left)
		volume.left = flShort(VI_VOLUMEUNIT * 20.0f * log10f(left / MINVOLUME)) + VI_MINIMUMVOLUME;

	if (MINVOLUME < right)
		volume.right = flShort(VI_VOLUMEUNIT * 20.0f * log10f(right / MINVOLUME)) + VI_MINIMUMVOLUME;
}
//------------------------------------------------------------------------------
