/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "Stdafx.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <streams.h>

#include <flFilterCommon/flSourceStream.h>
#include <flFilterCommon/flSource.h>
#include <flBase/flSystemTime.h>
//------------------------------------------------------------------------------
CFLSourceStream::CFLSourceStream(TCHAR *pObjectName, HRESULT *phr, CFLSource *source, LPCWSTR pPinName)
: CBaseOutputPin(pObjectName, source, source->pStateLock(), phr, pPinName)
{
	_source = source;
	*phr = _source->AddPin(this);

	_absTime = new flSystemTime();
}
//------------------------------------------------------------------------------
CFLSourceStream::~CFLSourceStream(void)
{
	delete _absTime;

	_source->RemovePin(this);
}
//------------------------------------------------------------------------------
HRESULT
CFLSourceStream::Active(void)
{
	CAutoLock lock(_source->pStateLock());

	HRESULT hr;

	if (_source->IsActive())
		return S_FALSE;

	if (!IsConnected())
		return NOERROR;

	hr = CBaseOutputPin::Active();
	if (FAILED(hr))
		return hr;

	ASSERT(!ThreadExists());

	if (!Create())
		return E_FAIL;

	hr = Init();
	if (FAILED(hr))
		return hr;

	return Pause();
}
//------------------------------------------------------------------------------
HRESULT
CFLSourceStream::Inactive(void)
{
	CAutoLock lock(_source->pStateLock());

	HRESULT hr;

	if (!IsConnected())
		return NOERROR;

	hr = CBaseOutputPin::Inactive();
	if (FAILED(hr))
		return hr;

	if (ThreadExists())
	{
		hr = Stop();

		if (FAILED(hr))
			return hr;

		hr = Exit();
		if (FAILED(hr))
			return hr;

		Close();
	}

	// hr = CBaseOutputPin::Inactive();  // call this first to Decommit the allocator
	//if (FAILED(hr)) {
	//	return hr;
	//}

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CFLSourceStream::ThreadProc(void)
{
	HRESULT hr;
	Command com;

	do {
		com = GetRequest();
		if (com != CMD_INIT)
		{
			DbgLog((LOG_ERROR, 1, TEXT("Thread expected init command")));
			Reply((DWORD) E_UNEXPECTED);
		}
	} while (com != CMD_INIT);

	DbgLog((LOG_TRACE, 1, TEXT("CFLSourceStream worker thread initializing")));

	hr = OnThreadCreate();
	if (FAILED(hr))
	{
		DbgLog((LOG_ERROR, 1, TEXT("CFLSourceStream::OnThreadCreate failed. Aborting thread.")));
		OnThreadDestroy();
		Reply(hr);
		return 1;
	}

	Reply(NOERROR);

	Command cmd;
	do {
		cmd = GetRequest();

		switch (cmd)
		{
		case CMD_EXIT:
			Reply(NOERROR);
			break;

		case CMD_RUN:
			DbgLog((LOG_ERROR, 1, TEXT("CMD_RUN received before a CMD_PAUSE???")));

		case CMD_PAUSE:
			Reply(NOERROR);
			DoBufferProcessingLoop();
			break;

		case CMD_STOP:
			Reply(NOERROR);
			break;

		default:
			DbgLog((LOG_ERROR, 1, TEXT("Unknown command %d received!"), cmd));
			Reply((DWORD) E_NOTIMPL);
			break;
		}
	} while (cmd != CMD_EXIT);

	hr = OnThreadDestroy();
	if (FAILED(hr))
	{
		DbgLog((LOG_ERROR, 1, TEXT("CFLSourceStream::OnThreadDestroy failed. Exiting thread.")));
		return 1;
	}

	DbgLog((LOG_TRACE, 1, TEXT("CFLSourceStream worker thread exiting")));
	return 0;
}
//------------------------------------------------------------------------------
HRESULT
CFLSourceStream::DoBufferProcessingLoop(void)
{
	Command com;

	OnThreadStartPlay();

	do {
		while (!CheckRequest(&com))
		{
			HRESULT hr;
			IMediaSample *pSample;

			hr = GetDeliveryBuffer(&pSample,NULL,NULL,0);
			if (FAILED(hr))
			{
				Sleep(1);
				continue;
			}

			// o^C~O̒p̎v
			_absTime->update();
			flInt64 deliverBegin = _absTime->getTime();

			hr = FillBuffer(pSample);
			if (hr == S_OK)
			{
				hr = Deliver(pSample);
				pSample->Release();

				if(hr != S_OK)
				{
					DbgLog((LOG_TRACE, 2, TEXT("Deliver() returned %08x; stopping"), hr));
					return S_OK;
				}

				// o^C~O̒
				_absTime->update();
				AdjustRate(_absTime->getTime() - deliverBegin);
			}
			else
			{
				pSample->Release();
			}
		}

		if (com == CMD_RUN || com == CMD_PAUSE)
		{
			Reply(NOERROR);
		}
		else if (com != CMD_STOP)
		{
			Reply((DWORD) E_UNEXPECTED);
			DbgLog((LOG_ERROR, 1, TEXT("Unexpected command!!!")));
		}

	} while (com != CMD_STOP);

	return S_FALSE;
}
//------------------------------------------------------------------------------
HRESULT
CFLSourceStream::CheckMediaType(const CMediaType *pMediaType)
{
	CAutoLock lock(_source->pStateLock());

	CMediaType mt;
	GetMediaType(&mt);

	if (mt == *pMediaType)
		return NOERROR;

	return E_FAIL;
}
//------------------------------------------------------------------------------
HRESULT
CFLSourceStream::GetMediaType(int iPosition, CMediaType *pMediaType)
{
	CAutoLock lock(_source->pStateLock());

	if (iPosition<0)
		return E_INVALIDARG;

	if (iPosition>0)
		return VFW_S_NO_MORE_ITEMS;

	return GetMediaType(pMediaType);
}
//------------------------------------------------------------------------------
STDMETHODIMP
CFLSourceStream::QueryId(LPWSTR *Id)
{
	CheckPointer(Id,E_POINTER);
	ValidateReadWritePtr(Id,sizeof(LPWSTR));

	flInt i = 1+ _source->FindPinNumber(this);
	if (i < 1)
		return VFW_E_NOT_FOUND;
	*Id = (LPWSTR)CoTaskMemAlloc(8);
	if (*Id==NULL)
		return E_OUTOFMEMORY;

	IntToWstr(i, *Id);
	return NOERROR;
}
//------------------------------------------------------------------------------
