/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
#include <windows.h>
#include <streams.h>
#include <initguid.h>
#include <olectl.h>
#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif

#include "transfrmn.h"

#include <flTypes/flUIDs.h>
#include <flTypes/IFormatTrans.h>

#include "formattransprop.h"
#include "formattrans.h"

#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue

// setup data

const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
    &MEDIATYPE_Video,       // Major type
    &MEDIASUBTYPE_NULL      // Minor type
};

const AMOVIESETUP_PIN psudPins[] =
{
    {
        L"Input",           // String pin name
        FALSE,              // Is it rendered
        FALSE,              // Is it an output
        FALSE,              // Allowed none
        FALSE,              // Allowed many
        &CLSID_NULL,        // Connects to filter
        L"Output",          // Connects to pin
        1,                  // Number of types
        &sudPinTypes },     // The pin details
      { L"Output",          // String pin name
        FALSE,              // Is it rendered
        TRUE,               // Is it an output
        FALSE,              // Allowed none
        FALSE,              // Allowed many
        &CLSID_NULL,        // Connects to filter
        L"Input",           // Connects to pin
        1,                  // Number of types
        &sudPinTypes        // The pin details
    }
};


const AMOVIESETUP_FILTER sudFormatTrans =
{
    &CLSID_FormatTrans,        // Filter CLSID
    L"NIME Video Format Transformer",      // Filter name
    MERIT_DO_NOT_USE,       // Its merit
    2,                      // Number of pins
    psudPins                // Pin details
};


// List of class IDs and creator functions for the class factory. This
// provides the link between the OLE entry point in the DLL and an object
// being created. The class factory will call the static CreateInstance

CFactoryTemplate g_Templates[2] = {

    { L"FormatTrans"
    , &CLSID_FormatTrans
    , CFormatTrans::CreateInstance
    , NULL
    , &sudFormatTrans }
  ,
    { L"FormatTransProp"
    , &CLSID_FormatTransPropertyPage
    , CFormatTransProperties::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);



#define DEF_INPUT_VIDEORESOLUTION	VR_AUTO
#define DEF_OUTPUT_VIDEORESOLUTION	VR_SQCIF
#define DEF_INPUT_VIDEOCOLOR		VC_AUTO

//
// Constructor
//
CFormatTrans::CFormatTrans(TCHAR *tszName,LPUNKNOWN punk,HRESULT *phr) :
CTransformNFilter(tszName, punk, CLSID_FormatTrans),
CPersistStream(punk, phr),
m_lBufferRequest(1),
_inVR(DEF_INPUT_VIDEORESOLUTION),
_outVR(DEF_OUTPUT_VIDEORESOLUTION),
_inVC(DEF_INPUT_VIDEOCOLOR)
{
    ASSERT(tszName);
    ASSERT(phr);

} // FormatTrans


//
// CreateInstance
//
// Provide the way for COM to create a CFormatTrans object
//
CUnknown * WINAPI CFormatTrans::CreateInstance(LPUNKNOWN punk, HRESULT *phr) {

    CFormatTrans *pNewObject = new CFormatTrans(NAME("FormatTrans"), punk, phr);
    if (pNewObject == NULL) {
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;

} // CreateInstance


//
// NonDelegatingQueryInterface
//
// Reveals IFormatTrans and ISpecifyPropertyPages
//
STDMETHODIMP CFormatTrans::NonDelegatingQueryInterface(REFIID riid, void **ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IFormatTrans) {
        return GetInterface((IFormatTrans *) this, ppv);
	} else if (riid == IID_IPersistStream) {
		return GetInterface((IPersistStream *) this, ppv);
    } else if (riid == IID_ISpecifyPropertyPages) {
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    } else {
        return CTransformNFilter::NonDelegatingQueryInterface(riid, ppv);
    }

} // NonDelegatingQueryInterface

//
// Transform
//
// Copy the input sample into the output sample
// Then transform the output sample 'in place'
//
HRESULT CFormatTrans::Transform(IMediaSample *pIn, IMediaSample *pOut)
{
    CAutoLock cAutoLock(&m_FormatTransLock);

	HRESULT hr = ScalingCopy(pIn, pOut);
    if (FAILED(hr)) {
        return hr;
    }
	return NOERROR;

} // Transform



HRESULT
CFormatTrans::ScalingCopy(IMediaSample *pSource, IMediaSample *pDest) const
{
    // Copy the sample data
    BYTE *pSourceBuffer, *pDestBuffer;
	BYTE *pSB, *pDB, *pSBt;
    long lSourceSize = pSource->GetActualDataLength();
    long lDestSize = pDest->GetActualDataLength();

    pSource->GetPointer(&pSourceBuffer);
    pDest->GetPointer(&pDestBuffer);

	pSB = pSourceBuffer;
	pDB = pDestBuffer;

	/////////// here

	CMediaType imt = m_pInput->CurrentMediaType();
	CMediaType omt = m_pOutput->CurrentMediaType();

	VIDEOINFO* ivinfo = (VIDEOINFO *)imt.Format();
	VIDEOINFO* ovinfo = (VIDEOINFO *)omt.Format();


	int i, j;
	long iw = ivinfo->bmiHeader.biWidth;
	long ih = ivinfo->bmiHeader.biHeight;
	long ow = ovinfo->bmiHeader.biWidth;
	long oh = ovinfo->bmiHeader.biHeight;

	int bCnt = ivinfo->bmiHeader.biBitCount / 8;
	long bcOW = bCnt * ow;
	long bcIW = bCnt * iw;

	float fx, fy;
	float fi, fj;

	fx = (float)iw / (float)ow;
	fy = (float)ih / (float)oh;

	if (iw == ow && ih == oh)
	{
		CopyMemory( (PVOID) pDestBuffer,(PVOID) pSourceBuffer,
				(lDestSize <= lSourceSize) ? lDestSize : lSourceSize);
	} else {

		fi = 0.0f;

		switch(bCnt)
		{
		case 1:
			for(i = 0; i < oh; i++)
			{
				pSBt = pSB;
				fj = 0.0f;

				for(j = 0; j < ow; j++)
				{
					*pDB = *pSBt;

					if (fj >= 1.0f)
					{
						pSBt += (int)fj;
						fj -= (int)fj;
					}

					fj += fx;
					pDB ++;

				}

				fi += fy;
				pSB = pSourceBuffer + (int)(fi) * bcIW;
			}
			break;
		case 2:
			for(i = 0; i < oh; i++)
			{
				pSBt = pSB;
				fj = 0.0f;

				for(j = 0; j < ow; j++)
				{
					CopyMemory( (PVOID) pDB,(PVOID) pSBt, 2);

					if (fj >= 1.0f)
					{
						pSBt += ((int)fj * 2);
						fj -= (int)fj;
					}

					fj += fx;
					pDB += 2;

				}

				fi += fy;
				pSB = pSourceBuffer + (int)(fi) * bcIW;
			}
			break;
		case 3:
			for(i = 0; i < oh; i++)
			{
				pSBt = pSB;
				fj = 0.0f;

				for(j = 0; j < ow; j++)
				{
					CopyMemory( (PVOID) pDB,(PVOID) pSBt, 3);

					if (fj >= 1.0f)
					{
						pSBt += ((int)fj * 3);
						fj -= (int)fj;
					}

					fj += fx;
					pDB += 3;

				}

				fi += fy;
				pSB = pSourceBuffer + (int)(fi) * bcIW;
			}
			break;
		}
	}



    // Copy the sample times
    REFERENCE_TIME TimeStart, TimeEnd;
    if (NOERROR == pSource->GetTime(&TimeStart, &TimeEnd)) {
        pDest->SetTime(&TimeStart, &TimeEnd);
    }

    LONGLONG MediaStart, MediaEnd;
    if (pSource->GetMediaTime(&MediaStart,&MediaEnd) == NOERROR) {
        pDest->SetMediaTime(&MediaStart,&MediaEnd);
    }

    // Copy the Sync point property
    HRESULT hr = pSource->IsSyncPoint();
    if (hr == S_OK) {
        pDest->SetSyncPoint(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetSyncPoint(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

	// Copy the media type
	/*
    AM_MEDIA_TYPE *pMediaType;
    pSource->GetMediaType(&pMediaType);
    pDest->SetMediaType(pMediaType);
    DeleteMediaType(pMediaType);
	*/

    // Copy the preroll property
    hr = pSource->IsPreroll();
    if (hr == S_OK) {
        pDest->SetPreroll(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetPreroll(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

    // Copy the discontinuity property
    hr = pSource->IsDiscontinuity();
    if (hr == S_OK) {
	pDest->SetDiscontinuity(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetDiscontinuity(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

    // Copy the actual data length
	/*
    long lDataLength = pSource->GetActualDataLength();
    pDest->SetActualDataLength(lDataLength);
	*/

	long actLen = pDest->GetActualDataLength();

    return NOERROR;

}


//
// CheckInputType
//
// Check the input type is OK, return an error otherwise
//
HRESULT CFormatTrans::CheckInputType(const CMediaType *mtIn)
{
	HRESULT hr = E_INVALIDARG;

	if (*mtIn->FormatType() != FORMAT_VideoInfo)
		return hr;

	if (_inVC == VC_AUTO &&
		!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB8) &&
		!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB555) &&
		!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB565) &&
		!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB24))
		return hr;
	else if (_inVC == VC_RGB8 && !IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB8))
		return hr;
	else if (_inVC == VC_RGB555 && !IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB555))
		return hr;
	else if (_inVC == VC_RGB565 && !IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB565))
		return hr;
	else if (_inVC == VC_RGB24 && !IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB24))
		return hr;


	VIDEOINFO *pvi = (VIDEOINFO *)mtIn->Format();
	long bWidth = pvi->bmiHeader.biWidth;
	long bHeight = pvi->bmiHeader.biHeight;

	switch(_inVR)
	{
	case VR_SQCIF:
		if (bWidth == 128 && bHeight == 96)
			hr = NOERROR;
		break;
	case VR_QCIF:
		if (bWidth == 176 && bHeight == 144)
			hr = NOERROR;
		break;
	case VR_CIF:
		if (bWidth == 352 && bHeight == 288)
			hr = NOERROR;
		break;
	case VR_160x120:
		if (bWidth == 160 && bHeight == 120)
			hr = NOERROR;
		break;
	case VR_320x240:
		if (bWidth == 320 && bHeight == 240)
			hr = NOERROR;
		break;
	case VR_360x240:
		if (bWidth == 360 && bHeight == 240)
			hr = NOERROR;
		break;
	case VR_640x480:
		if (bWidth == 640 && bHeight == 480)
			hr = NOERROR;
		break;
	case VR_720x480:
		if (bWidth == 720 && bHeight == 480)
			hr = NOERROR;
		break;
	case VR_800x600:
		if (bWidth == 800 && bHeight == 600)
			hr = NOERROR;
		break;
	case VR_1024x768:
		if (bWidth == 1024 && bHeight == 768)
			hr = NOERROR;
		break;
	case VR_1280x1024:
		if (bWidth == 1280 && bHeight == 1024)
			hr = NOERROR;
		break;
	case VR_1600x1200:
		if (bWidth == 1600 && bHeight == 1200)
			hr = NOERROR;
		break;
	case VR_HDTV720P:
		if (bWidth == 1280 && bHeight == 720)
			hr = NOERROR;
		break;
	case VR_HDTV1080I:
		if (bWidth == 1920 && bHeight == 1080)
			hr = NOERROR;
		break;
	case VR_SDTV480P:
		if (bWidth == 852 && bHeight == 480)
			hr = NOERROR;
		break;
	case VR_SDTV480I:
		if (bWidth == 852 && bHeight == 480)
			hr = NOERROR;
		break;
	case VR_AUTO:
		hr = NOERROR;
	}

	return hr;

} // CheckInputType

HRESULT CFormatTrans::CheckOutputType(const CMediaType *mtOut)
{
	if (*mtOut->FormatType() != FORMAT_VideoInfo){
		return E_INVALIDARG;
	}

	if (m_pInput->IsConnected())
	{
		CMediaType cmt = m_pInput->CurrentMediaType();
		if (!IsEqualGUID(*cmt.Subtype(), *mtOut->Subtype()))
			return E_INVALIDARG;
	}

	VIDEOINFO* pvi = (VIDEOINFO *)mtOut->Format();
	if (pvi->rcSource.right != pvi->bmiHeader.biWidth ||
		pvi->rcSource.bottom != pvi->bmiHeader.biHeight)
		return E_INVALIDARG;

	return NOERROR;
}

//
// DecideBufferSize
//
// Tell the output pin's allocator what size buffers we
// require. Can only do this when the input is connected
//
HRESULT CFormatTrans::DecideBufferSize(IMemAllocator *pAlloc,ALLOCATOR_PROPERTIES *pProp)
{
	if (m_pInput->IsConnected() == FALSE)
		return E_UNEXPECTED;

	ASSERT(pAlloc);
	ASSERT(pProp);
	ASSERT(m_pInput->CurrentMediaType().bFixedSizeSamples);

	VIDEOINFOHEADER *pvi = (VIDEOINFOHEADER *)m_pOutput->CurrentMediaType().pbFormat;
	pvi->bmiHeader.biSizeImage = GetBitmapSize(&pvi->bmiHeader);
	ASSERT(pvi->bmiHeader.biSizeImage != 0);

	pProp->cBuffers = 1;
	pProp->cbBuffer = pvi->bmiHeader.biSizeImage;

	ALLOCATOR_PROPERTIES Actual;
	HRESULT hr = pAlloc->SetProperties(pProp, &Actual);
	if (FAILED(hr))
		return hr;

	ASSERT(Actual.cBuffers == 1);

	if (Actual.cBuffers < pProp->cBuffers ||
		Actual.cbBuffer < pProp->cbBuffer)
		return E_FAIL;

    return NOERROR;
}


HRESULT
CFormatTrans::SetMediaType(PIN_DIRECTION direction,const CMediaType *pmt)
{
    // Reconnect where necessary.
    if( m_pInput->IsConnected() && m_pOutput->IsConnected() ){
        FILTER_INFO fInfo;

        QueryFilterInfo( &fInfo );

        if (direction == PINDIR_INPUT && *pmt != m_pOutput->CurrentMediaType() )
            fInfo.pGraph->Reconnect( m_pOutput );

        QueryFilterInfoReleaseGraph( fInfo );

    }

	return NOERROR;
}


//
// GetMediaType
//
// I support one type, namely the type of the input pin
// We must be connected to support the single output type
//
HRESULT CFormatTrans::GetMediaType(int iPosition, CMediaType *pMediaType)
{
    CAutoLock cAutoLock(&m_FormatTransLock);
    if (iPosition < 0) {
        return E_INVALIDARG;
    }
    if (iPosition > 0) {
        return VFW_S_NO_MORE_ITEMS;
    }
	
	ASSERT(m_pInput != NULL);
	if (m_pInput->IsConnected() == FALSE)
        return E_UNEXPECTED;
	CMediaType iMediaType = m_pInput->CurrentMediaType();

	*pMediaType = iMediaType;
	
	VIDEOINFO *pvi = (VIDEOINFO *)pMediaType->AllocFormatBuffer(sizeof(VIDEOINFO));
    if (pvi == NULL) {
		return(E_OUTOFMEMORY);
    }
    ZeroMemory(pvi, sizeof(VIDEOINFO));

	VIDEOINFO *pivi = (VIDEOINFO *)iMediaType.Format();

	pvi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	pvi->bmiHeader.biBitCount = pivi->bmiHeader.biBitCount;
	pvi->bmiHeader.biXPelsPerMeter = pivi->bmiHeader.biXPelsPerMeter;
	pvi->bmiHeader.biYPelsPerMeter = pivi->bmiHeader.biYPelsPerMeter;
	pvi->bmiHeader.biClrImportant = pivi->bmiHeader.biClrImportant;
	pvi->bmiHeader.biClrUsed = pivi->bmiHeader.biClrUsed;
	for(DWORD i = 0; i < pvi->bmiHeader.biClrUsed; i++)
		pvi->bmiColors[i] = pivi->bmiColors[i];

	switch(_outVR){
	case VR_SQCIF:
		pvi->bmiHeader.biWidth		= 128;
		pvi->bmiHeader.biHeight		= 96;
		break;
	case VR_QCIF:
		pvi->bmiHeader.biWidth		= 176;
		pvi->bmiHeader.biHeight		= 144;
		break;
	case VR_CIF:
		pvi->bmiHeader.biWidth		= 352;
		pvi->bmiHeader.biHeight		= 288;
		break;
	case VR_160x120:
		pvi->bmiHeader.biWidth		= 160;
		pvi->bmiHeader.biHeight		= 120;
		break;
	case VR_320x240:
		pvi->bmiHeader.biWidth		= 320;
		pvi->bmiHeader.biHeight		= 240;
		break;
	case VR_360x240:
		pvi->bmiHeader.biWidth		= 360;
		pvi->bmiHeader.biHeight		= 240;
		break;
	case VR_640x480:
		pvi->bmiHeader.biWidth		= 640;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_720x480:
		pvi->bmiHeader.biWidth		= 720;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_800x600:
		pvi->bmiHeader.biWidth		= 800;
		pvi->bmiHeader.biHeight		= 600;
		break;
	case VR_1024x768:
		pvi->bmiHeader.biWidth		= 1024;
		pvi->bmiHeader.biHeight		= 768;
		break;
	case VR_1280x1024:
		pvi->bmiHeader.biWidth		= 1280;
		pvi->bmiHeader.biHeight		= 1024;
		break;
	case VR_1600x1200:
		pvi->bmiHeader.biWidth		= 1600;
		pvi->bmiHeader.biHeight		= 1200;
		break;
	case VR_HDTV720P:
		pvi->bmiHeader.biWidth		= 1280;
		pvi->bmiHeader.biHeight		= 720;
		break;
	case VR_HDTV1080I:
		pvi->bmiHeader.biWidth		= 1920;
		pvi->bmiHeader.biHeight		= 1080;
		break;
	case VR_SDTV480P:
		pvi->bmiHeader.biWidth		= 852;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_SDTV480I:
		pvi->bmiHeader.biWidth		= 852;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_THROUGH:
		pvi->bmiHeader.biWidth		= pivi->bmiHeader.biWidth;
		pvi->bmiHeader.biHeight		= pivi->bmiHeader.biHeight;
		break;
	}

	pMediaType->SetTemporalCompression(FALSE);
    pvi->bmiHeader.biPlanes		= 1;
	pvi->bmiHeader.biSizeImage	= GetBitmapSize(&pvi->bmiHeader);

	SetRectEmpty(&(pvi->rcSource));	// we want the whole image area rendered.
    SetRectEmpty(&(pvi->rcTarget));	// no particular destination rectangle
	pvi->rcSource.right = pvi->bmiHeader.biWidth;
	pvi->rcSource.bottom = pvi->bmiHeader.biHeight;

	const GUID SubTypeGUID = GetBitmapSubtype(&pvi->bmiHeader);
	pMediaType->SetSampleSize(pvi->bmiHeader.biSizeImage);
	pMediaType->SetType(&MEDIATYPE_Video);
	pMediaType->SetFormatType(&FORMAT_VideoInfo);


	return NOERROR;

} // GetMediaType


STDMETHODIMP
CFormatTrans::get_InputVideoResolution(unsigned int *videoResolution)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	*videoResolution = (unsigned int)_inVR;

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::put_InputVideoResolution(unsigned int videoResolution)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	if (_inVR != videoResolution)
	{
		_inVR = (VideoResolutionEnum)videoResolution;
		SetDirty(TRUE);

		ReconnectPin((IPin *)m_pInput, NULL);
	}

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::put_DefaultInputVideoResolution(void)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	if (_inVR != DEF_INPUT_VIDEORESOLUTION)
	{
		_inVR = DEF_INPUT_VIDEORESOLUTION;
		SetDirty(TRUE);

		ReconnectPin((IPin *)m_pInput, NULL);
	}

	return NOERROR;
}


STDMETHODIMP
CFormatTrans::get_InputVideoColor(unsigned int *videoColor)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	*videoColor = (unsigned int)_inVC;

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::put_InputVideoColor(unsigned int videoColor)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	if (_inVC != videoColor)
	{
		_inVC = (VideoColorEnum)videoColor;
		SetDirty(TRUE);

		ReconnectPin((IPin *)m_pInput, NULL);
		ReconnectPin((IPin *)m_pOutput, NULL);
	}

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::put_DefaultInputVideoColor(void)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	if (_inVC != DEF_INPUT_VIDEOCOLOR)
	{
		_inVC = DEF_INPUT_VIDEOCOLOR;
		SetDirty(TRUE);

		ReconnectPin((IPin *)m_pInput, NULL);
		ReconnectPin((IPin *)m_pOutput, NULL);
	}

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::get_OutputVideoResolution(unsigned int *videoResolution)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	*videoResolution = (unsigned int)_outVR;

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::put_OutputVideoResolution(unsigned int videoResolution)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	if (_outVR != videoResolution)
	{
		_outVR = (VideoResolutionEnum)videoResolution;
		SetDirty(TRUE);

		ReconnectPin((IPin *)m_pOutput, NULL);
	}

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::put_DefaultOutputVideoResolution(void)
{
	CAutoLock cAutoLock(&m_FormatTransLock);
	if (_outVR != DEF_OUTPUT_VIDEORESOLUTION)
	{
		_outVR = DEF_OUTPUT_VIDEORESOLUTION;
		SetDirty(TRUE);

		ReconnectPin((IPin *)m_pOutput, NULL);
	}

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::get_ActualInputVideoResolution(unsigned int *videoResolution)
{
	CAutoLock cAutoLock(&m_FormatTransLock);

	*videoResolution = VR_UNRECOGNIZED;
	if (m_pInput == NULL)
		return NOERROR;

	CMediaType& mediaType = m_pInput->CurrentMediaType();
	
//	ASSERT(*(mediaType.FormatType()) == FORMAT_VideoInfo);
	if (*(mediaType.FormatType()) != FORMAT_VideoInfo)
		return NOERROR;

	VIDEOINFO *vinfo = (VIDEOINFO *)mediaType.Format();

	long bWidth = vinfo->bmiHeader.biWidth;
	long bHeight = vinfo->bmiHeader.biHeight;

	if (bWidth == 128 && bHeight == 96)
		*videoResolution = VR_SQCIF;
	else if (bWidth == 176 && bHeight == 144)
		*videoResolution = VR_QCIF;
	else if (bWidth == 352 && bHeight == 288)
		*videoResolution = VR_CIF;
	else if (bWidth == 160 && bHeight == 120)
		*videoResolution = VR_160x120;
	else if (bWidth == 320 && bHeight == 240)
		*videoResolution = VR_320x240;
	else if (bWidth == 360 && bHeight == 240)
		*videoResolution = VR_360x240;
	else if (bWidth == 640 && bHeight == 480)
		*videoResolution = VR_640x480;
	else if (bWidth == 720 && bHeight == 480)
		*videoResolution = VR_720x480;
	else if (bWidth == 800 && bHeight == 600)
		*videoResolution = VR_800x600;
	else if (bWidth == 1024 && bHeight == 768)
		*videoResolution = VR_1024x768;
	else if (bWidth == 1280 && bHeight == 1024)
		*videoResolution = VR_1280x1024;
	else if (bWidth == 1600 && bHeight == 1200)
		*videoResolution = VR_1600x1200;
	else if (bWidth == 1280 && bHeight == 720)
		*videoResolution = VR_HDTV720P;
	else if (bWidth == 1920 && bHeight == 1080)
		*videoResolution = VR_HDTV1080I;
	else if (bWidth == 852 && bHeight == 480)
		*videoResolution = VR_SDTV480P;
	else if (bWidth == 852 && bHeight == 480)
		*videoResolution = VR_SDTV480I;
	else
		*videoResolution = VR_UNRECOGNIZED;

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::get_ActualInputVideoColor(unsigned int *videoColor)
{
	CAutoLock cAutoLock(&m_FormatTransLock);

	*videoColor = VC_UNRECOGNIZED;
	if (m_pInput == NULL)
		return NOERROR;

	CMediaType& mediaType = m_pInput->CurrentMediaType();

//	ASSERT(*(mediaType.FormatType()) == FORMAT_VideoInfo);
	if (*(mediaType.FormatType()) != FORMAT_VideoInfo)
		return NOERROR;

	if (IsEqualGUID(*mediaType.Subtype(), MEDIASUBTYPE_RGB8))
		*videoColor = VC_RGB8;
	else if (IsEqualGUID(*mediaType.Subtype(), MEDIASUBTYPE_RGB555))
		*videoColor = VC_RGB555;
	else if (IsEqualGUID(*mediaType.Subtype(), MEDIASUBTYPE_RGB565))
		*videoColor = VC_RGB565;
	else if (IsEqualGUID(*mediaType.Subtype(), MEDIASUBTYPE_RGB24))
		*videoColor = VC_RGB24;
	else
		*videoColor = VC_UNRECOGNIZED;

	return NOERROR;
}

STDMETHODIMP
CFormatTrans::get_ActualOutputVideoResolution(unsigned int *videoResolution)
{
	CAutoLock cAutoLock(&m_FormatTransLock);

	*videoResolution = (unsigned int)_outVR;
	if (*videoResolution != VR_THROUGH)
		return NOERROR;

	*videoResolution = VR_UNRECOGNIZED;
	if (m_pInput == NULL)
		return NOERROR;

	CMediaType& mediaType = m_pInput->CurrentMediaType();
	
	//ASSERT(*(mediaType.FormatType()) == FORMAT_VideoInfo);
	if (*(mediaType.FormatType()) != FORMAT_VideoInfo)
		return NOERROR;

	VIDEOINFO *vinfo = (VIDEOINFO *)mediaType.Format();

	long bWidth = vinfo->bmiHeader.biWidth;
	long bHeight = vinfo->bmiHeader.biHeight;

	if (bWidth == 128 && bHeight == 96)
		*videoResolution = VR_SQCIF;
	else if (bWidth == 176 && bHeight == 144)
		*videoResolution = VR_QCIF;
	else if (bWidth == 352 && bHeight == 288)
		*videoResolution = VR_CIF;
	else if (bWidth == 160 && bHeight == 120)
		*videoResolution = VR_160x120;
	else if (bWidth == 320 && bHeight == 240)
		*videoResolution = VR_320x240;
	else if (bWidth == 360 && bHeight == 240)
		*videoResolution = VR_360x240;
	else if (bWidth == 640 && bHeight == 480)
		*videoResolution = VR_640x480;
	else if (bWidth == 720 && bHeight == 480)
		*videoResolution = VR_720x480;
	else if (bWidth == 800 && bHeight == 600)
		*videoResolution = VR_800x600;
	else if (bWidth == 1024 && bHeight == 768)
		*videoResolution = VR_1024x768;
	else if (bWidth == 1280 && bHeight == 1024)
		*videoResolution = VR_1280x1024;
	else if (bWidth == 1600 && bHeight == 1200)
		*videoResolution = VR_1600x1200;
	else if (bWidth == 1280 && bHeight == 720)
		*videoResolution = VR_HDTV720P;
	else if (bWidth == 1920 && bHeight == 1080)
		*videoResolution = VR_HDTV1080I;
	else if (bWidth == 852 && bHeight == 480)
		*videoResolution = VR_SDTV480P;
	else if (bWidth == 852 && bHeight == 480)
		*videoResolution = VR_SDTV480I;
	else
		*videoResolution = VR_UNRECOGNIZED;

	return NOERROR;
}



#define WRITEOUT(var)  hr = pStream->Write(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define WRITENOUT(var,n)  hr = pStream->Write(var, n, NULL); \
if (FAILED(hr)) return hr;
#define READIN(var)    hr = pStream->Read(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define READNIN(var,n)    hr = pStream->Read(var, n, NULL); \
if (FAILED(hr)) return hr;


// IPersitStream
STDMETHODIMP
CFormatTrans::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_FormatTrans;
	return S_OK;
} // GetClassID

HRESULT
CFormatTrans::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	WRITEOUT(_inVR);
	WRITEOUT(_outVR);
	WRITEOUT(_inVC);
	return NOERROR;
	
} // WriteToStream

HRESULT
CFormatTrans::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	READIN(_inVR);
	READIN(_outVR);
	READIN(_inVC);
	return NOERROR;
	
} // ReadFromStream

DWORD
CFormatTrans::GetSoftwareVersion(void)
{
	return 1;
} // GetSoftwareVersion

int
CFormatTrans::SizeMax()
{
	return sizeof (unsigned int) * 3;
} // SizeMax


//
// GetPages
//
// This is the sole member of ISpecifyPropertyPages
// Returns the clsid's of the property pages we support
//
STDMETHODIMP CFormatTrans::GetPages(CAUUID *pPages)
{
    pPages->cElems = 1;
    pPages->pElems = (GUID *) CoTaskMemAlloc(sizeof(GUID));
    if (pPages->pElems == NULL) {
        return E_OUTOFMEMORY;
    }
    *(pPages->pElems) = CLSID_FormatTransPropertyPage;
    return NOERROR;

} // GetPages


//////////////////////////////////////////////////////////////////////////////



//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2( TRUE );

} // DllRegisterServer

//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2( FALSE );

} // DllUnregisterServer

