/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flMulticastReceiver/MulticastReceiver.h>
#include <flMulticastReceiver/MulticastReceiverComProp.h>
#include <flMulticastReceiver/MulticastReceiverMediaProp.h>
#include <flMulticastReceiver/MulticastReceiverFrameProp.h>
#include <flMulticastReceiver/MulticastReceiverRejectProp.h>
#include <flMulticastReceiver/MulticastReceiverAddInfoProp.h>
#include <flMulticastReceiver/MulticastReceiverStream.h>
#include <flFilterCommon/flFilterCommon.h>
#include <flNetwork/flFMTReceiver.h>
#include <flNetwork/flNetwork.h>
#ifdef _DEBUG
#include <flBase/flNotify.h>
#endif
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
#define CHANGEMEDIATYPE_THRESHOLD		5
#define SAMPLE_BUFFERSIZE				4
//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudOpPinTypes =
{
    &MEDIATYPE_NULL,			// MajorType
    &MEDIASUBTYPE_NULL			// MinorType
};
//------------------------------------------------------------------------------
const AMOVIESETUP_PIN sudOpPin =
{
	L"Out",						// The Pins name
	FALSE,						// Is rendered
	TRUE,						// Is an output pin
	FALSE,						// Allowed none
	FALSE,						// Allowed many
	&CLSID_NULL,				// Connects to filter
	NULL,						// Connects to pin
	1,							// Number of types
	&sudOpPinTypes				// Pin details
};
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudMulticastReceiver =
{
    &CLSID_MulticastReceiver,	// Filter CLSID
    L"NIME Multicast Receiver",	// String name
    MERIT_DO_NOT_USE,			// Filter merit
    1,							// Number of pins
    &sudOpPin					// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[6] = {
    { L"MulticastReceiver"
    , &CLSID_MulticastReceiver
    , CMulticastReceiver::CreateInstance
    , NULL
    , &sudMulticastReceiver }
  ,
    { L"MulticastReceiverComProp"
    , &CLSID_MulticastReceiverComPropertyPage
    , CMulticastReceiverComProp::CreateInstance }
  ,
    { L"MulticastReceiverMediaProp"
    , &CLSID_MulticastReceiverMediaTypePropertyPage
    , CMulticastReceiverMediaProp::CreateInstance }
  ,
    { L"MulticastReceiverFrameProp"
    , &CLSID_MulticastReceiverFrameControlPropertyPage
    , CMulticastReceiverFrameProp::CreateInstance }
  ,
    { L"MulticastReceiverRejectProp"
    , &CLSID_MulticastReceiverRejectionAddressPropertyPage
    , CMulticastReceiverRejectProp::CreateInstance }
  ,
    { L"MulticastReceiverAddInfoProp"
    , &CLSID_MulticastReceiverAdditionalInfoPropertyPage
    , CMulticastReceiverAddInfoProp::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CMulticastReceiver::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
    CMulticastReceiver *pNewObject = new CMulticastReceiver(NAME("MulticastReceiver"), punk, phr);
    if (pNewObject == NULL)
	{
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CMulticastReceiver::CMulticastReceiver(TCHAR *tszName, LPUNKNOWN pUnk, HRESULT *phr) :
CFLSource(tszName, pUnk, CLSID_MulticastReceiver, phr),
CPersistStream(pUnk, phr)
{
	ASSERT(tszName != NULL);
	ASSERT(phr != NULL);

	CAutoLock cAutoLock(&_stateLock);

	_streams = new CFLSourceStream*[1];
	if (_streams == NULL)
	{
		*phr = E_OUTOFMEMORY;
		return ;
	}

	_streams[0] = new CMulticastReceiverStream(phr, this, L"Out");
	if (_streams[0] == NULL)
	{
		*phr = E_OUTOFMEMORY;
		return;
	}
	_multicastReceiverStream = (CMulticastReceiverStream *)_streams[0];

	// Media Samples
	_samplePool						= new flMediaSamplePool();

	// Received Media Type
	_receivedMediaType.InitMediaType();
	
	// Other variables
	_isRunning						= false;
	_isStreaming					= false;
	_isSameMediaType				= true;
	_absTime						= 0;
	_numOverwriteFrames				= 0;
	_userDataSize					= 0;
	ZeroMemory(_userData, MR_MAXIMUM_USERDATA_SIZE);
	_changingMediaType.InitMediaType();

	// Property
	_enableConnectReceiveMediaType	= true;
	_timeStampMode					= MRTM_INVALIDATE;
	_videoImageNormalizedPositionX	= 0.0f;
	_videoImageNormalizedPositionY	= 0.0f;
	_videoImageNormalizedWidth		= 1.0f;
	_videoImageNormalizedHeight		= 1.0f;

	// Initialize Network
	flNetwork::initialize();

	// Create FMT Receiver
	CAutoLock lock(&_filterLock);
	_fmtReceiver = new flFMTReceiver(MULTICAST_FILTER_UNIQUEID);
	_fmtReceiver->setReceiveProc(receiveProc, this);
	_fmtReceiver->open();
}
//------------------------------------------------------------------------------
CMulticastReceiver::~CMulticastReceiver()
{
	CAutoLock lock(&_filterLock);

	// Delete FMT Receiver
	_fmtReceiver->close();
	delete _fmtReceiver;

	// Shutdown Network
	flNetwork::shutdown();

	// Delete Media Samples
	for(flUInt i = 0; i < _receiveSamples.getSize(); i++)
		_receiveSamples[i]->release();
	_receiveSamples.clear();

	for(flUInt i = 0; i < _prerollSamples.getSize(); i++)
		_prerollSamples[i]->release();
	_prerollSamples.clear();
	
	for(flUInt i = 0; i < _changingSamples.getSize(); i++)
		_changingSamples[i]->release();
	_changingSamples.clear();
	delete _samplePool;
}
//------------------------------------------------------------------------------
// Reveals IMulticastReceiver & ISpecifyPropertyPages
STDMETHODIMP
CMulticastReceiver::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IMulticastReceiver)
	{
        return GetInterface((IMulticastReceiver *) this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    }
	else
	{
		return CFLSource::NonDelegatingQueryInterface(riid, ppv);
    }
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// IMediaFilter
STDMETHODIMP
CMulticastReceiver::Run(REFERENCE_TIME startTime)
{
	HRESULT hr = CFLSource::Run(startTime);
	if (FAILED(hr))
		return hr;

	CAutoLock lock(&_filterLock);

	_isRunning			= true;
	_isStreaming		= false;
	_curFrameId			= -1;
	_prevFrameId		= -1;
	_absTime			= 0;
	_numOverwriteFrames	= 0;
	_fmtReceiver->resetMonitorVariables();

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::Pause()
{
    return CFLSource::Pause();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::Stop()
{
	{
		CAutoLock lock(&_filterLock);

		_isRunning		= false;
		_isStreaming	= false;
		_fmtReceiver->resetMonitorVariables();
	}

	HRESULT hr = CFLSource::Stop();
	if (FAILED(hr))
		return hr;

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::GetState(DWORD dwMSecs, FILTER_STATE *State)
{
	HRESULT hr = CFLSource::GetState(dwMSecs, State);

	if (m_State == State_Paused)
		return VFW_S_CANT_CUE;
	else
		return hr;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_MulticastReceiver;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CMulticastReceiver::WriteToStream(IStream *pStream)
{
	HRESULT hr;

	// Property
	WRITEOUT(_enableConnectReceiveMediaType);
	WRITEOUT(_timeStampMode);
	WRITEOUT(_videoImageNormalizedPositionX);
	WRITEOUT(_videoImageNormalizedPositionY);
	WRITEOUT(_videoImageNormalizedWidth);
	WRITEOUT(_videoImageNormalizedHeight);

	// FMTReceiver property
	const flString& address	= _fmtReceiver->getAddress();
	flUInt addressLen		= address.length();
	flUShort port			= _fmtReceiver->getPort();
	flULong ttl				= _fmtReceiver->getTimeToLive();
	flUInt numReject		= _fmtReceiver->getNumRejectionAddress();
	WRITEOUT(addressLen);
	WRITENOUT(address.c_str(), addressLen + 1);
	WRITEOUT(port);
	WRITEOUT(ttl);
	WRITEOUT(numReject);
	for(flUInt i = 0; i < numReject; i++)
	{
		const flString rejectAddr = _fmtReceiver->getRejectionAddress(i);
		flUInt length = rejectAddr.length();
		WRITEOUT(length);
		WRITENOUT(rejectAddr.c_str(), length);
	}

	// Output Media type property
	flInt mediaType			= _outputMediaType.getMediaType();
	flInt videoFormat		= _outputMediaType.getVideoFormat();
	flInt videoResolution	= _outputMediaType.getVideoResolution();
	flInt audioFormat		= _outputMediaType.getAudioFormat();
	flInt audioBitCount		= _outputMediaType.getAudioBitCount();
	flInt audioFrequency	= _outputMediaType.getAudioFrequency();
	flInt audioStereo		= _outputMediaType.getAudioStereo();
	WRITEOUT(mediaType);
	WRITEOUT(videoFormat);
	WRITEOUT(videoResolution);
	WRITEOUT(audioFormat);
	WRITEOUT(audioBitCount);
	WRITEOUT(audioFrequency);
	WRITEOUT(audioStereo);

	// Receive Media type property (with buffer copy)
	WRITENOUT(&_receivedMediaType, sizeof(AM_MEDIA_TYPE));
	WRITEOUT(_receivedMediaType.cbFormat);
	if (_receivedMediaType.cbFormat != 0)
		WRITENOUT(_receivedMediaType.pbFormat, _receivedMediaType.cbFormat);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CMulticastReceiver::ReadFromStream(IStream *pStream)
{
	HRESULT hr;

	// Property
	READIN(_enableConnectReceiveMediaType);
	READIN(_timeStampMode);
	READIN(_videoImageNormalizedPositionX);
	READIN(_videoImageNormalizedPositionY);
	READIN(_videoImageNormalizedWidth);
	READIN(_videoImageNormalizedHeight);

	// FMTReceiver Property
	flBool isOpen = _fmtReceiver->isOpen();
	if (isOpen)	_fmtReceiver->close();

	flChar address[MR_MAXIMUM_ADDRESS_LENGTH];
	flUInt addressLen;
	flUShort port;
	flULong ttl;
	flUInt numReject;
	READIN(addressLen);
	READNIN(address, addressLen + 1);
	READIN(port);
	READIN(ttl);
	READIN(numReject);
	_fmtReceiver->setAddress(flString(address));
	_fmtReceiver->setPort(port);
	_fmtReceiver->setTimeToLive(ttl);
	_fmtReceiver->removeAllRejectionAddress();
	for(flUInt i = 0; i < numReject; i++)
	{
		flUInt length;
		READIN(length);
		flChar *pstr = new flChar[length + 1];
		READNIN(pstr, length);
		pstr[length] = '\0';
		_fmtReceiver->addRejectionAddress(flString(pstr));
		delete pstr;
	}

	if (isOpen)	_fmtReceiver->open();

	// Media type property
	flMediaType::MediaTypeEnum mediaType;
	flMediaType::VideoFormatEnum videoFormat;
	flMediaType::VideoResolutionEnum videoResolution;
	flMediaType::AudioFormatEnum audioFormat;
	flMediaType::AudioBitCountEnum audioBitCount;
	flMediaType::AudioFrequencyEnum audioFrequency;
	flMediaType::AudioStereoEnum audioStereo;
	READIN(mediaType);
	READIN(videoFormat);
	READIN(videoResolution);
	READIN(audioFormat);
	READIN(audioBitCount);
	READIN(audioFrequency);
	READIN(audioStereo);
	_outputMediaType.setMediaType(mediaType);
	_outputMediaType.setVideoFormat(videoFormat);
	_outputMediaType.setVideoResolution(videoResolution);
	_outputMediaType.setAudioFormat(audioFormat);
	_outputMediaType.setAudioBitCount(audioBitCount);
	_outputMediaType.setAudioFrequency(audioFrequency);
	_outputMediaType.setAudioStereo(audioStereo);

	// Receive Media type property (with buffer copy)
	_receivedMediaType.ResetFormatBuffer();
	READNIN(&_receivedMediaType, sizeof(AM_MEDIA_TYPE));
	_receivedMediaType.pbFormat = NULL;
	_receivedMediaType.cbFormat = 0;
	ULONG formatBufferSize;
	READIN(formatBufferSize);
	if (formatBufferSize != 0)
	{
		_receivedMediaType.AllocFormatBuffer(formatBufferSize);
		READNIN(_receivedMediaType.pbFormat, formatBufferSize);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CMulticastReceiver::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CMulticastReceiver::SizeMax()
{
	return	sizeof(flBool)  * 2 + 
			sizeof(flFloat) * 4 +
			sizeof(flInt)   * 8 +
			sizeof(flLong)  * 1 + 
			sizeof(flShort) * 1 + 
			sizeof(flChar)  * MR_MAXIMUM_ADDRESS_LENGTH + 
			sizeof(AM_MEDIA_TYPE) +
			sizeof(flChar) * 32768;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages method
STDMETHODIMP
CMulticastReceiver::GetPages(CAUUID *pPages)
{
	pPages->cElems = 5;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 5);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_MulticastReceiverComPropertyPage;
	pPages->pElems[1] = CLSID_MulticastReceiverMediaTypePropertyPage;
	pPages->pElems[2] = CLSID_MulticastReceiverFrameControlPropertyPage;
	pPages->pElems[3] = CLSID_MulticastReceiverRejectionAddressPropertyPage;
	pPages->pElems[4] = CLSID_MulticastReceiverAdditionalInfoPropertyPage;
	return NOERROR;
}
//------------------------------------------------------------------------------
// IMulticastReceiver
STDMETHODIMP
CMulticastReceiver::is_SocketOpen(flBool *flag)
{
	CAutoLock lock(&_filterLock);
	*flag = _fmtReceiver->isOpen();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::is_Streaming(flBool *flag)
{
	//CAutoLock lock(&_filterLock);
	*flag = _isStreaming;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::is_OutputMediaTypeAgreement(flBool *flag)
{
	*flag = _isSameMediaType;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_AbsoluteTime(LONGLONG *absTime)
{
	//CAutoLock lock(&_filterLock);
	*absTime = _absTime;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_ReceiveRate(flUInt *rate)
{
	//CAutoLock lock(&_filterLock);
	*rate = _fmtReceiver->getReceiveBitRate();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_FrameRate(flUInt *rate)
{
	//CAutoLock lock(&_filterLock);
	*rate = _fmtReceiver->getReceiveDataRate();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_PacketSize(flUInt *byte)
{
	//CAutoLock lock(&_filterLock);
	*byte = _fmtReceiver->getPacketSize();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_NumTotalPackets(flUInt *numPackets)
{
	//CAutoLock lock(&_filterLock);
	*numPackets = _fmtReceiver->getNumReceivedPackets();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_NumTotalFrames(flUInt *numFrames)
{
	//CAutoLock lock(&_filterLock);
	*numFrames = _fmtReceiver->getNumReceivedDatas() - _numOverwriteFrames;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_NumDropPackets(flUInt *numPackets)
{
	//CAutoLock lock(&_filterLock);
	*numPackets = _fmtReceiver->getNumDropPackets();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_NumDropFrames(flUInt *numFrames)
{
	//CAutoLock lock(&_filterLock);
	*numFrames = _fmtReceiver->getNumDropDatas() + _numOverwriteFrames;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_NumOverwriteFrames(flUInt *numFrames)
{
	//CAutoLock lock(&_filterLock);
	*numFrames = _numOverwriteFrames;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_UserData(flByte *userData, flUInt *size)
{
	//CAutoLock lock(&_filterLock);
	CopyMemory(userData, _userData, _userDataSize);
	*size = _userDataSize;
	return NOERROR;
}
//------------------------------------------------------------------------------
#include <flFilterCommon/flTextMediaType.h>
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_MediaTypeString(flChar *str)
{
	flTextMediaType textMediaType(*((AM_MEDIA_TYPE *)&_receivedMediaType));
	textMediaType.AsText(str, MR_MAXIMUM_MEDIATYPESTR_LENGTH);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_SenderAddress(flChar *address)
{
	if (address == NULL)
		return NOERROR;

	address[0] = '\0';

	const flString addr = _fmtReceiver->getSenderAddress();
	if (addr.length() != 0)
		strcpy(address, addr.c_str());

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_Address(flChar *address)
{
	//CAutoLock lock(&_filterLock);

	if (address == NULL)
		return NOERROR;

	const flString addr = _fmtReceiver->getAddress();
	if (addr.length() == 0)
		strcpy(address, "-");
	else
		strcpy(address, addr.c_str());

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_Address(const flChar *address)
{
	CAutoLock lock(&_filterLock);

	if (address == NULL || strlen(address) == 0)
		return NOERROR;

	if (!strcmp(address, _fmtReceiver->getAddress().c_str()))
		return NOERROR;

	_fmtReceiver->close();
	_fmtReceiver->setAddress(flString(address));
	_fmtReceiver->open();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_Port(flUShort *port)
{
	//CAutoLock lock(&_filterLock);
	*port = _fmtReceiver->getPort();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_Port(flUShort port)
{
	if (port == _fmtReceiver->getPort())
		return NOERROR;

	CAutoLock lock(&_filterLock);
	_fmtReceiver->close();
	_fmtReceiver->setPort(port);
	_fmtReceiver->open();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_TimeToLive(flULong *ttl)
{
	//CAutoLock lock(&_filterLock);
	*ttl = _fmtReceiver->getTimeToLive();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_TimeToLive(flULong ttl)
{
	if (ttl == _fmtReceiver->getTimeToLive())
		return NOERROR;

	CAutoLock lock(&_filterLock);
	_fmtReceiver->close();
	_fmtReceiver->setTimeToLive(ttl);
	_fmtReceiver->open();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_NumRejectionAddress(flUInt *num)
{
	//CAutoLock lock(&_filterLock);
	*num = _fmtReceiver->getNumRejectionAddress();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_RejectionAddress(flUInt index, flChar *address)
{
	//CAutoLock lock(&_filterLock);

	if (address == NULL)
		return NOERROR;

	const flString addr = _fmtReceiver->getRejectionAddress(index);
	strcpy(address, addr.c_str());
	
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::add_RejectionAddress(const flChar *address)
{
	CAutoLock lock(&_filterLock);

	if (address == NULL)
		return NOERROR;

	_fmtReceiver->addRejectionAddress(flString(address));
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::remove_RejectionAddress(flUInt index)
{
	CAutoLock lock(&_filterLock);
	_fmtReceiver->removeRejectionAddress(index);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::remove_AllRejectionAddress()
{
	CAutoLock lock(&_filterLock);
	_fmtReceiver->removeAllRejectionAddress();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_MediaType(flInt *type)
{
	//CAutoLock lock(&_filterLock);
	*type = _outputMediaType.getMediaType();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_MediaType(flInt type)
{
	CAutoLock lock(&_filterLock);
	_outputMediaType.setMediaType((flMediaType::MediaTypeEnum)type);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_VideoFormat(flInt *videoFormat)
{
	//CAutoLock lock(&_filterLock);
	*videoFormat = _outputMediaType.getVideoFormat();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_VideoFormat(flInt videoFormat)
{
	CAutoLock lock(&_filterLock);
	_outputMediaType.setVideoFormat((flMediaType::VideoFormatEnum)videoFormat);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_VideoResolution(flInt *videoResolution)
{
	//CAutoLock lock(&_filterLock);
	*videoResolution = _outputMediaType.getVideoResolution();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_VideoResolution(flInt videoResolution)
{
	CAutoLock lock(&_filterLock);
	_outputMediaType.setVideoResolution((flMediaType::VideoResolutionEnum)videoResolution);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_AudioFormat(flInt *audioFormat)
{
	//CAutoLock lock(&_filterLock);
	*audioFormat = _outputMediaType.getAudioFormat();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_AudioFormat(flInt audioFormat)
{
	CAutoLock lock(&_filterLock);
	_outputMediaType.setAudioFormat((flMediaType::AudioFormatEnum)audioFormat);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_AudioFrequency(flInt *audioFrequency)
{
	//CAutoLock lock(&_filterLock);
	*audioFrequency = _outputMediaType.getAudioFrequency();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_AudioFrequency(flInt audioFrequency)
{
	CAutoLock lock(&_filterLock);
	_outputMediaType.setAudioFrequency((flMediaType::AudioFrequencyEnum)audioFrequency);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_AudioStereo(flInt *audioStereo)
{
	//CAutoLock lock(&_filterLock);
	*audioStereo = _outputMediaType.getAudioStereo();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_AudioStereo(flInt audioStereo)
{
	CAutoLock lock(&_filterLock);
	_outputMediaType.setAudioStereo((flMediaType::AudioStereoEnum)audioStereo);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_AudioBitCount(flInt *audioBitCount)
{
	//CAutoLock lock(&_filterLock);
	*audioBitCount = _outputMediaType.getAudioBitCount();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_AudioBitCount(flInt audioBitCount)
{
	CAutoLock lock(&_filterLock);
	_outputMediaType.setAudioBitCount((flMediaType::AudioBitCountEnum)audioBitCount);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_EnableConnectReceiveMediaType(flBool *enable)
{
	//CAutoLock lock(&_filterLock);
	*enable = _enableConnectReceiveMediaType;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_EnableConnectReceiveMediaType(flBool enable)
{
	CAutoLock lock(&_filterLock);
	_enableConnectReceiveMediaType = enable;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_TimeStampMode(flInt *mode)
{
	//CAutoLock lock(&_filterLock);
	*mode = _timeStampMode;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_TimeStampMode(flInt mode)
{
	CAutoLock lock(&_filterLock);
	_timeStampMode = mode;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_VideoImageNormalizedPosition(flFloat *x, flFloat *y)
{
	//CAutoLock lock(&_filterLock);
	*x = _videoImageNormalizedPositionX;
	*y = _videoImageNormalizedPositionY;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_VideoImageNormalizedPosition(flFloat x, flFloat y)
{
	CAutoLock lock(&_filterLock);
	_videoImageNormalizedPositionX = x;
	_videoImageNormalizedPositionY = y;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::get_VideoImageNormalizedSize(flFloat *width, flFloat *height)
{
	//CAutoLock lock(&_filterLock);
	*width = _videoImageNormalizedWidth;
	*height = _videoImageNormalizedHeight;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastReceiver::put_VideoImageNormalizedSize(flFloat width, flFloat height)
{
	CAutoLock lock(&_filterLock);
	_videoImageNormalizedWidth = width;
	_videoImageNormalizedHeight = height;
	return NOERROR;
}
//------------------------------------------------------------------------------
void __stdcall
CMulticastReceiver::receiveProc(flChar* data, flUInt dataSize, void* receiveParam)
{
	CMulticastReceiver *receiver = (CMulticastReceiver *)receiveParam;
	receiver->receive(data, dataSize);
}
//------------------------------------------------------------------------------
void
CMulticastReceiver::receive(flChar* data, flUInt dataSize)
{
	CAutoLock lock(&_filterLock);

	//
	// Analyze Transmit Data
	//
	_iStream.set(data, dataSize);

	// Frame Id
	_prevFrameId = _curFrameId;
	_iStream.readUInt((flUInt *)&_curFrameId, 1);

	// User Data
	_iStream.readUInt(&_userDataSize, 1);
	_iStream.readByte(_userData, _userDataSize);

	// Media Type
	CMediaType mediaType;
	_iStream.readChar((flChar *)&mediaType, sizeof(AM_MEDIA_TYPE));
	mediaType.pbFormat = NULL;
	mediaType.cbFormat = 0;

	// Media Format
	flUInt formatSize;
	_iStream.readUInt(&formatSize, 1);
	if (formatSize != 0)
	{
		mediaType.AllocFormatBuffer(formatSize);
		_iStream.readChar((flChar *)mediaType.pbFormat, formatSize);
	}

	//
	// Check Media Type
	//

	// mediaType			:  ̌ďoŎ󂯎fBA^Cv
	// _receivedMediaType	:  ݂̃fBA^Cv
	// _changingMediaType	:  fBA^CvύXĎp
	if ((_receivedMediaType != mediaType) == TRUE)
	{
		if ((_changingMediaType != mediaType) == TRUE)
		{
			_changingMediaType = mediaType;

			for(flUInt i = 0; i < _changingSamples.getSize(); i++)
				_changingSamples[i]->release();
			_changingSamples.clear();
		}

		_changingSamples.add(createNewMediaSample(mediaType, _iStream));

		// Change MediaType
		if (CHANGEMEDIATYPE_THRESHOLD < _changingSamples.getSize())
		{
			_receivedMediaType = mediaType;

			{
				CAutoLock lock2(&_prerollSampleLock);
				for(flUInt i =0; i < _prerollSamples.getSize(); i++)
					_prerollSamples[i]->release();

				_prerollSamples = _changingSamples;
				setupPrerollSamples();
			}

			_changingMediaType.InitMediaType();
			_changingSamples.clear();

			// Fire MEDIATYPE_CHANGED Event
			fireMediaTypeChangedEvent();
		}

		return ;
	}
	_changingMediaType.InitMediaType();

	// Exit If Pin is not Connected.
	if (_multicastReceiverStream->IsConnected() != TRUE)
	{
		_isSameMediaType = true;
		return ;
	}

	// Exit If Received MediaType is different from Actual Output MediaType
	_isSameMediaType = ((mediaType == _actualOutputMediaType) == TRUE);
	if (!_isSameMediaType)
		return ;

	//
	// Create New Media Sample
	//
	flMediaSample* sample = createNewMediaSample(mediaType, _iStream);

	// Append Sample to Buffer
	CAutoLock lock3(&_receiveSampleLock);
	if (SAMPLE_BUFFERSIZE <= _receiveSamples.getSize())
	{
		_receiveSamples[0]->release();
		_receiveSamples.removeIndex(0);
		_numOverwriteFrames ++;
	}
	_receiveSamples.add(sample);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
flMediaSample*
CMulticastReceiver::createNewMediaSample(const CMediaType& mediaType, flInputByteStream& iStream)
{
	flMediaSample* sample = _samplePool->getFreeMediaSample();

	// Media Type
	sample->setMediaType(mediaType);

	// Absolute Time
	REFERENCE_TIME absTime;
	iStream.readInt64((flInt64 *)&absTime, 1);
	sample->setAbsoluteTime(absTime);

	//
	// Media Sample
	//

	// Stream Time
	flBool timeStartSet, timeEndSet;
	REFERENCE_TIME timeStart, timeEnd;
	iStream.readBool(&timeStartSet, 1);
	iStream.readBool(&timeEndSet, 1);
	iStream.readInt64((flInt64 *)&timeStart, 1);
	iStream.readInt64((flInt64 *)&timeEnd, 1);
	if (timeStartSet)	sample->setStreamTimeStart(&timeStart);
	else				sample->setStreamTimeStart(NULL);
	if (timeEndSet)		sample->setStreamTimeEnd(&timeEnd);
	else				sample->setStreamTimeEnd(NULL);

	// Media Time
	flBool mediaTimeStartSet, mediaTimeEndSet;
	LONGLONG mediaTimeStart, mediaTimeEnd;
	iStream.readBool(&mediaTimeStartSet, 1);
	iStream.readBool(&mediaTimeEndSet, 1);
	iStream.readInt64((flInt64 *)&mediaTimeStart, 1);
	iStream.readInt64((flInt64 *)&mediaTimeEnd, 1);
	if (mediaTimeStartSet)	sample->setMediaTimeStart(&mediaTimeStart);
	else					sample->setMediaTimeStart(NULL);
	if (mediaTimeEndSet)	sample->setMediaTimeEnd(&mediaTimeEnd);
	else					sample->setMediaTimeEnd(NULL);

	flBool flag;
	// Sync Point
	iStream.readBool(&flag, 1);
	sample->setSyncPoint(flag);

	// Discontinuity
	flBool dropFrame = (_prevFrameId != -1 && _prevFrameId + 1 != _curFrameId);
	iStream.readBool(&flag, 1);
	sample->setDiscontinuity(dropFrame || flag);

	// Preroll
	iStream.readBool(&flag, 1);
	sample->setPreroll(flag);

	// Media Data
	flUInt dataLength;
	iStream.readUInt(&dataLength, 1);
	sample->setDataLength(dataLength);
	iStream.readByte(sample->getDataPointer(), dataLength);

	return sample;
}
//------------------------------------------------------------------------------
void
CMulticastReceiver::setupPrerollSamples()
{
	for(flUInt i = 0; i < _prerollSamples.getSize(); i++)
	{
		flMediaSample* sample = _prerollSamples[i];
		sample->setMediaTimeStart(NULL);
		sample->setMediaTimeEnd(NULL);
		sample->setStreamTimeStart(NULL);
		sample->setStreamTimeEnd(NULL);
	}
}
//------------------------------------------------------------------------------
void
CMulticastReceiver::fireMediaTypeChangedEvent()
{
	CAutoLock graphLock(m_pLock);

	if (!_enableConnectReceiveMediaType || m_pGraph == NULL)
		return ;

	IMediaEventSink *mediaEventSink;
	HRESULT hr = m_pGraph->QueryInterface(IID_IMediaEventSink, (void **)&mediaEventSink);
	if (hr == S_OK)
	{
#ifdef _DEBUG
		flNotify::post(flNotify::L_INFO, "MulticastReceiver::FireEvent! Port(%d)", _fmtReceiver->getPort());
#endif
		mediaEventSink->Notify(EC_MEDIATYPE_CHANGED, (LONG_PTR)m_pGraph, (LONG_PTR)(IBaseFilter *)this);
		mediaEventSink->Release();
		mediaEventSink = NULL;
	}
}
//------------------------------------------------------------------------------
