/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flMulticastSender/MulticastSender.h>
#include <flMulticastSender/MulticastSenderComProp.h>
#include <flMulticastSender/MulticastSenderMediaProp.h>
#include <flMulticastSender/MulticastSenderFrameProp.h>
#include <flMulticastSender/MulticastSenderUserProp.h>
#include <flFilterCommon/flMediaType.h>
#include <flFilterCommon/flFilterCommon.h>
#include <flNetwork/flFMTSender.h>
#include <flNetwork/flNetwork.h>
#include <flBase/flSystemTime.h>
#include <flBase/flNotify.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudIpPinTypes =
{
    &MEDIATYPE_NULL,			// MajorType
    &MEDIASUBTYPE_NULL			// MinorType
};
//------------------------------------------------------------------------------
const AMOVIESETUP_PIN sudIpPin =
{
    L"Input",					// The Pins name
    FALSE,						// Is rendered
    FALSE,						// Is an output pin
    FALSE,						// Allowed none
    FALSE,						// Allowed many
    &CLSID_NULL,				// Connects to filter
    NULL,						// Connects to pin
    1,							// Number of types
    &sudIpPinTypes				// Pin details
};
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudMulticastSender =
{
    &CLSID_MulticastSender,		// Filter CLSID
    L"NIME Multicast Sender",	// String name
    MERIT_DO_NOT_USE,			// Filter merit
    1,							// Number of pins
    &sudIpPin					// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[5] = {
    { L"MulticastSender"
    , &CLSID_MulticastSender
    , CMulticastSender::CreateInstance
    , NULL
    , &sudMulticastSender }
  ,
    { L"MulticastSenderComProp"
    , &CLSID_MulticastSenderComPropertyPage
    , CMulticastSenderComProp::CreateInstance }
  ,
    { L"MulticastSenderMediaProp"
    , &CLSID_MulticastSenderMediaTypePropertyPage
    , CMulticastSenderMediaProp::CreateInstance }
  ,
    { L"MulticastSenderFrameProp"
    , &CLSID_MulticastSenderFrameControlPropertyPage
    , CMulticastSenderFrameProp::CreateInstance }
  ,
    { L"MulticastSenderUserProp"
    , &CLSID_MulticastSenderUserDataPropertyPage
    , CMulticastSenderUserProp::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CMulticastSender::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
    CMulticastSender *pNewObject = new CMulticastSender(NAME("MulticastSender"), punk, phr);
    if (pNewObject == NULL)
	{
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CMulticastSender::CMulticastSender(TCHAR *tszName, LPUNKNOWN pUnk, HRESULT *phr) :
CBaseRenderer(CLSID_MulticastSender, tszName, pUnk, phr),
CPersistStream(pUnk, phr)
{
    ASSERT(tszName != NULL);
    ASSERT(phr != NULL);

	CAutoLock autoLock(&_filterLock);

	// MediaPosition (CPosPassThru)
	_mediaPosition		= NULL;

	// Initialize Network
	flNetwork::initialize();

	// Create FMT Sender
	_fmtSender			= new flFMTSender(MULTICAST_FILTER_UNIQUEID);
	_fmtSender->open();

	// Absolute Time
	_absTime			= new flSystemTime();

	// Status
	_frameId			= 0;

	// Property
	_enableTransmit		= true;
	_userDataSize		= 0;
	ZeroMemory(_userData, MS_MAXIMUM_USERDATA_SIZE);

	// MediaType
	_inputMediaType.setMediaType((flMediaType::MediaTypeEnum)IMulticastSender::MSMT_ALL);
	_sendMediaType.InitMediaType();
}
//------------------------------------------------------------------------------
CMulticastSender::~CMulticastSender()
{
	CAutoLock autoLock(&_filterLock);

	// Absolute Time
	delete _absTime;

	// Shutdown FMT Sender
	_fmtSender->close();
	delete _fmtSender;

	// Shutdown Network
	flNetwork::shutdown();

	// MediaPosition (CPosPassThru)
	delete _mediaPosition;
}
//------------------------------------------------------------------------------
// Reveals IMulticastSender & ISpecifyPropertyPages
STDMETHODIMP
CMulticastSender::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IMulticastSender)
	{
        return GetInterface((IMulticastSender *) this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    }
	else if (riid == IID_IMediaPosition || riid == IID_IMediaSeeking)
	{
		if (_mediaPosition == NULL) 
		{
			HRESULT hr = S_OK;
			_mediaPosition = new CPosPassThru(NAME("MulticastSender Pass Through"),
				(IUnknown *)m_pInputPin->GetOwner(), (HRESULT *)&hr, m_pInputPin);

			if (_mediaPosition == NULL) 
				return E_OUTOFMEMORY;

			if (FAILED(hr)) 
			{
				delete _mediaPosition;
				_mediaPosition = NULL;
				return hr;
			}
		}
		return _mediaPosition->QueryInterface(riid, ppv);
	}
	else
	{
        return CBaseRenderer::NonDelegatingQueryInterface(riid, ppv);
    }

}
//------------------------------------------------------------------------------
HRESULT
CMulticastSender::BreakConnect()
{
	if (_mediaPosition != NULL)
	    _mediaPosition->ForceRefresh();

	return CBaseRenderer::BreakConnect();
}
//------------------------------------------------------------------------------
HRESULT
CMulticastSender::DoRenderSample(IMediaSample *pMediaSample)
{
/*
	checkMediaSample2Properties(pMediaSample);
*/

	ASSERT(pMediaSample != NULL);
	CAutoLock autoLock(&_filterLock);

	if (!_fmtSender->isOpen() || !_enableTransmit)
		return NOERROR;

	//
	// Build Transmit Data
	//
	_oStream.reset();

	// Frame Id
	_oStream.writeUInt(&_frameId, 1);
	_frameId++;

	// User Data
	_oStream.writeUInt(&_userDataSize, 1);
	_oStream.writeByte(_userData, _userDataSize);

	// Media Type
	_oStream.writeChar((const flChar *)&_sendMediaType, sizeof(AM_MEDIA_TYPE));

	// Media Format
	flUInt formatSize = _sendMediaType.cbFormat;

	_oStream.writeUInt(&formatSize, 1);
	_oStream.writeChar((const flChar *)_sendMediaType.pbFormat, formatSize);


	//
	// Build Media sample
	//

	// Absolute Time
	_absTime->update();
	flInt64 absTime = _absTime->getTime();
	_oStream.writeInt64(&absTime, 1);


	HRESULT hr;
	flBool timeStartSet, timeEndSet;

	// Stream Time
	REFERENCE_TIME timeStart, timeEnd;
	hr = pMediaSample->GetTime(&timeStart, &timeEnd);
	if (hr == VFW_S_NO_STOP_TIME)
	{
		timeStartSet = true;
		timeEndSet = false;
	}
	else if (hr == VFW_E_SAMPLE_TIME_NOT_SET)
	{
		timeStartSet = timeEndSet = false;
	}
	else
	{
		timeStartSet = timeEndSet = true;
	}
	_oStream.writeBool(&timeStartSet, 1);
	_oStream.writeBool(&timeEndSet, 1);
	_oStream.writeInt64(&timeStart, 1);
	_oStream.writeInt64(&timeEnd, 1);

	// Media Time
	LONGLONG mediaTimeStart, mediaTimeEnd;
	hr = pMediaSample->GetMediaTime(&mediaTimeStart, &mediaTimeEnd);
	if (hr == VFW_E_MEDIA_TIME_NOT_SET)
		timeStartSet = timeEndSet = false;
	else
		timeStartSet = timeEndSet = true;

	_oStream.writeBool(&timeStartSet, 1);
	_oStream.writeBool(&timeEndSet, 1);
	_oStream.writeInt64(&mediaTimeStart, 1);
	_oStream.writeInt64(&mediaTimeEnd, 1);

	flBool flag;

	// Sync Point
	flag = (pMediaSample->IsSyncPoint() == S_OK);
	_oStream.writeBool(&flag, 1);
	
	// Discontinuity
	flag = (pMediaSample->IsDiscontinuity() == S_OK);
	_oStream.writeBool(&flag, 1);

	// Preroll
	flag = (pMediaSample->IsPreroll() == S_OK);
	_oStream.writeBool(&flag, 1);

	// Media Data
	flByte* pointer;
	flLong dataLength;
	pMediaSample->GetPointer(&pointer);
	dataLength = pMediaSample->GetActualDataLength();
	_oStream.writeUInt((const flUInt *)&dataLength, 1);
	_oStream.writeByte(pointer, dataLength);

	//
	// Send Datas
	//
	const flChar* data;
	flUInt dataSize;
	_oStream.get(data, dataSize);
	_fmtSender->send(data, dataSize);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CMulticastSender::CheckMediaType(const CMediaType *pmt)
{
	flMediaType mediaType;

	mediaType = *pmt;

	flMediaType::MediaTypeEnum inputMediaType = _inputMediaType.getMediaType();
	if (inputMediaType == flMediaType::MT_UNKNOWN)
	{
		return S_OK;
	}
	else if (inputMediaType == mediaType.getMediaType())
	{
		switch(inputMediaType)
		{
		case flMediaType::MT_VIDEO:
			if ((_inputMediaType.getVideoFormat() == flMediaType::VF_UNKNOWN ||
				 _inputMediaType.getVideoFormat() == mediaType.getVideoFormat()) &&
				(_inputMediaType.getVideoResolution() == flMediaType::VR_UNKNOWN ||
				 _inputMediaType.getVideoResolution() == mediaType.getVideoResolution()))
			{
				return S_OK;
			}
			break;
		case flMediaType::MT_AUDIO:
			if ((_inputMediaType.getAudioFormat() == flMediaType::AF_UNKNOWN ||
				 _inputMediaType.getAudioFormat() == mediaType.getAudioFormat()) &&
				(_inputMediaType.getAudioFrequency() == flMediaType::AQ_UNKNOWN ||
				 _inputMediaType.getAudioFrequency() == mediaType.getAudioFrequency()) &&
				(_inputMediaType.getAudioStereo() == flMediaType::AS_UNKNOWN ||
				 _inputMediaType.getAudioStereo() == mediaType.getAudioStereo()) &&
				(_inputMediaType.getAudioBitCount() == flMediaType::AB_UNKNOWN ||
				 _inputMediaType.getAudioBitCount() == mediaType.getAudioBitCount()))
			{
				return S_OK;
			}
			break;
		case flMediaType::MT_POINTER:
			{
				return S_OK;
			}
			break;
		}
	}

	return S_FALSE;
}
//------------------------------------------------------------------------------
HRESULT
CMulticastSender::SetMediaType(const CMediaType *pmt)
{
	_sendMediaType.ResetFormatBuffer();
	_sendMediaType = *pmt;

// 031106
// AbvXg[̃TvTCY̒lۂ菬
// ȂĂ܂ꍇւ̑ΉB̕s̂ƂA󂯑ŃfBAf[^
// ĐȂƂB
// mFfBA^Cv
//   PCM :   4-byte
//   MP3 :   1-byte
	if (IsEqualGUID(_sendMediaType.majortype, MEDIATYPE_Audio) && 
		_sendMediaType.IsFixedSize() == TRUE &&
		_sendMediaType.GetSampleSize() < 100000)
		_sendMediaType.SetSampleSize(100000);
// 031106

	return S_OK;
}
//------------------------------------------------------------------------------
// IMediaFilter
STDMETHODIMP
CMulticastSender::Run(REFERENCE_TIME startTime)
{
	HRESULT hr = CBaseRenderer::Run(startTime);
	if (FAILED(hr))
		return hr;

	CAutoLock autoLock(&_filterLock);

	_frameId = 0;
	_fmtSender->resetMonitorVariables();

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::Pause()
{
    return CBaseRenderer::Pause();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::Stop()
{
	HRESULT hr = CBaseRenderer::Stop();
	if (FAILED(hr))
		return hr;

	CAutoLock autoLock(&_filterLock);

	_fmtSender->resetMonitorVariables();

	return hr;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_MulticastSender;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CMulticastSender::WriteToStream(IStream *pStream)
{
	HRESULT hr;

	// Property
	WRITEOUT(_enableTransmit);
	WRITEOUT(_userDataSize);
	WRITENOUT(_userData, _userDataSize);

	// Input Media Type
	flInt mediaType = _inputMediaType.getMediaType();
	flInt videoFormat = _inputMediaType.getVideoFormat();
	flInt videoResolution = _inputMediaType.getVideoResolution();
	flInt audioFormat = _inputMediaType.getAudioFormat();
	flInt audioFrequency = _inputMediaType.getAudioFrequency();
	flInt audioStereo = _inputMediaType.getAudioStereo();
	flInt audioBitCount = _inputMediaType.getAudioBitCount();
	WRITEOUT(mediaType);
	WRITEOUT(videoFormat);
	WRITEOUT(videoResolution);
	WRITEOUT(audioFormat);
	WRITEOUT(audioFrequency);
	WRITEOUT(audioStereo);
	WRITEOUT(audioBitCount);

	// FMTSender property
	const flString& address	= _fmtSender->getAddress();
	flUInt addressLen		= address.length();
	flUShort port			= _fmtSender->getPort();
	flULong ttl				= _fmtSender->getTimeToLive();
	flUInt packetSize		= _fmtSender->getPacketSize();
	flUInt transmissionRate	= _fmtSender->getMaximumTransmissionRate();
	flUInt transmissionCount= _fmtSender->getTransmissionCount();
	flUInt systemBufferSize	= _fmtSender->getSystemBufferSize();
	flFloat clockMinimumPeriod = _fmtSender->getClockMinimumPeriod();
	WRITEOUT(addressLen);
	WRITENOUT(address.c_str(), addressLen + 1);
	WRITEOUT(port);
	WRITEOUT(ttl);
	WRITEOUT(packetSize);
	WRITEOUT(transmissionRate);
	WRITEOUT(transmissionCount);
	WRITEOUT(systemBufferSize);
	WRITEOUT(clockMinimumPeriod);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CMulticastSender::ReadFromStream(IStream *pStream)
{
	HRESULT hr;

	// Property
	READIN(_enableTransmit);
	READIN(_userDataSize);
	READNIN(_userData, _userDataSize);

	// Input Media Type
	flInt mediaType;
	flInt videoFormat;
	flInt videoResolution;
	flInt audioFormat;
	flInt audioFrequency;
	flInt audioStereo;
	flInt audioBitCount;
	READIN(mediaType);
	READIN(videoFormat);
	READIN(videoResolution);
	READIN(audioFormat);
	READIN(audioFrequency);
	READIN(audioStereo);
	READIN(audioBitCount);
	_inputMediaType.setMediaType((flMediaType::MediaTypeEnum)mediaType);
	_inputMediaType.setVideoFormat((flMediaType::VideoFormatEnum)videoFormat);
	_inputMediaType.setVideoResolution((flMediaType::VideoResolutionEnum)videoResolution);
	_inputMediaType.setAudioFormat((flMediaType::AudioFormatEnum)audioFormat);
	_inputMediaType.setAudioFrequency((flMediaType::AudioFrequencyEnum)audioFrequency);
	_inputMediaType.setAudioStereo((flMediaType::AudioStereoEnum)audioStereo);
	_inputMediaType.setAudioBitCount((flMediaType::AudioBitCountEnum)audioBitCount);

	// FMTSender property
	flBool isOpen = _fmtSender->isOpen();
	if (isOpen)	_fmtSender->close();

	flChar address[MS_MAXIMUM_ADDRESS_LENGTH];
	flUInt addressLen;
	flUShort port;
	flULong ttl;
	flUInt packetSize;
	flUInt transmissionRate;
	flUInt transmissionCount;
	flUInt systemBufferSize;
	flFloat clockMinimumPeriod;
	READIN(addressLen);
	READNIN(address, addressLen + 1);
	READIN(port);
	READIN(ttl);
	READIN(packetSize);
	READIN(transmissionRate);
	READIN(transmissionCount);
	READIN(systemBufferSize);
	READIN(clockMinimumPeriod);
	_fmtSender->setAddress(flString(address));
	_fmtSender->setPort(port);
	_fmtSender->setTimeToLive(ttl);
	_fmtSender->setPacketSize(packetSize);
	_fmtSender->setMaximumTransmissionRate(transmissionRate);
	_fmtSender->setTransmissionCount(transmissionCount);
	_fmtSender->setSystemBufferSize(systemBufferSize);
	_fmtSender->setClockMinimumPeriod(clockMinimumPeriod);

	if (isOpen)	_fmtSender->open();

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CMulticastSender::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CMulticastSender::SizeMax()
{
	return	sizeof(flInt)   * 13 +
			sizeof(flShort) * 1 + 
			sizeof(flLong)  * 1 +
			sizeof(flFloat) * 1 +
			sizeof(flChar)  * MS_MAXIMUM_ADDRESS_LENGTH +
			sizeof(flChar)  * MS_MAXIMUM_USERDATA_SIZE +
			sizeof(flBool);
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages method
STDMETHODIMP
CMulticastSender::GetPages(CAUUID *pPages)
{
	pPages->cElems = 4;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 4);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_MulticastSenderComPropertyPage;
	pPages->pElems[1] = CLSID_MulticastSenderMediaTypePropertyPage;
	pPages->pElems[2] = CLSID_MulticastSenderFrameControlPropertyPage;
	pPages->pElems[3] = CLSID_MulticastSenderUserDataPropertyPage;
	return NOERROR;
}
//------------------------------------------------------------------------------
// IMulticastSender
STDMETHODIMP
CMulticastSender::is_SocketOpen(flBool *flag)
{
	CAutoLock autoLock(&_filterLock);
	*flag = _fmtSender->isOpen();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_AbsoluteTime(flInt64 *absoluteTime)
{
	//CAutoLock autoLock(&_filterLock);
	_absTime->update();
	*absoluteTime = _absTime->getTime();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_TransmissionRate(flUInt *rate)
{
	//CAutoLock autoLock(&_filterLock);
	*rate = _fmtSender->getTransmissionBitRate();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_FrameRate(flUInt *rate)
{
	//CAutoLock autoLock(&_filterLock);
	*rate = _fmtSender->getTransmissionDataRate();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_NumTotalPackets(flUInt *numPackets)
{
	//CAutoLock autoLock(&_filterLock);
	*numPackets = _fmtSender->getNumSendPackets();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_NumTotalFrames(flUInt *numFrames)
{
	//CAutoLock autoLock(&_filterLock);
	*numFrames = _fmtSender->getNumSendDatas();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_MediaType(flInt *type)
{
	*type = _inputMediaType.getMediaType();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_MediaType(flInt type)
{
	_inputMediaType.setMediaType((flMediaType::MediaTypeEnum)type);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_InputMediaType(flInt *type)
{
	//CAutoLock autoLock(&_filterLock);

	*type = MSMT_UNKNOWN;

	if (m_pInputPin == NULL)
		return NOERROR;

	AM_MEDIA_TYPE mt;
	if (m_pInputPin->ConnectionMediaType(&mt) == S_OK)
	{
		flMediaType mediaType;
		mediaType = mt;
		*type = mediaType.getMediaType();
		CoTaskMemFree(mt.pbFormat);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_VideoFormat(flInt *videoFormat)
{
	*videoFormat = _inputMediaType.getVideoFormat();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_VideoFormat(flInt videoFormat)
{
	_inputMediaType.setVideoFormat((flMediaType::VideoFormatEnum)videoFormat);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_InputVideoFormat(flInt *videoFormat)
{
	//CAutoLock autoLock(&_filterLock);

	*videoFormat = MSVF_UNKNOWN;

	if (m_pInputPin == NULL)
		return NOERROR;

	AM_MEDIA_TYPE mt;
	if (m_pInputPin->ConnectionMediaType(&mt) == S_OK)
	{
		flMediaType mediaType;
		mediaType = mt;
		*videoFormat = mediaType.getVideoFormat();
		CoTaskMemFree(mt.pbFormat);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_VideoResolution(flInt *videoResolution)
{
	*videoResolution = _inputMediaType.getVideoResolution();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_VideoResolution(flInt videoResolution)
{
	_inputMediaType.setVideoResolution((flMediaType::VideoResolutionEnum)videoResolution);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_InputVideoResolution(flInt *videoResolution)
{
	//CAutoLock autoLock(&_filterLock);

	*videoResolution = MSVR_UNKNOWN;

	if (m_pInputPin == NULL)
		return NOERROR;

	AM_MEDIA_TYPE mt;
	if (m_pInputPin->ConnectionMediaType(&mt) == S_OK)
	{
		flMediaType mediaType;
		mediaType = mt;
		*videoResolution = mediaType.getVideoResolution();
		CoTaskMemFree(mt.pbFormat);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_AudioFormat(flInt *audioFormat)
{
	*audioFormat = _inputMediaType.getAudioFormat();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_AudioFormat(flInt audioFormat)
{
	_inputMediaType.setAudioFormat((flMediaType::AudioFormatEnum)audioFormat);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_InputAudioFormat(flInt *audioFormat)
{
	//CAutoLock autoLock(&_filterLock);

	*audioFormat = MSAF_UNKNOWN;

	if (m_pInputPin == NULL)
		return NOERROR;

	AM_MEDIA_TYPE mt;
	if (m_pInputPin->ConnectionMediaType(&mt) == S_OK)
	{
		flMediaType mediaType;
		mediaType = mt;
		*audioFormat = mediaType.getAudioFormat();
		CoTaskMemFree(mt.pbFormat);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_AudioFrequency(flInt *audioFrequency)
{
	*audioFrequency = _inputMediaType.getAudioFrequency();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_AudioFrequency(flInt audioFrequency)
{
	_inputMediaType.setAudioFrequency((flMediaType::AudioFrequencyEnum)audioFrequency);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_InputAudioFrequency(flInt *audioFrequency)
{
	//CAutoLock autoLock(&_filterLock);

	*audioFrequency = MSAQ_UNKNOWN;

	if (m_pInputPin == NULL)
		return NOERROR;

	AM_MEDIA_TYPE mt;
	if (m_pInputPin->ConnectionMediaType(&mt) == S_OK)
	{
		flMediaType mediaType;
		mediaType = mt;
		*audioFrequency = mediaType.getAudioFrequency();
		CoTaskMemFree(mt.pbFormat);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_AudioStereo(flInt *audioStereo)
{
	*audioStereo = _inputMediaType.getAudioStereo();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_AudioStereo(flInt audioStereo)
{
	_inputMediaType.setAudioStereo((flMediaType::AudioStereoEnum)audioStereo);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_InputAudioStereo(flInt *audioStereo)
{
	//CAutoLock autoLock(&_filterLock);

	*audioStereo = MSAS_UNKNOWN;

	if (m_pInputPin == NULL)
		return NOERROR;

	AM_MEDIA_TYPE mt;
	if (m_pInputPin->ConnectionMediaType(&mt) == S_OK)
	{
		flMediaType mediaType;
		mediaType = mt;
		*audioStereo = mediaType.getAudioStereo();
		CoTaskMemFree(mt.pbFormat);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_AudioBitCount(flInt *audioBitCount)
{
	*audioBitCount = _inputMediaType.getAudioBitCount();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_AudioBitCount(flInt audioBitCount)
{
	_inputMediaType.setAudioBitCount((flMediaType::AudioBitCountEnum)audioBitCount);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_InputAudioBitCount(flInt *audioBitCount)
{
	//CAutoLock autoLock(&_filterLock);

	*audioBitCount = MSAB_UNKNOWN;

	if (m_pInputPin == NULL)
		return NOERROR;

	AM_MEDIA_TYPE mt;
	if (m_pInputPin->ConnectionMediaType(&mt) == S_OK)
	{
		flMediaType mediaType;
		mediaType = mt;
		*audioBitCount = mediaType.getAudioBitCount();
		CoTaskMemFree(mt.pbFormat);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_Address(flChar *address)
{
	//CAutoLock autoLock(&_filterLock);

	if (address == NULL)
		return NOERROR;

	const flString addr = _fmtSender->getAddress();
	if (addr.length() == 0)
		strcpy(address, "-");
	else
		strcpy(address, addr.c_str());

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_Address(const flChar *address)
{
	CAutoLock autoLock(&_filterLock);

	if (address == NULL || strlen(address) == 0)
		return NOERROR;

	if (!strcmp(address, _fmtSender->getAddress().c_str()))
		return NOERROR;

	_fmtSender->close();
	_fmtSender->setAddress(flString(address));
	_fmtSender->open();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_Port(flUShort *port)
{
	//CAutoLock autoLock(&_filterLock);
	*port = _fmtSender->getPort();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_Port(flUShort port)
{
	CAutoLock autoLock(&_filterLock);

	if (port == _fmtSender->getPort())
		return NOERROR;

	_fmtSender->close();
	_fmtSender->setPort(port);
	_fmtSender->open();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_TimeToLive(flULong *ttl)
{
	//CAutoLock autoLock(&_filterLock);
	*ttl = _fmtSender->getTimeToLive();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_TimeToLive(flULong ttl)
{
	CAutoLock autoLock(&_filterLock);

	if (ttl == _fmtSender->getTimeToLive())
		return NOERROR;

	_fmtSender->close();
	_fmtSender->setTimeToLive(ttl);
	_fmtSender->open();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_EnableTransmit(flBool *flag)
{
	*flag = _enableTransmit;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_EnableTransmit(flBool flag)
{
	_enableTransmit = flag;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_TransmissionCount(flUInt *count)
{
	//CAutoLock autoLock(&_filterLock);
	*count = _fmtSender->getTransmissionCount();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_TransmissionCount(flUInt count)
{
	CAutoLock autoLock(&_filterLock);
	_fmtSender->setTransmissionCount(count);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_MaximumTransmissionRate(flUInt *maximumTransmissionRate)
{
	//CAutoLock autoLock(&_filterLock);
	*maximumTransmissionRate = _fmtSender->getMaximumTransmissionRate();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_MaximumTransmissionRate(flUInt maximumTransmissionRate)
{
	CAutoLock autoLock(&_filterLock);
	_fmtSender->setMaximumTransmissionRate(maximumTransmissionRate);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_SystemBufferSize(flUInt *bit)
{
	//CAutoLock autoLock(&_filterLock);
	*bit = _fmtSender->getSystemBufferSize();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_SystemBufferSize(flUInt bit)
{
	CAutoLock autoLock(&_filterLock);
	_fmtSender->setSystemBufferSize(bit);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_ClockMinimumPeriod(flFloat *millisecond)
{
	//CAutoLock autoLock(&_filterLock);
	*millisecond = _fmtSender->getClockMinimumPeriod();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_ClockMinimumPeriod(flFloat millisecond)
{
	CAutoLock autoLock(&_filterLock);
	_fmtSender->setClockMinimumPeriod(millisecond);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::compute_ClockMinimumPeriod()
{
	CAutoLock autoLock(&_filterLock);
	_fmtSender->computeClockMinimumPeriod();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_MaximumPacketSize(flUInt *byte)
{
	//CAutoLock autoLock(&_filterLock);
	*byte = _fmtSender->getPacketSize();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_MaximumPacketSize(flUInt byte)
{
	CAutoLock autoLock(&_filterLock);
	_fmtSender->setPacketSize(byte);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::get_UserData(flByte *userData, flUInt* size)
{
	//CAutoLock autoLock(&_filterLock);
	CopyMemory(userData, _userData, _userDataSize);
	*size = _userDataSize;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CMulticastSender::put_UserData(const flByte *userData, flUInt size)
{
	CAutoLock autoLock(&_filterLock);

	if (MS_MAXIMUM_USERDATA_SIZE < size)
		_userDataSize = MS_MAXIMUM_USERDATA_SIZE;
	else
		_userDataSize = size;

	CopyMemory(_userData, userData, _userDataSize);
	return NOERROR;
}
//------------------------------------------------------------------------------
/*
void
CMulticastSender::checkMediaSample2Properties(IMediaSample *pms)
{
	IMediaSample2* pms2 = NULL;
	if (pms->QueryInterface(IID_IMediaSample2, (void **)&pms2) != S_OK)
		return ;

	AM_SAMPLE2_PROPERTIES prop;
	pms2->GetProperties(sizeof(AM_SAMPLE2_PROPERTIES), (BYTE *)&prop);

	flNotify::post(flNotify::L_INFO, "---------------------------------");


	flNotify::post(flNotify::L_INFO, "SplicePoint [%d]",
		((prop.dwSampleFlags & AM_SAMPLE_SPLICEPOINT) != 0));

	flNotify::post(flNotify::L_INFO, "Preroll [%d]",
		((prop.dwSampleFlags & AM_SAMPLE_PREROLL) != 0));

	flNotify::post(flNotify::L_INFO, "Type Changed. [%d]",
		((prop.dwSampleFlags & AM_SAMPLE_TYPECHANGED) != 0));

	flNotify::post(flNotify::L_INFO, "DataDiscontinuity [%d]",
		((prop.dwSampleFlags & AM_SAMPLE_DATADISCONTINUITY) != 0));

	flNotify::post(flNotify::L_INFO, "TimeDiscontinuity [%d]",
		((prop.dwSampleFlags & AM_SAMPLE_TIMEDISCONTINUITY) != 0));

	flNotify::post(flNotify::L_INFO, "Sample Time Valid. [%d] [%I64d][%I64d]",
		((prop.dwSampleFlags & AM_SAMPLE_TIMEVALID) != 0),
		prop.tStart, prop.tStop);

	flNotify::post(flNotify::L_INFO, "Sample Stop Valid. [%d]",
		((prop.dwSampleFlags & AM_SAMPLE_STOPVALID) != 0));

	flNotify::post(flNotify::L_INFO, "Sample Flush On Pause [%d]",
		((prop.dwSampleFlags & AM_SAMPLE_FLUSH_ON_PAUSE) != 0));

	flNotify::post(flNotify::L_INFO, "Is DataStream. [%d]",
		((prop.dwSampleFlags & AM_STREAM_MEDIA) != 0));

	flNotify::post(flNotify::L_INFO, "Is ControlStream. [%d]",
		((prop.dwSampleFlags & AM_STREAM_CONTROL) != 0));

	flNotify::post(flNotify::L_INFO, "Media type = %x", prop.pMediaType);
	flNotify::post(flNotify::L_INFO, "Stream ID = %d", prop.dwStreamId);

	pms2->Release();
}
//------------------------------------------------------------------------------
*/