/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include <Stdafx.h>
#include <flNetwork/flFMTReceiver.h>
//------------------------------------------------------------------------------
#define	DEF_ADDRESS		"234.5.6.7"
#define	DEF_PORT		8000
#define	DEF_TTL			1
#define DEF_TRYUNICAST	true
#define UDPIPHEADERSIZE	((IP_HEADER_SIZE + UDP_HEADER_SIZE) / 8)
//------------------------------------------------------------------------------
flFMTReceiver::flFMTReceiver(flUInt appID)
{
	_multicastGroup		= new flMulticastGroup();

	_address			= DEF_ADDRESS;
	_port				= DEF_PORT;
	_ttl				= DEF_TTL;
	_tryUnicast			= DEF_TRYUNICAST;

	_isOpen				= false;

	_receiveThread		= NULL;
	_receiveThreadRun	= false;

	_receiveProc		= NULL;
	_receiveParam		= NULL;

	_dataDecoder		= new flFMTDataDecoder;

	_appID				= appID;

	_rejectOn			= false;

	resetMonitorVariables();
}
//------------------------------------------------------------------------------
flFMTReceiver::~flFMTReceiver()
{
	if (_isOpen)
		close();

	delete _multicastGroup;
}
//------------------------------------------------------------------------------
void
flFMTReceiver::setReceiveProc(ReceiveProc receiveProc, void *receiveParam)
{
	_receiveProc = receiveProc;
	_receiveParam = receiveParam;
}
//------------------------------------------------------------------------------
flBool
flFMTReceiver::receive(flChar*& data, flUInt& dataSize)
{
	data = NULL;
	dataSize = 0;

	if (_receiveThread != NULL || !_isOpen)
		return false;

	flChar pcmd[FMT_COMMAND_PACKET_SIZE];
	flChar pdata[FMT_DATA_PACKET_MAX_SIZE];

	flBool succeed = false;
	flInt ret = 0;

	while(!succeed && _isOpen)
	{
		// Receive Command Packet
		ret = _multicastGroup->receive(pcmd, FMT_COMMAND_PACKET_SIZE);
		if (ret == FLS_FAILED)
			continue;
		else if (ret == FLS_CANCELED)
			return false;

		if (!IS_SENDBEGIN_CMD(pcmd))
			continue;

		// Monitor
		_senderAddress = _multicastGroup->getSenderAddress();

		// Check Rejection Address
		if (_rejectOn && _rejectAddressList.find(_senderAddress))
			continue;

		// Check Application ID
		FMTSendBeginCmd *sendBeginCmd = (FMTSendBeginCmd *)pcmd;
		if (_appID != FMT_APPID_ANY && sendBeginCmd->_appId != _appID)
			continue;

		// Monitor
		_bytesRate.nextTerm();
		_datasRate.nextTerm();
		_bytesRate.record(UDPIPHEADERSIZE + FMT_COMMAND_PACKET_SIZE);

		// Receive Data Packet
		_packetSize = sendBeginCmd->_dataPacketSize;
		flUInt pdataId = sendBeginCmd->_dataId;
		flUInt pdataSize = sendBeginCmd->_dataSize;
		flUInt pnumPackets = sendBeginCmd->_numPackets;

		_dataDecoder->reset(pdataId, pdataSize, pnumPackets);

		while(!succeed && _isOpen)
		{
			ret = _multicastGroup->receive(pdata, _packetSize);
			if (ret == FLS_FAILED)
				break;
			else if (ret == FLS_CANCELED)
				return false;

			if (!IS_DATA(pdata))
				break;

			_dataDecoder->setPacket(pdata);

			// Monitor
			_numReceivedPackets++;
			_bytesRate.record(UDPIPHEADERSIZE + _packetSize);

			// Check Complete
			if (!_dataDecoder->completeData())
			{
				// Check Terminate Packet
				if (((FMTDataPacket *)pdata)->_end)
					break;
				else
					continue;
			}

			_dataDecoder->getData(data, dataSize);
			succeed = true;

			// Monitor
			_numReceivedDatas++;
			_datasRate.record(1);

			break;
		}

		if (!_dataDecoder->completeData())
		{
			_numDropDatas++;
			_numDropPackets += (pnumPackets - _dataDecoder->getNumSetPacket());
		}
	};

	return succeed;
}
//------------------------------------------------------------------------------
void
flFMTReceiver::setAddress(const flString& address)
{
	_address = address;
}
//------------------------------------------------------------------------------
const flString&
flFMTReceiver::getAddress() const
{
	return _address;
}
//------------------------------------------------------------------------------
void
flFMTReceiver::setPort(flUShort port)
{
	_port = port;
}
//------------------------------------------------------------------------------
flUShort
flFMTReceiver::getPort() const
{
	return _port;
}
//------------------------------------------------------------------------------
void
flFMTReceiver::setTimeToLive(flULong ttl)
{
	_ttl = ttl;
}
//------------------------------------------------------------------------------
flULong
flFMTReceiver::getTimeToLive() const
{
	return _ttl;
}
//------------------------------------------------------------------------------
void
flFMTReceiver::setTryUnicast(flBool flag)
{
	_tryUnicast = flag;
}
//------------------------------------------------------------------------------
flBool
flFMTReceiver::getTryUnicast() const
{
	return _tryUnicast;
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::getNumRejectionAddress() const
{
	return _rejectAddressList.getSize();
}
//------------------------------------------------------------------------------
const flString&
flFMTReceiver::getRejectionAddress(const flUInt index)
{
	return _rejectAddressList.getValue(index);
}
//------------------------------------------------------------------------------
void
flFMTReceiver::addRejectionAddress(const flString& address)
{
	if (address.length() == 0)
		return ;
	_rejectAddressList.insert(address, address);
	_rejectOn = (_rejectAddressList.getSize() != 0);
}
//------------------------------------------------------------------------------
void
flFMTReceiver::removeRejectionAddress(flUInt index)
{
	const flString& address =  _rejectAddressList.getValue(index);
	_rejectAddressList.remove(address);
	_rejectOn = (_rejectAddressList.getSize() != 0);
}
//------------------------------------------------------------------------------
void
flFMTReceiver::removeAllRejectionAddress()
{
	_rejectAddressList.clear();
	_rejectOn = (_rejectAddressList.getSize() != 0);
}
//------------------------------------------------------------------------------
flBool
flFMTReceiver::isOpen() const
{
	return _isOpen;
}
//------------------------------------------------------------------------------
flBool
flFMTReceiver::open()
{
	if (_isOpen)
		return false;

	// Create Multicast group
	_isOpen = _multicastGroup->initialize(_address,
										  _port,
										  _ttl,
										  true,
										  8192,
										  flMulticastGroup::MPO_RECEIVER_ONLY,
										  _tryUnicast);

	// Create Receive Data Thread
	if (_isOpen)
	{
		if (_receiveProc != NULL)
		{
			_receiveThreadRun = true;
			_receiveThread = new flThread(receiveLoopProc, this);
		}
		else
		{
			_receiveThread = NULL;
		}
	}

	resetMonitorVariables();

	Sleep(20);

	return _isOpen;
}
//------------------------------------------------------------------------------
flBool
flFMTReceiver::close()
{
	if (!_isOpen)
		return false;

	resetMonitorVariables();

	// Break send() and receive()
	_multicastGroup->breakSend();
	_multicastGroup->breakReceive();

	// Exit Receive Thread
	if (_receiveThread != NULL)
	{
		_receiveThreadRun = false;
#ifdef USE_OVERLAPPED
		//_receiveThread->waitFor();
		_receiveThread->waitFor(1000);
#endif
		delete _receiveThread;
		_receiveThread = NULL;
	}

	// Shutdown Multicast group
	_multicastGroup->shutdown();

	_isOpen = false;

	return true;
}
//------------------------------------------------------------------------------
// Monitor
flString
flFMTReceiver::getSenderAddress()
{
	return _senderAddress;
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::getPacketSize() const
{
	return _packetSize;
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::getReceiveBitRate()
{
	return flUInt(_bytesRate.getRate() * 8.0f);
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::getReceiveDataRate()
{
	return flUInt(_datasRate.getRate());
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::getNumReceivedDatas() const
{
	return _numReceivedDatas;
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::getNumReceivedPackets() const
{
	return _numReceivedPackets;
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::getNumDropDatas() const
{
	return _numDropDatas;
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::getNumDropPackets() const
{
	return _numDropPackets;
}
//------------------------------------------------------------------------------
void
flFMTReceiver::resetMonitorVariables()
{
	_senderAddress		= "";
	_packetSize			= 0;
	_bytesRate.init();
	_datasRate.init();
	_numReceivedDatas	= 0;
	_numReceivedPackets	= 0;
	_numDropDatas		= 0;
	_numDropPackets		= 0;
}
//------------------------------------------------------------------------------
flUInt
flFMTReceiver::receiveLoop()
{
	flChar cmd[FMT_COMMAND_PACKET_SIZE];
	flChar data[FMT_DATA_PACKET_MAX_SIZE];

	flInt ret;

	while(_receiveThreadRun)
	{
		// Receive Command Packet
		ret = _multicastGroup->receive(cmd, FMT_COMMAND_PACKET_SIZE);
		if (ret == FLS_FAILED)
			continue;
		else if (ret == FLS_CANCELED)
			return 0;

		if (!IS_SENDBEGIN_CMD(cmd))
			continue;

		// Monitor
		_senderAddress = _multicastGroup->getSenderAddress();

		// Check Rejection Address
		if (_rejectOn && _rejectAddressList.find(_senderAddress))
			continue;

		// Check Application ID
		FMTSendBeginCmd *sendBeginCmd = (FMTSendBeginCmd *)cmd;
		if (_appID != FMT_APPID_ANY && sendBeginCmd->_appId != _appID)
			continue;

		// Monitor
		_bytesRate.nextTerm();
		_datasRate.nextTerm();
		_bytesRate.record(UDPIPHEADERSIZE + FMT_COMMAND_PACKET_SIZE);

		// Receive Data Packet
		_packetSize = sendBeginCmd->_dataPacketSize;
		flUInt dataId = sendBeginCmd->_dataId;
		flUInt dataSize = sendBeginCmd->_dataSize;
		flUInt numPackets = sendBeginCmd->_numPackets;

		_dataDecoder->reset(dataId, dataSize, numPackets);

		while(_receiveThreadRun)
		{
			ret = _multicastGroup->receive(data, _packetSize);
			if (ret == FLS_FAILED)
				break;
			else if (ret == FLS_CANCELED)
				return 0;

			if (!IS_DATA(data))
				break;

			_dataDecoder->setPacket(data);

			// Monitor
			_numReceivedPackets++;
			_bytesRate.record(UDPIPHEADERSIZE + _packetSize);

			// Check Complete
			if (!_dataDecoder->completeData())
			{
				// Check Terminate Packet
				if (((FMTDataPacket *)data)->_end)
					break;
				else
					continue;
			}

			// Notify Receive
			if (_receiveProc != NULL)
			{
				flChar* cdata;
				flUInt cdataSize;
				_dataDecoder->getData(cdata, cdataSize);
				_receiveProc(cdata, cdataSize, _receiveParam);
			}

			// Monitor
			_numReceivedDatas++;
			_datasRate.record(1);

			Sleep(0);
			break;
		}

		if (!_dataDecoder->completeData())
		{
			_numDropDatas++;
			_numDropPackets += (numPackets - _dataDecoder->getNumSetPacket());
		}
	};

	return 0;
}
//------------------------------------------------------------------------------
flUInt __stdcall
flFMTReceiver::receiveLoopProc(void* data)
{
	flFMTReceiver *receiver = (flFMTReceiver *)data;
	return receiver->receiveLoop();
}
//------------------------------------------------------------------------------
