/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "Stdafx.h"
#include <flNetwork/flFMTSender.h>
//------------------------------------------------------------------------------
#define DEF_ADDRESS				"234.5.6.7"
#define DEF_PORT				8000
#define DEF_TTL					1
#define DEF_TRYUNICAST			true
#define DEF_TRANSMISSIONCOUNT	1
//------------------------------------------------------------------------------
flFMTSender::flFMTSender(flUInt appID)
{
	_sendBeginCmd._appId	= appID;

	_multicastGroup			= new flMulticastGroup();

	_address				= DEF_ADDRESS;
	_port					= DEF_PORT;
	_ttl					= DEF_TTL;
	_tryUnicast				= DEF_TRYUNICAST;

	_isOpen					= false;

	_transmissionCount		= DEF_TRANSMISSIONCOUNT;
	_timeKeeper.computeClockMinimumPeriod();

	_dataEncoder			= new flFMTDataEncoder();

	resetMonitorVariables();
}
//------------------------------------------------------------------------------
flFMTSender::~flFMTSender()
{
	delete _dataEncoder;

	if (_isOpen)
		close();

	delete _multicastGroup;
}
//------------------------------------------------------------------------------
void
flFMTSender::setAddress(const flString& address)
{
	_address = address;
}
//------------------------------------------------------------------------------
const flString&
flFMTSender::getAddress() const
{
	return _address;
}
//------------------------------------------------------------------------------
void
flFMTSender::setPort(flUShort port)
{
	_port = port;
}
//------------------------------------------------------------------------------
flUShort
flFMTSender::getPort() const
{
	return _port;
}
//------------------------------------------------------------------------------
void
flFMTSender::setTimeToLive(flULong ttl)
{
	_ttl = ttl;
}
//------------------------------------------------------------------------------
flULong
flFMTSender::getTimeToLive() const
{
	return _ttl;
}
//------------------------------------------------------------------------------
void
flFMTSender::setTryUnicast(flBool flag)
{
	_tryUnicast = flag;
}
//------------------------------------------------------------------------------
flBool
flFMTSender::getTryUnicast() const
{
	return _tryUnicast;
}
//------------------------------------------------------------------------------
flBool
flFMTSender::isOpen() const
{
	return _isOpen;
}
//------------------------------------------------------------------------------
flBool
flFMTSender::open()
{
	if (_isOpen)
		return false;

	// Create Multicast group
	_isOpen = _multicastGroup->initialize(_address,
										  _port,
										  _ttl,
										  false,
										  8192,
										  flMulticastGroup::MPO_SENDER_ONLY,
										  _tryUnicast);

	resetMonitorVariables();

	return _isOpen;
}
//------------------------------------------------------------------------------
flBool
flFMTSender::close()
{
	if (!_isOpen)
		return false;

	resetMonitorVariables();

	// Break send() and receive()
	_multicastGroup->breakSend();
	_multicastGroup->breakReceive();

	// Shutdown Multicast group 
	_multicastGroup->shutdown();

	_isOpen = false;

	return true;
}
//------------------------------------------------------------------------------
void
flFMTSender::setTransmissionCount(flUInt count)
{
	if (count < 1)
		count = 1;
	_transmissionCount = count;
}
//------------------------------------------------------------------------------
flUInt
flFMTSender::getTransmissionCount() const
{
	return _transmissionCount;
}
//------------------------------------------------------------------------------
void
flFMTSender::setMaximumTransmissionRate(flUInt bitPerSec)
{
	_timeKeeper.setMaximumTransmissionRate(bitPerSec);
}
//------------------------------------------------------------------------------
flUInt
flFMTSender::getMaximumTransmissionRate() const
{
	return _timeKeeper.getMaximumTransmissionRate();
}
//------------------------------------------------------------------------------
void
flFMTSender::setSystemBufferSize(flUInt bit)
{
	_timeKeeper.setSystemBufferSize(bit);
}
//------------------------------------------------------------------------------
flUInt
flFMTSender::getSystemBufferSize() const
{
	return _timeKeeper.getSystemBufferSize();
}
//------------------------------------------------------------------------------
void
flFMTSender::setClockMinimumPeriod(flFloat second)
{
	_timeKeeper.setClockMinimumPeriod((flDouble)second);
}
//------------------------------------------------------------------------------
flFloat
flFMTSender::getClockMinimumPeriod() const
{
	return (flFloat)_timeKeeper.getClockMinimumPeriod();
}
//------------------------------------------------------------------------------
void
flFMTSender::computeClockMinimumPeriod()
{
	_timeKeeper.computeClockMinimumPeriod();
}
//------------------------------------------------------------------------------
void
flFMTSender::setPacketSize(flUInt byte)
{
	_dataEncoder->setPacketSize(byte);
}
//------------------------------------------------------------------------------
flUInt
flFMTSender::getPacketSize() const
{
	return _dataEncoder->getPacketSize();
}
//------------------------------------------------------------------------------
// Monitor
//------------------------------------------------------------------------------
flUInt
flFMTSender::getTransmissionBitRate()
{
	return flUInt(_bytesRate.getRate() * 8.0f);
}
//------------------------------------------------------------------------------
flUInt
flFMTSender::getTransmissionDataRate()
{
	return flUInt(_datasRate.getRate());
}
//------------------------------------------------------------------------------
flUInt
flFMTSender::getNumSendDatas() const
{
	return _numSendDatas;
}
//------------------------------------------------------------------------------
flUInt
flFMTSender::getNumSendPackets() const
{
	return _numSendPackets;
}
//------------------------------------------------------------------------------
void
flFMTSender::resetMonitorVariables()
{
	_bytesRate.init();
	_datasRate.init();
	_numSendDatas		= 0;
	_numSendPackets		= 0;
}
//------------------------------------------------------------------------------
flBool
flFMTSender::send(const flChar* data, flUInt dataSize)
{
	const flUInt UDPIPHeaderSize = (IP_HEADER_SIZE + UDP_HEADER_SIZE) / 8;

	if (!_isOpen)
		return false;

	flChar cmd[FMT_COMMAND_PACKET_SIZE];
	flUInt dataPacketSize = _dataEncoder->getPacketSize();

	_timeKeeper.start();
	_bytesRate.nextTerm();
	_datasRate.nextTerm();

	// Setup Data Packets
	_dataEncoder->setData(data, dataSize);

	// Send Begin
	_sendBeginCmd._dataId = _dataEncoder->getId();
	_sendBeginCmd._dataPacketSize = dataPacketSize;
	_sendBeginCmd._dataSize = dataSize;
	_sendBeginCmd._numPackets = _dataEncoder->getNumPackets();
	memcpy(cmd, &_sendBeginCmd, sizeof(FMTSendBeginCmd));
	_multicastGroup->send(cmd, FMT_COMMAND_PACKET_SIZE);

	_timeKeeper.adjustRate(FMT_COMMAND_PACKET_SIZE);

	flUInt numPackets = _dataEncoder->getNumPackets();

	// Send multiple
	for(flUInt j = 0; j < _transmissionCount; j++)
	{
		// Send Data Packets
		for(flUInt i = 0; i < numPackets; i++)
		{
			flChar* dataPacket = _dataEncoder->getPacket(i);

			// Stamp Terminate
			if (j == _transmissionCount - 1 && i == numPackets - 1)
			{
				FMTDataPacket *header = (FMTDataPacket *)dataPacket;
				header->_end = true;
			}

			_multicastGroup->send(dataPacket, dataPacketSize);

			_timeKeeper.adjustRate(dataPacketSize);
		}
	}

	// Monitor
	_bytesRate.record((UDPIPHeaderSize + FMT_COMMAND_PACKET_SIZE +
					   _transmissionCount * numPackets * (UDPIPHeaderSize + dataPacketSize)));
	_datasRate.record(1);
	_numSendPackets += _transmissionCount * numPackets;
	_numSendDatas ++;

	return true;
}
//------------------------------------------------------------------------------
