/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include "Stdafx.h"
#include <flNetwork/flMulticastGroup.h>
//------------------------------------------------------------------------------

LPSTR DecodeError(int ErrorCode)
{
    static char Message[1024];

    // If this program was multi-threaded, we'd want to use
    // FORMAT_MESSAGE_ALLOCATE_BUFFER instead of a static buffer here.
    // (And of course, free the buffer when we were done with it)

    FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS |
                  FORMAT_MESSAGE_MAX_WIDTH_MASK, NULL, ErrorCode,
                  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                  (LPSTR)Message, 1024, NULL);
    return Message;
}

//------------------------------------------------------------------------------
flMulticastGroup::flMulticastGroup()
{
	_isInit					= false;
	_socket					= 0;
}
//------------------------------------------------------------------------------
flMulticastGroup::~flMulticastGroup()
{
	if (_isInit)
		shutdown();
}
//------------------------------------------------------------------------------
flBool
flMulticastGroup::initialize(const flString& address,
						flUShort port,
						flULong ttl,
						flBool loopBackOn,
						flUInt socketBufSize,
						MultiPointOptEnum multiPointOpt,
						flBool tryUnicast)
{
	memset(&_sockAddr, 0, sizeof(SOCKADDR_STORAGE));
	memset(&_destSockAddr, 0, sizeof(SOCKADDR_STORAGE));

	_sockAddrLen			= 0;
	_destSockAddrLen		= 0;

	_destSockAddrIsValid	= false;
	_sendEvents[0]			= WSACreateEvent();
	_sendEvents[1]			= _breakSendTimer;
	_receiveEvents[0]		= WSACreateEvent();
	_receiveEvents[1]		= _breakReceiveTimer;


	// Try to create Multicast Socket
	_isInit = _initializeMulticast(address,
								port,
								ttl,
								loopBackOn,
								socketBufSize,
								multiPointOpt);

	// Try to create Unicast Socket
	if (!_isInit && tryUnicast)
		_isInit = _initializeUnicast(address,
								port,
								ttl,
								socketBufSize);

	return _isInit;
}
//------------------------------------------------------------------------------
flBool
flMulticastGroup::shutdown()
{
	_isInit = false;

	_breakSendTimer.deactive();
	_breakReceiveTimer.deactive();

	if (_sendEvents[0] != WSA_INVALID_EVENT)
		WSACloseEvent(_sendEvents[0]);
	if (_receiveEvents[0] != WSA_INVALID_EVENT)
		WSACloseEvent(_receiveEvents[0]);

	closesocket(_socket);
	_socket = 0;

	return true;
}
//------------------------------------------------------------------------------
flBool
flMulticastGroup::isInitialized() const
{
	return _isInit;
}
//------------------------------------------------------------------------------
flInt
flMulticastGroup::send(const flChar* buffer, flUInt bufferSize)
{
	if (!_isInit)
		return FLS_FAILED;
	return _send((SOCKADDR *)&_sockAddr, (flInt)_sockAddrLen, buffer, bufferSize);
}
//------------------------------------------------------------------------------
flInt
flMulticastGroup::sendResponse(const flChar* buffer, flUInt bufferSize)
{
	if (!_isInit || !_destSockAddrIsValid)
		return FLS_FAILED;

	return _send((SOCKADDR *)&_destSockAddr, (flInt)_destSockAddrLen, buffer, bufferSize);
}
//------------------------------------------------------------------------------
flInt
flMulticastGroup::receive(flChar* buffer, flUInt bufferSize)
{
	if (!_isInit)
		return FLS_FAILED;
	return _receive((SOCKADDR *)&_destSockAddr, (flInt &)_destSockAddrLen, buffer, bufferSize);
}
//------------------------------------------------------------------------------
flBool
flMulticastGroup::breakSend(flLong millisecond)
{
	return _breakSendTimer.active(-10000 * millisecond, 0);
}
//------------------------------------------------------------------------------
flBool
flMulticastGroup::breakReceive(flLong millisecond)
{
	return _breakReceiveTimer.active(-10000 * millisecond, 0);
}
//------------------------------------------------------------------------------
flString
flMulticastGroup::getSenderAddress() const
{
	if (!_isInit || !_destSockAddrIsValid)
		return flString("");

	flChar host[256];
	host[0] = '\0';

	if (getnameinfo((const sockaddr *)&_destSockAddr, _destSockAddrLen,
							host, 256, NULL, 0, NI_NUMERICHOST) != 0)
		return flString("");
	return flString(host);
}
//------------------------------------------------------------------------------
flBool
flMulticastGroup::_initializeMulticast(const flString& address,
									   flUShort port,
									   flULong ttl,
									   flBool loopBackOn,
									   flUInt socketBufSize,
									   MultiPointOptEnum multiPointOpt)
{
	flInt ret;
	flBool flag;
	flChar str[256];

	DWORD numReturn;
	const flUInt BUFFER_SIZE = 2048;
	flChar buffer[BUFFER_SIZE];


	// Enumeration Address info
	addrinfo hints, *res;
	memset(&hints, 0, sizeof(addrinfo));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_NUMERICHOST;

	sprintf(str, "%d", port);
	ret = getaddrinfo(address.c_str(), str, &hints, &res);

	if (ret != 0 || sizeof(_sockAddr) < res->ai_addrlen)
		return false;

	_sockAddrLen = res->ai_addrlen;
	_destSockAddrLen = res->ai_addrlen;

	memcpy(&_sockAddr, res->ai_addr, _sockAddrLen);
	freeaddrinfo(res);


	// Create Socket
	_socket = WSASocket(
				_sockAddr.ss_family,
				SOCK_DGRAM,
				0,
				NULL,
				0,
#ifdef USE_OVERLAPPED
				WSA_FLAG_MULTIPOINT_C_LEAF | WSA_FLAG_MULTIPOINT_D_LEAF | WSA_FLAG_OVERLAPPED);
#else
				WSA_FLAG_MULTIPOINT_C_LEAF | WSA_FLAG_MULTIPOINT_D_LEAF);
#endif

	if (_socket == INVALID_SOCKET)
		return false;


	// Set Reuse
	flag = true;
	ret = setsockopt(
				_socket,
				SOL_SOCKET,
				SO_REUSEADDR,
				(const flChar *)&flag,
				sizeof(flBool));

	if (ret == SOCKET_ERROR)
	{
		closesocket(_socket);
		return false;
	}


	// Set Socket Buffer size
	if (multiPointOpt == MPO_BOTH || multiPointOpt == MPO_RECEIVER_ONLY)
	{
		ret = setsockopt(
				_socket,
				SOL_SOCKET,
				SO_RCVBUF,
				(const flChar *)&socketBufSize,
				sizeof(flUInt));
	}
	if (multiPointOpt == MPO_BOTH || multiPointOpt == MPO_SENDER_ONLY)
	{
		ret = setsockopt(
				_socket,
				SOL_SOCKET,
				SO_SNDBUF,
				(const flChar *)&socketBufSize,
				sizeof(flUInt));
	}


	// Bind Socket
	SOCKADDR_STORAGE ownSockAddr;

	memset(&hints, 0, sizeof(addrinfo));
	hints.ai_family = _sockAddr.ss_family;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_PASSIVE;

	sprintf(str, "%d", port);
	ret = getaddrinfo(NULL, str, &hints, &res);

	if (ret != 0 || sizeof(ownSockAddr) < res->ai_addrlen)
	{
		closesocket(_socket);
		return false;
	}

	assert(_sockAddrLen == res->ai_addrlen);

	memcpy(&ownSockAddr, res->ai_addr, _sockAddrLen);
	freeaddrinfo(res);

	ret = bind(_socket, (const sockaddr *)&ownSockAddr, _sockAddrLen);
	if (ret == SOCKET_ERROR)
	{
		closesocket(_socket);
		return false;
	}


	// Set IP Time to Live
	ret = WSAIoctl(_socket,
				SIO_MULTICAST_SCOPE,
				&ttl,
				sizeof(flULong),
				buffer,
				BUFFER_SIZE,
				&numReturn,
				NULL,
				NULL);

	if (ret == SOCKET_ERROR)
	{
		closesocket(_socket);
		return false;
	}

	
	// Set Loopback enable
	flInt lbo = (loopBackOn ? 1 : 0);

//////	Depend on Address family ( This is Temporary. )
//////
	if (_sockAddr.ss_family == AF_INET6)
		lbo = 1;
//////
//////

	ret = WSAIoctl(_socket,
				SIO_MULTIPOINT_LOOPBACK,
				&lbo,
				sizeof(flInt),
				buffer,
				BUFFER_SIZE,
				&numReturn,
				NULL,
				NULL);
								
	if (ret == SOCKET_ERROR)
	{
		closesocket(_socket);
		return false;
	}


	// Join Leaf to MultiPoint session
	DWORD joinFlag;
	switch(multiPointOpt)
	{
	case MPO_SENDER_ONLY:
		joinFlag = JL_SENDER_ONLY;
		break;
	case MPO_RECEIVER_ONLY:
		joinFlag = JL_RECEIVER_ONLY;
		break;
	case MPO_BOTH:
		joinFlag = JL_BOTH;
		break;
	}

	SOCKET newSocket = WSAJoinLeaf(_socket,
				(const struct sockaddr FAR *)&_sockAddr,
				_sockAddrLen,
				NULL,
				NULL,
				NULL,
				NULL,
				joinFlag);
		
	if (newSocket == INVALID_SOCKET)
	{
		closesocket(_socket);
		return false;
	}

	return true;
}
//------------------------------------------------------------------------------
flBool
flMulticastGroup::_initializeUnicast(const flString& address,
									 flUShort port,
									 flULong ttl,
									 flUInt socketBufSize)
{
	flInt ret;
	flBool flag;
	flChar str[256];

	DWORD numReturn;
	const flUInt BUFFER_SIZE = 2048;
	flChar buffer[BUFFER_SIZE];


	// Enumeration Address info
	addrinfo hints, *res;
	memset(&hints, 0, sizeof(addrinfo));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_NUMERICHOST;

	sprintf(str, "%d", port);
	ret = getaddrinfo(address.c_str(), str, &hints, &res);

	if (ret != 0 || sizeof(_sockAddr) < res->ai_addrlen)
		return false;

	_sockAddrLen = res->ai_addrlen;
	_destSockAddrLen = res->ai_addrlen;

	memcpy(&_sockAddr, res->ai_addr, _sockAddrLen);
	freeaddrinfo(res);


	// Create Socket
#ifdef USE_OVERLAPPED
	_socket = WSASocket(
				_sockAddr.ss_family,
				SOCK_DGRAM,
				0,
				NULL,
				0,
				WSA_FLAG_OVERLAPPED);
#else
	_socket = socket(AF_INET, SOCK_DGRAM, 0);
#endif

	if (_socket == INVALID_SOCKET)
		return false;

#if 1
	// Set Reuse
	flag = true;
	ret = setsockopt(
				_socket,
				SOL_SOCKET,
				SO_REUSEADDR,
				(const flChar *)&flag,
				sizeof(flBool));

	if (ret == SOCKET_ERROR)
	{
		closesocket(_socket);
		return false;
	}
#endif

	// Set Socket Buffer size
	ret = setsockopt(
			_socket,
			SOL_SOCKET,
			SO_RCVBUF,
			(const flChar *)&socketBufSize,
			sizeof(flUInt));
	ret = setsockopt(
			_socket,
			SOL_SOCKET,
			SO_SNDBUF,
			(const flChar *)&socketBufSize,
			sizeof(flUInt));


	// Bind Socket
	SOCKADDR_STORAGE ownSockAddr;

	memset(&hints, 0, sizeof(addrinfo));
	hints.ai_family = _sockAddr.ss_family;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_PASSIVE;

	sprintf(str, "%d", port);
	ret = getaddrinfo(NULL, str, &hints, &res);

	if (ret != 0 || sizeof(ownSockAddr) < res->ai_addrlen)
	{
		closesocket(_socket);
		return false;
	}

	assert(_sockAddrLen == res->ai_addrlen);

	memcpy(&ownSockAddr, res->ai_addr, _sockAddrLen);
	freeaddrinfo(res);

	ret = bind(_socket, (const sockaddr *)&ownSockAddr, _sockAddrLen);
	if (ret == SOCKET_ERROR)
	{
		closesocket(_socket);
		return false;
	}


	// Set IP Time to Live
	ret = WSAIoctl(_socket,
				SIO_MULTICAST_SCOPE,
				&ttl,
				sizeof(flULong),
				buffer,
				BUFFER_SIZE,
				&numReturn,
				NULL,
				NULL);

	if (ret == SOCKET_ERROR)
	{
		closesocket(_socket);
		return false;
	}

	return true;
}
//------------------------------------------------------------------------------
flInt
flMulticastGroup::_send(SOCKADDR* sockAddr, flInt sockAddrLen, const flChar* buffer, flUInt bufferSize)
{
#ifdef USE_OVERLAPPED
	flInt result;

	WSABUF wsabuf;

	WSAOVERLAPPED overlapped;
	overlapped.hEvent = _sendEvents[0];

	DWORD flags = 0;

	flUInt size = bufferSize, numsent;

WRITE_LOOP:

	wsabuf.len = size;
	wsabuf.buf = (flChar*)buffer;
	result = WSASendTo(_socket, &wsabuf, 1, (DWORD*)&numsent, flags,
		sockAddr, sockAddrLen, &overlapped, NULL);

	if (result == 0)
	{
		if (numsent < 1)
			return FLS_FAILED;
		size -= numsent;
		buffer  += numsent;
		if (size == 0)
			return FLS_SUCCEEDED;
		goto WRITE_LOOP;
	}
	
	if ((result = WSAGetLastError()) == WSA_IO_PENDING)
	{
		DWORD eventRet = WSAWaitForMultipleEvents(2, _sendEvents, FALSE, INFINITE, FALSE);
		if (eventRet == WSA_WAIT_EVENT_0)
		{
			if (WSAGetOverlappedResult(_socket, &overlapped,
				(DWORD*)&numsent, FALSE, &flags))
			{
				if (numsent < 1)
					return FLS_FAILED;
				size -= numsent;
				buffer  += numsent;
				if (size == 0)
					return FLS_SUCCEEDED;
				goto WRITE_LOOP;
			}
		}
		else
		{
			return FLS_CANCELED;
		}
	}

	return FLS_FAILED;

#else
	flInt numsent = 0, temp;

	while (numsent < (flInt)bufferSize)
	{
		temp = sendto(_socket, &buffer[numsent], bufferSize-numsent, 0,
			sockAddr, sockAddrLen);
		if (temp == SOCKET_ERROR)
			return 0;
		numsent += temp;
	}

	return numsent;
#endif
}
//------------------------------------------------------------------------------
flInt
flMulticastGroup::_receive(SOCKADDR* sockAddr, flInt& sockAddrLen, flChar* buffer, flUInt bufferSize)
{
#ifdef USE_OVERLAPPED
	flInt result;

	WSABUF wsabuf;

	WSAOVERLAPPED overlapped;
	overlapped.hEvent = _receiveEvents[0];

	DWORD flags = 0;
	
	flUInt size = bufferSize, numrecv;

READ_LOOP:

	wsabuf.len = size;
	wsabuf.buf = buffer;
	result = WSARecvFrom(_socket, &wsabuf, 1, (DWORD*)&numrecv, &flags,
		sockAddr, &sockAddrLen, &overlapped, NULL);

	if (result == 0)
	{
		if (numrecv < 1)
			return FLS_FAILED;
		size -= numrecv;
		buffer  += numrecv;
		if (size == 0)
		{
			_destSockAddrIsValid = true;
			return FLS_SUCCEEDED;
		}

		goto READ_LOOP;
	}

	if ((result = WSAGetLastError()) == WSA_IO_PENDING)
	{
		DWORD eventRet = WSAWaitForMultipleEvents(2, _receiveEvents, FALSE,	INFINITE, FALSE);
		if (eventRet == WSA_WAIT_EVENT_0)
		{
			if (WSAGetOverlappedResult(_socket, &overlapped,
				(DWORD*)&numrecv, FALSE, &flags))
			{
				if (numrecv < 1)
					return FLS_FAILED;
				size -= numrecv;
				buffer  += numrecv;
				if (size == 0)
				{
					_destSockAddrIsValid = true;
					return FLS_SUCCEEDED;
				}

				goto READ_LOOP;
			}
		}
		else
		{
			return FLS_CANCELED;
		}
	}

	_destSockAddrIsValid = false;

	return FLS_FAILED;

#else
	flUInt numrecv = 0, temp;
	
	while (numrecv < bufferSize)
	{
		temp = recvfrom(_socket, &buffer[numrecv], bufferSize - numrecv, 0,
						sockAddr, &sockAddrLen);
		if (temp == SOCKET_ERROR)
			return 0;

		numrecv += temp;
	}

	return numrecv;
#endif
}
//------------------------------------------------------------------------------
