/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/

#include <windows.h>
#include <streams.h>
#include <initguid.h>
#include <olectl.h>
#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif

#include <stdio.h>
#include <limits.h>
#include <math.h>

#include <flFilterCommon/flPointerInfo.h>
#include <flTypes/flUIDs.h>
#include <flTypes/INonLinearZoom.h>

#include "transfrmn3.h"

#include "nonlinearzoomprop.h"
#include "nonlinearzoom.h"

#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue

// setup data

const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
    &MEDIATYPE_Video,			// Major type
    &MEDIASUBTYPE_NULL			// Minor type
};

const AMOVIESETUP_MEDIATYPE sudPinTypesI =
{
    &MEDIATYPE_Stream,			// Major type
    &MEDIASUBTYPE_PointerInfo	// Minor type
};

const AMOVIESETUP_PIN psudPins[] =
{
	{
		L"Input",           // String pin name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Output",          // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Output",          // String pin name
		FALSE,              // Is it rendered
		TRUE,               // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Input",           // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Input Pointer",   // Pins string name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Are we allowed none
		FALSE,              // And allowed many
		&CLSID_NULL,        // Connects to filter
		NULL,               // Connects to pin
		1,                  // Number of types
		&sudPinTypesI       // Pin information
	}
};


const AMOVIESETUP_FILTER sudNonLinearZoom =
{
    &CLSID_NonLinearZoom,   // Filter CLSID
    L"NIME Non-linear Zooming Filter",     // Filter name
    MERIT_DO_NOT_USE,       // Its merit
    3,                      // Number of pins
    psudPins                // Pin details
};


// List of class IDs and creator functions for the class factory. This
// provides the link between the OLE entry point in the DLL and an object
// being created. The class factory will call the static CreateInstance

CFactoryTemplate g_Templates[2] = {

    { L"NonLinearZoom"
    , &CLSID_NonLinearZoom
    , CNonLinearZoom::CreateInstance
    , NULL
    , &sudNonLinearZoom }
  ,
    { L"NonLinearZoomProp"
    , &CLSID_NonLinearZoomPropertyPage
    , CNonLinearZoomProperties::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);



#define DEF_OUTPUT_VIDEORESOLUTION		VR_SQCIF
#define DEF_MASK						M_RECTANGLE
#define DEF_MASKPOSX					0
#define DEF_MASKPOSY					0
#define DEF_MASKSIZEWIDTH				128
#define DEF_MASKSIZEHEIGHT				96
#define DEF_POINTERINDISABLE			false


#define M_PI							3.14159265358979323846f

#define EMIIDX(_y, _x, _d)				(((_x) + (_y) * (_emW)) * 3 + (_d))


//
// Constructor
//
CNonLinearZoom::CNonLinearZoom(TCHAR *tszName,LPUNKNOWN punk,HRESULT *phr) :
CTransformN3Filter(tszName, punk, CLSID_NonLinearZoom),
CPersistStream(punk, phr),
_pPointer(NULL),
_outVR(DEF_OUTPUT_VIDEORESOLUTION),
_mask(DEF_MASK),
_maskPosX(DEF_MASKPOSX),
_maskPosY(DEF_MASKPOSY),
_maskSizeWidth(DEF_MASKSIZEWIDTH),
_maskSizeHeight(DEF_MASKSIZEHEIGHT),
_pointerInDisable(DEF_POINTERINDISABLE),
_baseBufSize(0),
_baseBuf(NULL),
_maskSizeIsChanged(true),
_elipMaskInfoSize(0),
_elipMaskInfo(NULL),
_emW(0),
_emH(0),
_emHW(0),
_emHH(0)
{
    ASSERT(tszName);
    ASSERT(phr);

} // NonLinearZoom

CNonLinearZoom::~CNonLinearZoom()
{
	delete _pPointer;
	delete _baseBuf;

	delete [] _elipMaskInfo;
}

//
// CreateInstance
//
// Provide the way for COM to create a CNonLinearZoom object
//
CUnknown * WINAPI CNonLinearZoom::CreateInstance(LPUNKNOWN punk, HRESULT *phr) {

    CNonLinearZoom *pNewObject = new CNonLinearZoom(NAME("NonLinearZoom"), punk, phr);
    if (pNewObject == NULL) {
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;

} // CreateInstance


//
// NonDelegatingQueryInterface
//
// Reveals INonLinearZoom and ISpecifyPropertyPages
//
STDMETHODIMP CNonLinearZoom::NonDelegatingQueryInterface(REFIID riid, void **ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_INonLinearZoom) {
        return GetInterface((INonLinearZoom *) this, ppv);
	} else if (riid == IID_IPersistStream) {
		return GetInterface((IPersistStream *) this, ppv);
    } else if (riid == IID_ISpecifyPropertyPages) {
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    } else {
        return CTransformN3Filter::NonDelegatingQueryInterface(riid, ppv);
    }

} // NonDelegatingQueryInterface

int
CNonLinearZoom::GetPinCount()
{
	return 3;
}

CBasePin*
CNonLinearZoom::GetPin(int n)
{
	HRESULT hr= S_OK;
    WCHAR szbuf[20];

	if(_pPointer == NULL)
	{
		swprintf(szbuf, L"PointerIn");
		_pPointer = new CPointerInputPin(NAME("Pointer Input Pin"), this, &hr, szbuf);

        ASSERT(SUCCEEDED(hr));
        if(_pPointer == NULL) {
            return NULL;
        }
	}

	if (n < 2)
	{
		return CTransformN3Filter::GetPin(n);
	}
	else if (n == 2)
	{
		return _pPointer;
	}
	else
	{
		return NULL;
	}
}

STDMETHODIMP
CNonLinearZoom::FindPin(LPCWSTR Id, IPin **ppPin)
{
	CheckPointer(ppPin, E_POINTER);
    ValidateReadWritePtr(ppPin, sizeof(IPin *));

    if (0 == lstrcmpW(Id, L"In"))
	{
        *ppPin = GetPin(0);
    }
    else if (0 == lstrcmpW(Id, L"Out"))
	{
        *ppPin = GetPin(1);
    }
	else if (0 == lstrcmpW(Id, L"PointerIn"))
	{
		*ppPin = GetPin(2);
	}
    else
	{
        *ppPin = NULL;
        return VFW_E_NOT_FOUND;
    }

    HRESULT hr = NOERROR;

    //  AddRef() returned pointer - but GetPin could fail if memory is low.
    if(*ppPin)
	{
        (*ppPin)->AddRef();
    }
    else
	{
        hr = E_OUTOFMEMORY;  // probably.  There's no pin anyway.
    }

	return hr;
}

//
// Transform
//
// Copy the input sample into the output sample
// Then transform the output sample 'in place'
//
HRESULT CNonLinearZoom::Transform(IMediaSample *pIn, IMediaSample *pOut)
{
    CAutoLock cAutoLock(&m_NonLinearZoomLock);

	if (_maskSizeIsChanged)
	{
		_maskSizeIsChanged = false;
		CreateEllipseMaskInfo(_maskSizeWidth, _maskSizeHeight);
	}

	HRESULT hr = Zoom(pIn, pOut);
    if (FAILED(hr)) {
        return hr;
    }
	return NOERROR;

} // Transform



HRESULT
CNonLinearZoom::Zoom(IMediaSample *pSource, IMediaSample *pDest) const
{
	long lSourceSize, lDestSize;
	BYTE *pSourceBuffer, *pDestBuffer;

    lSourceSize = pSource->GetActualDataLength();
    lDestSize = pDest->GetActualDataLength();

    pSource->GetPointer(&pSourceBuffer);
    pDest->GetPointer(&pDestBuffer);

	CMediaType imt = m_pInput->CurrentMediaType();
	CMediaType omt = m_pOutput->CurrentMediaType();
	VIDEOINFO *ivinfo = (VIDEOINFO *)imt.Format();
	VIDEOINFO *ovinfo = (VIDEOINFO *)omt.Format();

	if (ivinfo->bmiHeader.biWidth == ovinfo->bmiHeader.biWidth &&
		ivinfo->bmiHeader.biHeight == ovinfo->bmiHeader.biHeight)
	{
		CopyMemory(pDestBuffer, pSourceBuffer, lDestSize);
	}
	else
	{
		int bCnt;
		int srcWidth, srcHeight, srcMaskX, srcMaskY;
		int destWidth, destHeight, destMaskX, destMaskY;
		float scaleX, scaleY;
		int destWidthBC, srcWidthBC;
		int i, j, is, js, x1, y1, x2, y2;


		bCnt = ivinfo->bmiHeader.biBitCount / 8;

		// Source and Destination Parameters
		srcWidth = ivinfo->bmiHeader.biWidth;
		srcHeight = ivinfo->bmiHeader.biHeight;
		destWidth = ovinfo->bmiHeader.biWidth;
		destHeight = ovinfo->bmiHeader.biHeight;

		// Source Mask position
		if (_pointerInDisable || _pPointer->IsConnected() == FALSE)
		{
			srcMaskX = _maskPosX;
			srcMaskY = -_maskPosY + srcHeight - _maskSizeHeight;
		} else 
		{
			flFloat cx, cy;
			_pointerInfo.getNormalizedPointerPosition(cx, cy);
			srcMaskX = (int)(float(srcWidth) * cx);
			srcMaskY = -(int)(float(srcHeight) * cy) + srcHeight - _maskSizeHeight;
		}

		if (srcMaskX < 0)
			srcMaskX = 0;
		else if (srcWidth - _maskSizeWidth < srcMaskX)
			srcMaskX = srcWidth - _maskSizeWidth;
		
		if (srcMaskY < 0)
			srcMaskY = 0;
		else if (srcHeight - _maskSizeHeight < srcMaskY)
			srcMaskY = srcHeight - _maskSizeHeight;

		// Destination Mask position
		destMaskX = int(float(destWidth - _maskSizeWidth) / float(srcWidth - _maskSizeWidth) * float(srcMaskX));
		destMaskY = int(float(destHeight - _maskSizeHeight) / float(srcHeight - _maskSizeHeight) * float(srcMaskY));

		if (destMaskX < 0)
			destMaskX = 0;
		else if (destWidth - _maskSizeWidth < destMaskX)
			destMaskX = destWidth - _maskSizeWidth;
		
		if (destMaskY < 0)
			destMaskY = 0;
		else if (destHeight - _maskSizeHeight < destMaskY)
			destMaskY = destHeight - _maskSizeHeight;

		// misc
		scaleX = float(srcWidth) / float(destWidth);
		scaleY = float(srcHeight) / float(destHeight);

		destWidthBC = destWidth * bCnt;
		srcWidthBC = srcWidth * bCnt;


		// Clear Buffer
		FillMemory(pDestBuffer, lDestSize, 0);


		// Mask Rectangle's points
		x1 = destMaskX;
		x2 = destMaskX + _maskSizeWidth;
		y1 = destMaskY;
		y2 = destMaskY + _maskSizeHeight;


		switch(_mask)
		{
		// Rectangle MASK
		case M_RECTANGLE:
			{
				float wa, wb;
				float fl, fld, fr, frd, scaleXwa, scaleYwa;
				int begin, end;

				BYTE *pSB, *pDB;

				// Part1
				fl = 0.0f;
				fld = (y1 != 0) ? (float(x1) / float(y1)) : 0.0f;
				fr = 0.0f;
				frd = (y1 != 0) ? (float(destWidth - x2) / float(y1)) : 0.0f;
				pSB = pSourceBuffer;
				pDB = pDestBuffer;

				for(i = 0; i <= y1; i++)
				{
					begin = (int)fl;
					end = (destWidth - (int)fr);

					wb = float(i) / float(y1);
					wb *= wb;
					wa = 1.0f - wb;

					is = (int)(float(i) * scaleY * wa + float(i - destMaskY + srcMaskY) * wb);

					pDB = pDestBuffer + i * destWidthBC;
					pSB = pSourceBuffer + is * srcWidthBC;

					scaleXwa = scaleX * wa;

					for(j = begin; j < end; j++)
					{
						js = (int)(float(j) * scaleXwa + float(j - destMaskX + srcMaskX) * wb);

						if (0 > is || is >= srcHeight || 0 > js || js >= srcWidth)
							continue;

						CopyMemory(pDB + j * 3, pSB + js * 3, 3);
					}

					fl += fld;
					fr += frd;
				}

				// Part2
				fl = 0.0f;
				fld = (x2 != destWidth) ? (float(y1) / float(destWidth - x2)) : 0.0f;
				fr = 0.0f;
				frd = (x2 != destWidth) ? (float(destHeight - y2) / float(destWidth - x2)) : 0.0f;

				for(i = destWidth - 1; i >= x2; i--)
				{
					begin = (int)fl;
					end = (destHeight - (int)fr);

					wb = float(destWidth - i) / float(destWidth - x2);
					wb *= wb;
					wa = 1.0f - wb;

					is = (int)(float(i) * scaleX * wa + float(i - destMaskX + srcMaskX) * wb);

					pDB = pDestBuffer + i * bCnt;
					pSB = pSourceBuffer + is * bCnt;

					scaleYwa = scaleY * wa;

					for(j = begin; j < end; j++)
					{
						js = (int)(float(j) * scaleYwa + float(j - destMaskY + srcMaskY) * wb);

						if (0 > js || js >= srcHeight || 0 > is || is >= srcWidth)
							continue;

						CopyMemory(pDB + j * destWidthBC, pSB + js * srcWidthBC, 3);
					}

					fl += fld;
					fr += frd;
				}

				// Part3
				fl = 0.0f;
				fld = (y2 != destHeight) ? (float(x1) / float(destHeight - y2)) : 0.0f;
				fr = 0.0f;
				frd = (y2 != destHeight) ? (float(destWidth - x2) / float(destHeight - y2)) : 0.0f;
				pSB = pSourceBuffer;
				pDB = pDestBuffer;

				for(i = destHeight - 1; i >= y2; i--)
				{
					begin = (int)fl;
					end = (destWidth - (int)fr);

					wb = float(destHeight - i) / float(destHeight - y2);
					wb *= wb;
					wa = 1.0f - wb;

					is = (int)(float(i) * scaleY * wa + float(i - destMaskY + srcMaskY) * wb);

					pDB = pDestBuffer + i * destWidthBC;
					pSB = pSourceBuffer + is * srcWidthBC;

					scaleXwa = scaleX * wa;

					for(j = begin; j < end; j++)
					{
						js = (int)(float(j) * scaleXwa + float(j - destMaskX + srcMaskX) * wb);

						if (0 > is || is >= srcHeight || 0 > js || js >= srcWidth)
							continue;

						CopyMemory(pDB + j * 3, pSB + js * 3, 3);
					}

					fl += fld;
					fr += frd;
				}

				// Part4
				fl = 0.0f;
				fld = (x1 != 0) ? (float(y1) / float(x1)) : 0.0f;
				fr = 0.0f;
				frd = (x1 != 0) ? (float(destHeight - y2) / float(x1)) : 0.0f;

				for(i = 0; i <= x1; i++)
				{
					begin = (int)fl;
					end = (destHeight - (int)fr);

					wb = float(i) / float(x1);
					wb *= wb;
					wa = 1.0f - wb;

					is = (int)(float(i) * scaleX * wa + float(i - destMaskX + srcMaskX) * wb);

					pDB = pDestBuffer + i * 3;
					pSB = pSourceBuffer + is * 3;

					scaleYwa = scaleY * wa;

					for(j = begin; j < end; j++)
					{
						js = (int)(float(j) * scaleYwa + float(j - destMaskY + srcMaskY) * wb);

						if (0 > js || js >= srcHeight || 0 > is || is >= srcWidth)
							continue;

						CopyMemory(pDB + j * destWidthBC, pSB + js * srcWidthBC, 3);
					}

					fl += fld;
					fr += frd;
				}

				// center
				pSB = pSourceBuffer;
				pDB = pDestBuffer;
				for(i = y1; i < y2; i++)
				{
					is = i - destMaskY + srcMaskY;

					pDB = pDestBuffer + i * destWidthBC;
					pSB = pSourceBuffer + is * srcWidthBC;

					CopyMemory(pDB + x1 * 3, pSB + (x1 - destMaskX + srcMaskX) * 3, (x2 - x1) * 3);
				}
			}
			break;

		// Ellipse MASK
		case M_ELLIPSE:
			{
				float wa, wb;
				int elipCentX, elipCentY;
				float fl, fr, fld, frd;
				int begin, end, iEnd;

				BYTE *pSB, *pDB;
				short *pEMI;

				elipCentX = destMaskX + _maskSizeWidth / 2;
				elipCentY = destMaskY + _maskSizeHeight / 2;

				pSB = pSourceBuffer;
				pDB = pDestBuffer;


				// Part1
				if (0 < elipCentY)
				{
					iEnd = elipCentY;

					if (0 < elipCentX)
					{
						fld = float(elipCentX) / float(elipCentY);
					} else 
					{
						fld = 0.0f;
						iEnd -= int(float(-elipCentX * elipCentY) / float(destWidth - elipCentX));
					}

					if (elipCentX < destWidth)
					{
						frd = float(destWidth - elipCentX) / float(elipCentY);
					} else 
					{
						frd = 0.0f;
						iEnd -= int(float((elipCentX - destWidth) * elipCentY) / float(elipCentX));
					}

					if (destHeight < iEnd)
						iEnd = destHeight;

					fl = 0.0f;
					fr = 0.0f;

					for(i = 0; i < iEnd; i++)
					{
						begin = (int)fl;
						end = destWidth - (int)fr;

						pEMI = _elipMaskInfo + EMIIDX(_emHH + i - elipCentY, _emHW + begin - elipCentX, 0);

						for(j = begin; j < end; j++)
						{

							if (*pEMI)
							{
								is = i - destMaskY + srcMaskY;
								js = j - destMaskX + srcMaskX;

								pEMI += 3;

								if (0 > js || js >= srcWidth || 0 > is || is >= srcHeight)
									continue;

								CopyMemory(pDestBuffer + j * 3 + i * destWidthBC,
										pSourceBuffer + js * 3 + is * srcWidthBC,
										3);

							} else
							{
								pEMI++;
								pEMI++;

								wa = float((int)*pEMI + elipCentY - i) / float((int)*pEMI + elipCentY);
								wb = 1.0f - wa;

								wb *= wb;
								wa = 1.0f - wb;

								pEMI++;

								is = (int)(float(i) * scaleY * wa + float(i - destMaskY + srcMaskY) * wb);
								js = (int)(float(j) * scaleX * wa + float(j - destMaskX + srcMaskX) * wb);

								if (0 > js || js >= srcWidth || 0 > is || is >= srcHeight)
									continue;

								CopyMemory(pDestBuffer + j * 3 + i * destWidthBC,
											pSourceBuffer + js * 3 + is * srcWidthBC,
											3);

							}

						}

						fl += fld;
						fr += frd;
					}

				}

				// Part2
				if (elipCentY < destHeight)
				{
					iEnd = elipCentY;

					if (0 < elipCentX)
					{
						fld = float(elipCentX) / float(destHeight - elipCentY);
					} else 
					{
						fld = 0.0f;
						iEnd += int(float(-elipCentX * (destHeight - elipCentY)) / float(destWidth - elipCentX));
					}

					if (elipCentX < destWidth)
					{
						frd = float(destWidth - elipCentX) / float(destHeight - elipCentY);
					} else 
					{
						frd = 0.0f;
						iEnd += int(float((elipCentX - destWidth) * (destHeight - elipCentY)) / float(elipCentX));
					}

					if (iEnd < 0)
						iEnd = 0;

					fl = 0.0f;
					fr = 0.0f;

					for(i = destHeight - 1; i >= iEnd; i--)
					{
						begin = (int)fl;
						end = destWidth - (int)fr;

						pEMI = _elipMaskInfo + EMIIDX(_emHH + i - elipCentY, _emHW + begin - elipCentX, 0);

						for(j = begin; j < end; j++)
						{

							if (*pEMI)
							{
								is = i - destMaskY + srcMaskY;
								js = j - destMaskX + srcMaskX;

								pEMI += 3;

								if (0 > js || js >= srcWidth || 0 > is || is >= srcHeight)
									continue;

								CopyMemory(pDestBuffer + j * 3 + i * destWidthBC,
										pSourceBuffer + js * 3 + is * srcWidthBC,
										3);

							} else
							{
								pEMI++;
								pEMI++;

								wa = float(i - ((int)*pEMI + elipCentY)) / float(destHeight - ((int)*pEMI + elipCentY));
								wb = 1.0f - wa;

								wb *= wb;
								wa = 1.0f - wb;

								pEMI++;

								is = (int)(float(i) * scaleY * wa + float(i - destMaskY + srcMaskY) * wb);
								js = (int)(float(j) * scaleX * wa + float(j - destMaskX + srcMaskX) * wb);

								if (0 > js || js >= srcWidth || 0 > is || is >= srcHeight)
									continue;

								CopyMemory(pDestBuffer + j * 3 + i * destWidthBC,
											pSourceBuffer + js * 3 + is * srcWidthBC,
											3);
							}

						}

						fl += fld;
						fr += frd;
					}

				}

				// Part3
				if (0 < elipCentX)
				{
					iEnd = elipCentX;

					if (0 < elipCentY)
					{
						fld = float(elipCentY) / float(elipCentX);
					} else 
					{
						fld = 0.0f;
						iEnd -= int(float(-elipCentY * elipCentX) / float(destHeight - elipCentY));
					}

					if (elipCentY < destHeight)
					{
						frd = float(destHeight - elipCentY) / float(elipCentX);
					} else 
					{
						frd = 0.0f;
						iEnd -= int(float((elipCentY - destHeight) * elipCentX) / float(elipCentY));
					}

					if (destWidth < iEnd)
						iEnd = destWidth;

					fl = 0.0f;
					fr = 0.0f;

					for(i = 0; i < iEnd; i++)
					{
						begin = (int)fl;
						end = destHeight - (int)fr;

						pEMI = _elipMaskInfo + EMIIDX(_emHH + begin - elipCentY, _emHW + i - elipCentX, 0);

						for(j = begin; j < end; j++)
						{
							if (*pEMI)
							{
								is = i - destMaskX + srcMaskX;
								js = j - destMaskY + srcMaskY;

								pEMI += _emW * 3;

								if (0 > js || js >= srcHeight || 0 > is || is >= srcWidth)
									continue;

								CopyMemory(pDestBuffer + i * 3 + j * destWidthBC,
										pSourceBuffer + is * 3 + js * srcWidthBC,
										3);

							} else
							{
								wa = float((int)*(pEMI+1) + elipCentX - i) / float((int)*(pEMI+1) + elipCentX);
								wb = 1.0f - wa;

								wb *= wb;
								wa = 1.0f - wb;

								pEMI += _emW * 3;

								is = (int)(float(i) * scaleX * wa + float(i - destMaskX + srcMaskX) * wb);
								js = (int)(float(j) * scaleY * wa + float(j - destMaskY + srcMaskY) * wb);

								if (0 > js || js >= srcHeight || 0 > is || is >= srcWidth)
									continue;

								CopyMemory(pDestBuffer + i * 3 + j * destWidthBC,
											pSourceBuffer + is * 3 + js * srcWidthBC,
											3);

							}

						}

						fl += fld;
						fr += frd;
					}

				}

				// Part4
				if (elipCentX < destWidth)
				{
					iEnd = elipCentX;

					if (0 < elipCentY)
					{
						fld = float(elipCentY) / float(destWidth - elipCentX);
					} else 
					{
						fld = 0.0f;
						iEnd += int(float(-elipCentY * (destWidth - elipCentX)) / float(destHeight - elipCentY));
					}

					if (elipCentY < destHeight)
					{
						frd = float(destHeight - elipCentY) / float(destWidth - elipCentX);
					} else 
					{
						frd = 0.0f;
						iEnd += int(float((elipCentY - destHeight) * (destWidth - elipCentX)) / float(elipCentY));
					}

					if (iEnd < 0)
						iEnd = 0;

					fl = 0.0f;
					fr = 0.0f;

					for(i = destWidth - 1; iEnd <= i; i--)
					{
						begin = (int)fl;
						end = destHeight - (int)fr;

						pEMI = _elipMaskInfo + EMIIDX(_emHH + begin - elipCentY, _emHW + i - elipCentX, 0);

						for(j = begin; j < end; j++)
						{
							if (*pEMI)
							{
								is = i - destMaskX + srcMaskX;
								js = j - destMaskY + srcMaskY;

								pEMI += _emW * 3;

								if (0 > js || js >= srcHeight || 0 > is || is >= srcWidth)
									continue;

								CopyMemory(pDestBuffer + i * 3 + j * destWidthBC,
										pSourceBuffer + is * 3 + js * srcWidthBC,
										3);

							} else
							{
								wa = float(i - ((int)*(pEMI+1) + elipCentX)) / float(destWidth - ((int)*(pEMI+1) + elipCentX));
								wb = 1.0f - wa;

								wb *= wb;
								wa = 1.0f - wb;

								pEMI += _emW * 3;

								is = (int)(float(i) * scaleX * wa + float(i - destMaskX + srcMaskX) * wb);
								js = (int)(float(j) * scaleY * wa + float(j - destMaskY + srcMaskY) * wb);

								if (0 > js || js >= srcHeight || 0 > is || is >= srcWidth)
									continue;

								CopyMemory(pDestBuffer + i * 3 + j * destWidthBC,
											pSourceBuffer + is * 3 + js * srcWidthBC,
											3);
							}

						}

						fl += fld;
						fr += frd;
					}

				}

			}
			break;
		}
	}


    // Copy the sample times
    REFERENCE_TIME TimeStart, TimeEnd;
    if (NOERROR == pSource->GetTime(&TimeStart, &TimeEnd)) {
        pDest->SetTime(&TimeStart, &TimeEnd);
    }

    LONGLONG MediaStart, MediaEnd;
    if (pSource->GetMediaTime(&MediaStart,&MediaEnd) == NOERROR) {
        pDest->SetMediaTime(&MediaStart,&MediaEnd);
    }

    // Copy the Sync point property
    HRESULT hr = pSource->IsSyncPoint();
    if (hr == S_OK) {
        pDest->SetSyncPoint(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetSyncPoint(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

	// Copy the media type
    //AM_MEDIA_TYPE *pMediaType;
    //pSource->GetMediaType(&pMediaType);
    //pDest->SetMediaType(pMediaType);
    //DeleteMediaType(pMediaType);

    // Copy the preroll property
    hr = pSource->IsPreroll();
    if (hr == S_OK) {
        pDest->SetPreroll(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetPreroll(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

    // Copy the discontinuity property
    hr = pSource->IsDiscontinuity();
    if (hr == S_OK) {
	pDest->SetDiscontinuity(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetDiscontinuity(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

    // Copy the actual data length
    //long lDataLength = pSource->GetActualDataLength();
    //pDest->SetActualDataLength(lDataLength);

	long actLen = pDest->GetActualDataLength();

    return NOERROR;
}

HRESULT
CNonLinearZoom::AllocBaseBuffer(unsigned int size)
{
	delete [] _baseBuf;

	_baseBuf = new BYTE[size];
	if (!_baseBuf)
	{
		_baseBufSize = 0;
		return E_FAIL;
	}
	_baseBufSize = size;

	return NOERROR;
}

HRESULT
CNonLinearZoom::CreateEllipseMaskInfo(int width, int height)
{
	int dx, dy, ex, ey, dist;

	_emW = 1440;
	_emH = 960;
	_emHW = _emW / 2;
	_emHH = _emH / 2;

	if (!_elipMaskInfo)
	{
		_elipMaskInfo = new short[_emH * _emW * 3];
	}

	int hw = width / 2;
	int hh = height / 2;

	for(int i = 0; i < _emH; i++)
	{
		for(int j = 0; j < _emW; j++)
		{
			dx = j - _emHW;
			dy = i - _emHH;
			dx = int(float(dx) * (float(height) / float(width)));
			dist = (int)sqrtf(float(dx * dx + dy * dy));

			if (dist != 0.0f)
			{
				ex = int(float(dx) / float(dist) * float(hw));
				ey = int(float(dy) / float(dist) * float(hh));
			} else 
			{
				ex = 0;
				ey = 0;
			}
			_elipMaskInfo[EMIIDX(i, j, 0)] = 0;
			_elipMaskInfo[EMIIDX(i, j, 1)] = ex;
			_elipMaskInfo[EMIIDX(i, j, 2)] = ey;
		}
	}


	float y;
	int x, x2;

	for(int ii = 0, i = _emHH - (int)hh; i < _emHH + (int)hh; i++, ii++)
	{
		y = float(ii) / float(height - 1) * 2.0f - 1.0f;
		x = int(float(width / 2) * (1.0f - sqrt(1.0f - y * y)));
		x2 = width - x;

		for(int j = _emHW - (int)hw + x; j < _emHW - (int)hw + x2; j++)
		{
			_elipMaskInfo[EMIIDX(i, j, 0)] = 1;
		}
	}

	return NOERROR;
}

//
// CheckInputType
//
// Check the input type is OK, return an error otherwise
//
HRESULT CNonLinearZoom::CheckInputType(const CMediaType *mtIn)
{
	if (*mtIn->FormatType() != FORMAT_VideoInfo)
		return E_INVALIDARG;

	if (!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB24))
		return E_INVALIDARG;

	return NOERROR;

} // CheckInputType

HRESULT CNonLinearZoom::CheckOutputType(const CMediaType *mtOut)
{
	if (*mtOut->FormatType() != FORMAT_VideoInfo)
		return E_INVALIDARG;

	if (!IsEqualGUID(*mtOut->Subtype(), MEDIASUBTYPE_RGB24))
		return E_INVALIDARG;

	VIDEOINFO* pvi = (VIDEOINFO *)mtOut->Format();
	if (pvi->rcSource.right != pvi->bmiHeader.biWidth ||
		pvi->rcSource.bottom != pvi->bmiHeader.biHeight)
		return E_INVALIDARG;

	return NOERROR;
}

//
// DecideBufferSize
//
// Tell the output pin's allocator what size buffers we
// require. Can only do this when the input is connected
//
HRESULT CNonLinearZoom::DecideBufferSize(IMemAllocator *pAlloc,ALLOCATOR_PROPERTIES *pProp)
{
	if (m_pInput->IsConnected() == FALSE)
		return E_UNEXPECTED;

	ASSERT(pAlloc);
	ASSERT(pProp);
	ASSERT(m_pInput->CurrentMediaType().bFixedSizeSamples);

	VIDEOINFOHEADER *pvi = (VIDEOINFOHEADER *)m_pOutput->CurrentMediaType().pbFormat;
	pvi->bmiHeader.biSizeImage = GetBitmapSize(&pvi->bmiHeader);
	ASSERT(pvi->bmiHeader.biSizeImage != 0);

	pProp->cBuffers = 1;
	pProp->cbBuffer = pvi->bmiHeader.biSizeImage;

	ALLOCATOR_PROPERTIES Actual;
	HRESULT hr = pAlloc->SetProperties(pProp, &Actual);
	if (FAILED(hr))
		return hr;

	ASSERT(Actual.cBuffers == 1);

	if (Actual.cBuffers < pProp->cBuffers ||
		Actual.cbBuffer < pProp->cbBuffer)
		return E_FAIL;


	/// Allocate BASE BUFFER
	hr = AllocBaseBuffer(Actual.cbBuffer);
	if (FAILED(hr))
		return E_FAIL;

    return NOERROR;

} // DecideBufferSize


HRESULT
CNonLinearZoom::SetMediaType(PIN_DIRECTION direction, const CMediaType *pmt)
{
	// Check Mask Size
	VIDEOINFO *vinfo = (VIDEOINFO *)pmt->Format();

	long width = vinfo->bmiHeader.biWidth;
	long height = vinfo->bmiHeader.biHeight;

	if (width <= _maskSizeWidth)
	{
		_maskSizeWidth = width - 1;
		_maskSizeIsChanged = true;
	}
	if (height <= _maskSizeHeight)
	{
		_maskSizeHeight = height - 1;
		_maskSizeIsChanged = true;
	}

	// Reconnect Output PIN
    if (m_pInput->IsConnected() && m_pOutput->IsConnected())
	{
        FILTER_INFO fInfo;

        QueryFilterInfo( &fInfo );

        if (direction == PINDIR_INPUT && *pmt != m_pOutput->CurrentMediaType())
            fInfo.pGraph->Reconnect( m_pOutput );

        QueryFilterInfoReleaseGraph( fInfo );
    }

	return NOERROR;
}


//
// GetMediaType
//
// I support one type, namely the type of the input pin
// We must be connected to support the single output type
//
HRESULT CNonLinearZoom::GetMediaType(int iPosition, CMediaType *pMediaType)
{
    CAutoLock cAutoLock(&m_NonLinearZoomLock);
    if (iPosition < 0) {
        return E_INVALIDARG;
    }
    if (iPosition > 0) {
        return VFW_S_NO_MORE_ITEMS;
    }
	
	ASSERT(m_pInput != NULL);
	if (m_pInput->IsConnected() == FALSE)
        return E_UNEXPECTED;
	CMediaType iMediaType = m_pInput->CurrentMediaType();

	*pMediaType = iMediaType;
	
	VIDEOINFO *pvi = (VIDEOINFO *)pMediaType->AllocFormatBuffer(sizeof(VIDEOINFO));
    if (pvi == NULL) {
		return(E_OUTOFMEMORY);
    }
    ZeroMemory(pvi, sizeof(VIDEOINFO));

	VIDEOINFO *pivi = (VIDEOINFO *)iMediaType.Format();

	pvi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	pvi->bmiHeader.biBitCount = pivi->bmiHeader.biBitCount;
	pvi->bmiHeader.biXPelsPerMeter = pivi->bmiHeader.biXPelsPerMeter;
	pvi->bmiHeader.biYPelsPerMeter = pivi->bmiHeader.biYPelsPerMeter;
	pvi->bmiHeader.biClrImportant = pivi->bmiHeader.biClrImportant;
	pvi->bmiHeader.biClrUsed = pivi->bmiHeader.biClrUsed;
	for(DWORD i = 0; i < pvi->bmiHeader.biClrUsed; i++)
		pvi->bmiColors[i] = pivi->bmiColors[i];

	switch(_outVR){
	case VR_SQCIF:
		pvi->bmiHeader.biWidth		= 128;
		pvi->bmiHeader.biHeight		= 96;
		break;
	case VR_QCIF:
		pvi->bmiHeader.biWidth		= 176;
		pvi->bmiHeader.biHeight		= 144;
		break;
	case VR_CIF:
		pvi->bmiHeader.biWidth		= 352;
		pvi->bmiHeader.biHeight		= 288;
		break;
	case VR_160x120:
		pvi->bmiHeader.biWidth		= 160;
		pvi->bmiHeader.biHeight		= 120;
		break;
	case VR_320x240:
		pvi->bmiHeader.biWidth		= 320;
		pvi->bmiHeader.biHeight		= 240;
		break;
	case VR_360x240:
		pvi->bmiHeader.biWidth		= 360;
		pvi->bmiHeader.biHeight		= 240;
		break;
	case VR_640x480:
		pvi->bmiHeader.biWidth		= 640;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_720x480:
		pvi->bmiHeader.biWidth		= 720;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_800x600:
		pvi->bmiHeader.biWidth		= 800;
		pvi->bmiHeader.biHeight		= 600;
		break;
	case VR_1024x768:
		pvi->bmiHeader.biWidth		= 1024;
		pvi->bmiHeader.biHeight		= 768;
		break;
	case VR_1280x1024:
		pvi->bmiHeader.biWidth		= 1280;
		pvi->bmiHeader.biHeight		= 1024;
		break;
	case VR_1600x1200:
		pvi->bmiHeader.biWidth		= 1600;
		pvi->bmiHeader.biHeight		= 1200;
		break;
	case VR_HDTV720P:
		pvi->bmiHeader.biWidth		= 1280;
		pvi->bmiHeader.biHeight		= 720;
		break;
	case VR_HDTV1080I:
		pvi->bmiHeader.biWidth		= 1920;
		pvi->bmiHeader.biHeight		= 1080;
		break;
	case VR_SDTV480P:
		pvi->bmiHeader.biWidth		= 852;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_SDTV480I:
		pvi->bmiHeader.biWidth		= 852;
		pvi->bmiHeader.biHeight		= 480;
		break;
	}

	pMediaType->SetTemporalCompression(FALSE);
    pvi->bmiHeader.biPlanes		= 1;
	pvi->bmiHeader.biSizeImage	= GetBitmapSize(&pvi->bmiHeader);

	SetRectEmpty(&(pvi->rcSource));	// we want the whole image area rendered.
    SetRectEmpty(&(pvi->rcTarget));	// no particular destination rectangle
	pvi->rcSource.right = pvi->bmiHeader.biWidth;
	pvi->rcSource.bottom = pvi->bmiHeader.biHeight;

	const GUID SubTypeGUID = GetBitmapSubtype(&pvi->bmiHeader);
	pMediaType->SetSampleSize(pvi->bmiHeader.biSizeImage);
	pMediaType->SetType(&MEDIATYPE_Video);
	pMediaType->SetFormatType(&FORMAT_VideoInfo);

	return NOERROR;

} // GetMediaType


STDMETHODIMP
CNonLinearZoom::get_OutputVideoResolution(unsigned int *videoResolution)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	*videoResolution = (unsigned int)_outVR;

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::put_OutputVideoResolution(unsigned int videoResolution)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	if (_outVR != videoResolution)
	{
		_outVR = (VideoResolutionEnum)videoResolution;
		SetDirty(TRUE);

		ReconnectPin((IPin *)m_pOutput, NULL);
	}

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::get_Mask(unsigned int *mask)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	*mask = (unsigned int)_mask;

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::put_Mask(unsigned int mask)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	if (_mask != mask)
	{
		_mask = (MaskEnum)mask;
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::get_MaskPosition(int* x, int* y)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	*x = _maskPosX;
	*y = _maskPosY;

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::put_MaskPosition(int x, int y)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	if (_maskPosX != x || _maskPosY != y)
	{
		_maskPosX = x;
		_maskPosY = y;
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::get_MaskSize(int* width, int* height)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	*width = _maskSizeWidth;
	*height = _maskSizeHeight;

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::put_MaskSize(int width, int height)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);

	if (_maskSizeWidth == width && _maskSizeHeight == height)
		return NOERROR;

	if (width == 0 || height == 0)
		return NOERROR;

	// Update Mask Size
	_maskSizeWidth = width;
	_maskSizeHeight = height;

	int videoWidth, videoHeight;

	// Check Input Video size
	if (m_pInput != NULL)
	{
		const CMediaType& imt = m_pInput->CurrentMediaType();
		VIDEOINFO *ivinfo = (VIDEOINFO *)imt.Format();
		if (ivinfo != NULL)
		{
			videoWidth = ivinfo->bmiHeader.biWidth;
			videoHeight = ivinfo->bmiHeader.biHeight;

			if (videoWidth <= _maskSizeWidth)
				_maskSizeWidth = videoWidth - 1;
			if (videoHeight <= _maskSizeHeight)
				_maskSizeHeight = videoHeight - 1;
		}
	}

	// Check Output Video size
	if (m_pOutput != NULL)
	{
		const CMediaType& omt = m_pOutput->CurrentMediaType();
		VIDEOINFO *ovinfo = (VIDEOINFO *)omt.Format();
		if (ovinfo != NULL)
		{
			videoWidth = ovinfo->bmiHeader.biWidth;
			videoHeight = ovinfo->bmiHeader.biHeight;

			if (videoWidth <= _maskSizeWidth)
				_maskSizeWidth = videoWidth - 1;
			if (videoHeight <= _maskSizeHeight)
				_maskSizeHeight = videoHeight - 1;
		}
	}

	_maskSizeIsChanged = true;
	SetDirty(TRUE);

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::get_PointerInDisable(bool* flag)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	*flag = _pointerInDisable;

	return NOERROR;
}

STDMETHODIMP
CNonLinearZoom::put_PointerInDisable(bool flag)
{
	CAutoLock cAutoLock(&m_NonLinearZoomLock);
	if (_pointerInDisable != flag)
	{
		_pointerInDisable = flag;
		SetDirty(TRUE);
	}

	return NOERROR;
}

#define WRITEOUT(var)  hr = pStream->Write(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define WRITENOUT(var,n)  hr = pStream->Write(var, n, NULL); \
if (FAILED(hr)) return hr;
#define READIN(var)    hr = pStream->Read(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define READNIN(var,n)    hr = pStream->Read(var, n, NULL); \
if (FAILED(hr)) return hr;


// IPersitStream
STDMETHODIMP
CNonLinearZoom::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_NonLinearZoom;
	return S_OK;
} // GetClassID

HRESULT
CNonLinearZoom::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	WRITEOUT(_outVR);
	WRITEOUT(_mask);
	WRITEOUT(_maskPosX);
	WRITEOUT(_maskPosY);
	WRITEOUT(_maskSizeWidth);
	WRITEOUT(_maskSizeHeight);
	WRITEOUT(_pointerInDisable);
	return NOERROR;
	
} // WriteToStream

HRESULT
CNonLinearZoom::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	READIN(_outVR);
	READIN(_mask);
	READIN(_maskPosX);
	READIN(_maskPosY);
	READIN(_maskSizeWidth);
	READIN(_maskSizeHeight);
	READIN(_pointerInDisable);
	return NOERROR;
	
} // ReadFromStream

DWORD
CNonLinearZoom::GetSoftwareVersion(void)
{
	return 1;
} // GetSoftwareVersion

int
CNonLinearZoom::SizeMax()
{
	return sizeof(unsigned int) * 6 + sizeof(bool) * 1;
} // SizeMax


//
// GetPages
//
// This is the sole member of ISpecifyPropertyPages
// Returns the clsid's of the property pages we support
//
STDMETHODIMP CNonLinearZoom::GetPages(CAUUID *pPages)
{
    pPages->cElems = 1;
    pPages->pElems = (GUID *) CoTaskMemAlloc(sizeof(GUID));
    if (pPages->pElems == NULL) {
        return E_OUTOFMEMORY;
    }
    *(pPages->pElems) = CLSID_NonLinearZoomPropertyPage;
    return NOERROR;

} // GetPages

HRESULT
CNonLinearZoom::ReceivePointer(IMediaSample *pSource, CPointerInputPin *ppin)
{
	// Get Media Sample data
	flLong dataLength;
	BYTE *data;
	dataLength = pSource->GetActualDataLength();
	pSource->GetPointer(&data);

	// Setup New PointerInfo
	CAutoLock autoLock(&m_NonLinearZoomLock);
	_pointerInfo.setChanged(data, dataLength);

	return NO_ERROR;
}

//////////////////////////////////////////////////////////////////////////////

// =================================================================
// Implements the CPointerInputPin class
// =================================================================

// Check streaming status
HRESULT
CPointerInputPin::CheckStreaming()
{

	//  Shouldn't be able to get any data if we're not connected!
	ASSERT(IsConnected());

	//  we're flushing
	if(m_bFlushing)
	{
		return S_FALSE;
	}
	//  Don't process stuff in Stopped state
	if(IsStopped())
	{
		return VFW_E_WRONG_STATE;
	}
	if(m_bRunTimeError)
	{
		return VFW_E_RUNTIME_ERROR;
	}
	return S_OK;
}


// constructor
CPointerInputPin::CPointerInputPin(TCHAR *pObjectName,  CTransformN3Filter *pTransformFilter,
								   HRESULT * phr, LPCWSTR pName)
: CBaseInputPin(pObjectName, pTransformFilter, &((CNonLinearZoom *)pTransformFilter)->getCSFilter(), phr, pName)
{
    DbgLog((LOG_TRACE,2,TEXT("CPointerInputPin::CPointerInputPin")));
    m_pTransformFilter = (CNonLinearZoom *)pTransformFilter;

}

// provides derived filter a chance to grab extra interfaces
HRESULT
CPointerInputPin::CheckConnect(IPin *pPin)
{
    HRESULT hr = m_pTransformFilter->CheckConnect(PINDIR_INPUT,pPin);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseInputPin::CheckConnect(pPin);
}


// provides derived filter a chance to release it's extra interfaces
HRESULT
CPointerInputPin::BreakConnect()
{
    ASSERT(IsStopped());
    m_pTransformFilter->BreakConnect(PINDIR_INPUT);
    return CBaseInputPin::BreakConnect();
}


// Let derived class know when the input pin is connected
HRESULT
CPointerInputPin::CompleteConnect(IPin *pReceivePin)
{
    HRESULT hr = m_pTransformFilter->CompleteConnect(PINDIR_INPUT,pReceivePin);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseInputPin::CompleteConnect(pReceivePin);
}


// check that we can support a given media type
HRESULT
CPointerInputPin::CheckMediaType(const CMediaType* pmt)
{
 	if(*pmt->Type() == MEDIATYPE_Stream &&
	   *pmt->Subtype() == MEDIASUBTYPE_PointerInfo){
        return NOERROR;
	}else {
		return S_FALSE;
	}
}


// set the media type for this connection
HRESULT
CPointerInputPin::SetMediaType(const CMediaType* mtIn)
{
    HRESULT hr = CBasePin::SetMediaType(mtIn);
    if(FAILED(hr)) {
        return hr;
    }

	if(*mtIn->Type() == MEDIATYPE_Stream &&
	   *mtIn->Subtype() == MEDIASUBTYPE_PointerInfo)
		return S_OK;
	else
		return S_FALSE;
}


// =================================================================
// Implements IMemInputPin interface
// =================================================================


// provide EndOfStream that passes straight downstream
// (there is no queued data)
STDMETHODIMP
CPointerInputPin::EndOfStream(void)
{
    CAutoLock lck(&m_pTransformFilter->getCSReceive());
    HRESULT hr = CheckStreaming();
    return hr;
}


// enter flushing state. Call default handler to block Receives, then
// pass to overridable method in filter
STDMETHODIMP
CPointerInputPin::BeginFlush(void)
{
	CAutoLock lck(&m_pTransformFilter->getCSFilter());
    HRESULT hr = CBaseInputPin::BeginFlush();
    if(FAILED(hr)) {
        return hr;
    }

	return S_OK;
}


// leave flushing state.
// Pass to overridable method in filter, then call base class
// to unblock receives (finally)
STDMETHODIMP
CPointerInputPin::EndFlush(void)
{
    CAutoLock lck(&m_pTransformFilter->getCSFilter());
    return CBaseInputPin::EndFlush();
}


// here's the next block of data from the stream.
// AddRef it yourself if you need to hold it beyond the end
// of this call.
HRESULT
CPointerInputPin::Receive(IMediaSample * pSample)
{
    HRESULT hr;
    CAutoLock lck(&m_pTransformFilter->getCSReceive());
    ASSERT(pSample);

    hr = CBaseInputPin::Receive(pSample);
    if(S_OK == hr)
	{
        hr = ((CNonLinearZoom *)m_pTransformFilter)->ReceivePointer(pSample, this);
    }
    return hr;
}


// override to pass downstream
STDMETHODIMP
CPointerInputPin::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate)
{
    return CBasePin::NewSegment(tStart, tStop, dRate);
}

//////////////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2( TRUE );

} // DllRegisterServer

//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2( FALSE );

} // DllUnregisterServer





/*

#ifdef MARKERON
								if (wb >= 0.95f)
								{
									char bbb[3];
									bbb[0] = int(float(255) * wb);
									bbb[1] = 0;
									bbb[2] = 0;
									CopyMemory(pDestBuffer + j * 3 + i * destWidthBC, bbb,	3);
								}
								if (wa >= 0.97f)
								{
									char bbb[3];
									bbb[0] = int(float(255) * wa);
									bbb[1] = 0;
									bbb[2] = 0;
									CopyMemory(pDestBuffer + j * 3 + i * destWidthBC, bbb,	3);
								}
#else

*/