/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flPointerMixer/PointerInfoInputPin.h>
#include <flPointerMixer/PointerMixer.h>
#include <flBase/flNotify.h>
//------------------------------------------------------------------------------
CPointerInfoInputPin::CPointerInfoInputPin(TCHAR *pObjName,
											CPointerMixer *pointerMixer,
											HRESULT *phr,
											LPCWSTR pPinName) :
CTransformInputPin(pObjName, (CTransformFilter *)pointerMixer, phr, pPinName)
{
    _pointerMixer			= pointerMixer;
    _referenceCount			= 0;

	_pinValid		= false;
	_pinReceiveTime	= 0;
}
//------------------------------------------------------------------------------
CPointerInfoInputPin::~CPointerInfoInputPin()
{
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInfoInputPin::QueryId(LPWSTR * Id)
{
    return AMGetWideString(L"PointerInfo", Id);
}
//------------------------------------------------------------------------------
HRESULT
CPointerInfoInputPin::CheckConnect(IPin *pPin)
{
	return CBasePin::CheckConnect(pPin);
}
//------------------------------------------------------------------------------
HRESULT
CPointerInfoInputPin::BreakConnect()
{
	return CBaseInputPin::BreakConnect();
}
//------------------------------------------------------------------------------
HRESULT
CPointerInfoInputPin::CompleteConnect(IPin *pReceivePin)
{
	CAutoLock lock_it(m_pLock);

	HRESULT hr = CBaseInputPin::CompleteConnect(pReceivePin);
	if (FAILED(hr))
		return hr;

	flUInt n = _pointerMixer->GetNumFreePins();
	ASSERT(n <= 1);
	if (n == 1 || _pointerMixer->_pointerInfoInputPins.getSize() == MAXIMUM_POINTERINFOPINS)
		return NOERROR;

	_pointerMixer->CreateInputPin(_pointerMixer);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CPointerInfoInputPin::SetMediaType(const CMediaType* pmt)
{
    HRESULT hr = CBasePin::SetMediaType(pmt);
    if(FAILED(hr))
        return hr;

    return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CPointerInfoInputPin::CheckMediaType(const CMediaType* pmt)
{
	if (IsEqualGUID(*pmt->Type(), MEDIATYPE_Stream) &&
		IsEqualGUID(*pmt->Subtype(), MEDIASUBTYPE_PointerInfo) &&
		IsEqualGUID(*pmt->FormatType(), FORMAT_None))
		return NOERROR;
	
	return E_INVALIDARG;
}
//------------------------------------------------------------------------------
STDMETHODIMP_(ULONG)
CPointerInfoInputPin::NonDelegatingAddRef()
{
    CAutoLock lock_it(m_pLock);

    _referenceCount++;
    ASSERT(_referenceCount > 0);

	return _referenceCount;
}
//------------------------------------------------------------------------------
STDMETHODIMP_(ULONG)
CPointerInfoInputPin::NonDelegatingRelease()
{
    CAutoLock lock_it(m_pLock);

    _referenceCount--;
    ASSERT(_referenceCount >= 0);

    if (_referenceCount <= 1)
	{
        flInt n = 2;
        if (_referenceCount == 1)
		{
            n = _pointerMixer->GetNumFreePins();
        }

        if (n >= 2) 
		{
            _referenceCount = 0;
            _pointerMixer->DeleteInputPin(this);
            return (ULONG)0;
        }
    }

	return (ULONG)_referenceCount;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInfoInputPin::Receive(IMediaSample * pSample)
{
	ASSERT(pSample != NULL);

	HRESULT hr = CBaseInputPin::Receive(pSample);
	if(FAILED(hr))
		return hr;

	// Notify "This pin is Valid"
	_pinValid = true;
	_pinReceiveTime = timeGetTime();

	// Get Media Sample data
	flLong dataLength;
	BYTE *data;
	dataLength = pSample->GetActualDataLength();
	pSample->GetPointer(&data);

	// Setup New PointerInfo
	CAutoLock autoLock(&_pinLock);
	_pointerInfo.setChanged(data, dataLength);

#ifdef _DEBUG
//	flNotify::post(flNotify::L_INFO, "Actual Data Length (%d)------------------", dataLength);
/*
	if (_pointerInfo.isChanged(PIID_POINTERSHAPE))
	{
		flNotify::post(flNotify::L_INFO, "PointerInfo:: PointerShape=%d", 
			_pointerInfo.getPointerShape());
	}
	if (_pointerInfo.isChanged(PIID_POINTERCOLOR))
	{
		const flByte* color;
		color = _pointerInfo.getPointerColor();
		flNotify::post(flNotify::L_INFO, "PointerInfo:: PointerColor=(r:%d g:%d b:%d)", 
			color[0], color[1], color[2]);
	}
	if (_pointerInfo.isChanged(PIID_SCREENSIZE))
	{
		flUShort width, height;
		_pointerInfo.getScreenSize(width, height);
		flNotify::post(flNotify::L_INFO, "PointerInfo:: ScreenSize(%d,%d)", 
			width, height);
	}
	if (_pointerInfo.isChanged(PIID_POINTERPOSITION))
	{
		flUShort x, y;
		_pointerInfo.getPointerPosition(x, y);
		flNotify::post(flNotify::L_INFO, "PointerInfo:: PointerPosition(%d,%d)", 
			x, y);
	}
	if (_pointerInfo.isChanged(PIID_BUTTONSTATUS))
	{
		flUShort buttonStatus = _pointerInfo.getButtonStatus();
		flNotify::post(flNotify::L_INFO, "PointerInfo:: ButtonStatus=%x", 
			buttonStatus);
	}
	if (_pointerInfo.isChanged(PIID_WHEELSTATUS))
	{
		flUShort wheelStatus = _pointerInfo.getWheelStatus();
		flNotify::post(flNotify::L_INFO, "PointerInfo:: WheelStatus=%x", 
			wheelStatus);
	}
	if (_pointerInfo.isChanged(PIID_DRAWSTATUS))
	{
		flBool drawLocusOn = _pointerInfo.getDrawStatusDrawLocusOn();
		flBool showPhraseOn = _pointerInfo.getDrawStatusShowPhraseOn();
		flBool drawLocus = _pointerInfo.getDrawStatusDrawLocus();
		flBool eraseLocus = _pointerInfo.getDrawStatusEraseLocus();
		flBool showPhrase = _pointerInfo.getDrawStatusShowPhrase();
		flBool edgingOn = _pointerInfo.getDrawStatusLocusEdgingOn();
		flBool brokenLineOn = _pointerInfo.getDrawStatusLocusBrokenLineOn();
		flUShort locusWidth = _pointerInfo.getDrawStatusLocusWidth();
		flNotify::post(flNotify::L_INFO, "PointerInfo:: DrawLoOn(%d) ShowPhrOn(%d) \
DrawLo(%d) EraseLo(%d) ShowPhr(%d) EdgeOn(%d) BrkLnOn(%d) LoWidth(%d)",
			drawLocusOn, showPhraseOn, drawLocus, eraseLocus, showPhrase, edgingOn, brokenLineOn, locusWidth);
	}
	if (_pointerInfo.isChanged(PIID_PHRASE))
	{
		flNotify::post(flNotify::L_INFO, "PointerInfo:: Phrase = (%s)",
			_pointerInfo.getPhrase());
	}
	if (_pointerInfo.isChanged(PIID_NORMALIZEDORIGIN))
	{
		flFloat ox, oy;
		_pointerInfo.getNormalizedOrigin(ox, oy);
		flNotify::post(flNotify::L_INFO, "PointerInfo:: NormalizedOrigin(%5.3f,%5.3f)", 
			ox, oy);
	}
	if (_pointerInfo.isChanged(PIID_NORMALIZEDSIZE))
	{
		flFloat ow, oh;
		_pointerInfo.getNormalizedSize(ow, oh);
		flNotify::post(flNotify::L_INFO, "PointerInfo:: NormalizedSize(%5.3f,%5.3f)", 
			ow, oh);
	}
	if (_pointerInfo.isChanged(PIID_NORMALIZEDPOINTERPOSITION))
	{
		flFloat cx, cy;
		_pointerInfo.getNormalizedPointerPosition(cx, cy);
		flNotify::post(flNotify::L_INFO, "PointerInfo:: NormalizedPointerPosition(%5.3f,%5.3f)", 
			cx, cy);
	}
	if (_pointerInfo.isChanged(PIID_USERDATA))
	{
		flByte *userData;
		flUInt size;

		_pointerInfo.getUserData(userData, size);

		flNotify::post(flNotify::L_INFO, "PointerInfo:: UserData(%s) size[%d]", 
			userData, size);
	}
*/
#endif

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInfoInputPin::EndOfStream(void)
{
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInfoInputPin::BeginFlush(void)
{
	return CBaseInputPin::BeginFlush();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInfoInputPin::EndFlush(void)
{
    return CBaseInputPin::EndFlush();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerInfoInputPin::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate)
{
    return CBasePin::NewSegment(tStart, tStop, dRate);
}
//------------------------------------------------------------------------------
