/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flPointerMixer/PointerInfoInputPin.h>
#include <flPointerMixer/PointerMixer.h>
#include <flPointerMixer/PointerMixerProp.h>
#include <flPointerMixer/PointerMixerExtProp.h>
#include <flPointerMixer/flBitmap.h>
#include <flTypes/IPointerPosition.h>
#include <flBase/flImageIO.h>
#include <flBase/flNotify.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudPinStreamTypes =
{
    &MEDIATYPE_Stream,		// MajorType
    &MEDIASUBTYPE_NULL		// MinorType
};
//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudPinVideoTypes =
{
    &MEDIATYPE_Video,		// MajorType
    &MEDIASUBTYPE_NULL		// MinorType
};
//------------------------------------------------------------------------------
const AMOVIESETUP_PIN sudPins [] =
{
	{
		L"In",				// The Pins name
		FALSE,				// Is rendered
		FALSE,				// Is an output pin
		FALSE,				// Allowed none
		FALSE,				// Allowed many
		&CLSID_NULL,		// Connects to filter
		NULL,				// Connects to pin
		1,					// Number of types
		&sudPinVideoTypes	// Pin details
	},
	{
		L"Out",				// The Pins name
		FALSE,				// Is rendered
		TRUE,				// Is an output pin
		FALSE,				// Allowed none
		FALSE,				// Allowed many
		&CLSID_NULL,		// Connects to filter
		NULL,				// Connects to pin
		1,					// Number of types
		&sudPinVideoTypes	// Pin details
	}
};
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudPointerMixer =
{
    &CLSID_PointerMixer,	// Filter CLSID
    L"NIME Video/Pointer Mixer",		// String name
    MERIT_DO_NOT_USE,		// Filter merit
    2,						// Number of pins
    sudPins					// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[3] =
{
    { L"PointerMixer"
    , &CLSID_PointerMixer
    , CPointerMixer::CreateInstance
    , NULL
    , &sudPointerMixer }
  ,
    { L"PointerMixerProp"
    , &CLSID_PointerMixerPropertyPage
    , CPointerMixerProp::CreateInstance }
  ,
    { L"PointerMixerExtProp"
    , &CLSID_PointerMixerExtensionPropertyPage
    , CPointerMixerExtProp::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CPointerMixer::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
    CPointerMixer *pNewObject = new CPointerMixer(NAME("PointerMixer"), punk, phr);
    if (pNewObject == NULL)
	{
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CPointerMixer::CPointerMixer(TCHAR *tszName, LPUNKNOWN pUnk, HRESULT *phr) :
CTransInPlaceFilter(tszName, pUnk, CLSID_PointerMixer, phr),
CPersistStream(pUnk, phr)
{
	ASSERT(tszName != NULL);
	ASSERT(phr != NULL);

	// Initialize Shape Draw 
	_pointerDraws.setGrowSize(16);
	_locusDraws.setGrowSize(16);
	_phraseDraws.setGrowSize(16);
	createDefaultPointerDraws();

	// Initialize Dynamic input pins
	InitPointerInfoInputPins();

	// Create First input pin
	CreateInputPin(this);

	_windowPositionX	= 0.0f;
	_windowPositionY	= 0.0f;
	_windowWidth		= 1.0f;
	_windowHeight		= 1.0f;
	_mappingPositionX	= 0.0f;
	_mappingPositionY	= 0.0f;
	_mappingWidth		= 1.0f;
	_mappingHeight		= 1.0f;
}
//------------------------------------------------------------------------------
CPointerMixer::~CPointerMixer()
{
	InitPointerInfoInputPins();

	for(flUInt i = 0; i < _pointerDraws.getSize(); i++)
	{
		delete _pointerDraws[i];
		_pointerDraws.removeIndex(i--);
	}
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
    CheckPointer(ppv, E_POINTER);

    if (riid == IID_IPointerMixer)
	{
        return GetInterface((IPointerMixer *) this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    }
	else
	{
		return CTransInPlaceFilter::NonDelegatingQueryInterface(riid, ppv);
    }
}
//------------------------------------------------------------------------------
int
CPointerMixer::GetPinCount()
{
	return 2 + _pointerInfoInputPins.getSize();
}
//------------------------------------------------------------------------------
CBasePin*
CPointerMixer::GetPin(int n)
{
	if (m_pInput == NULL || m_pOutput == NULL)
	{
		HRESULT hr = S_OK;

		m_pInput = new CTransInPlaceInputPin(NAME("TransInPlace input pin"),
											this, &hr, L"Input");

		if (FAILED(hr) || m_pInput == NULL) 
			goto PINCREATEERROR;

		m_pOutput = new CTransInPlaceOutputPin(NAME("TransInPlace output pin"),
											this, &hr, L"Output");

		if (FAILED(hr) || m_pOutput == NULL) 
			goto PINCREATEERROR;
	}

	if (n == 0)
		return m_pInput;
	else if (n == 1)
		return m_pOutput;
	else if (n - 2 < (flInt)_pointerInfoInputPins.getSize())
		return _pointerInfoInputPins[n - 2];

	return NULL;

PINCREATEERROR:
	delete m_pInput;	m_pInput	= NULL;
	delete m_pOutput;	m_pOutput	= NULL;
	return NULL;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::FindPin(LPCWSTR Id, IPin **ppPin)
{
    CheckPointer(ppPin,E_POINTER);
    ValidateReadWritePtr(ppPin, sizeof(IPin *));

    if (0 == lstrcmpW(Id, L"In"))
	{
        *ppPin = GetPin(0);
    }
    else if (0 == lstrcmpW(Id, L"Out"))
	{
        *ppPin = GetPin(1);
    }
	else if (0 == lstrcmpW(Id, L"PointerInfo"))
	{
		*ppPin = GetPin(_pointerInfoInputPins.getSize() - 1 + 2);
	}
    else
	{
        *ppPin = NULL;
        return VFW_E_NOT_FOUND;
    }

    HRESULT hr = NOERROR;
    if(*ppPin)
	{
        (*ppPin)->AddRef();
    }
    else
	{
        hr = E_OUTOFMEMORY;
    }
    return hr;
}
//------------------------------------------------------------------------------
HRESULT
CPointerMixer::SetMediaType(PIN_DIRECTION direction, const CMediaType *pmt)
{
	// Setup ShapeDraw Video size
	if (direction == PINDIR_OUTPUT)
	{
		ASSERT(IsEqualGUID(FORMAT_VideoInfo, *pmt->FormatType()) == TRUE);

		VIDEOINFO *pvi = (VIDEOINFO *)pmt->Format();

		ASSERT(pvi->bmiHeader.biBitCount == 24);

		_videoWidth = pvi->bmiHeader.biWidth;
		_videoHeight = pvi->bmiHeader.biHeight;

		for(flUInt i = 0; i < _pointerDraws.getSize(); i++)
		{
			_pointerDraws[i]->setVideoSize(_videoWidth, _videoHeight);
		}

		for(flUInt i = 0; i < _pointerInfoInputPins.getSize(); i++)
		{
			_locusDraws[i]->setVideoSize(_videoWidth, _videoHeight);
			_phraseDraws[i]->setVideoSize(_videoWidth, _videoHeight);
		}
	}

	// Reconnect Output PIN
    if (m_pInput->IsConnected() && m_pOutput->IsConnected())
	{
        FILTER_INFO fInfo;

        QueryFilterInfo( &fInfo );

        if (direction == PINDIR_INPUT && *pmt != m_pOutput->CurrentMediaType())
            fInfo.pGraph->Reconnect( m_pOutput );

        QueryFilterInfoReleaseGraph( fInfo );
    }


	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CPointerMixer::Transform(IMediaSample *pSample)
{
	flULong dataLength;
	flByte *data;
	
	pSample->GetPointer(&data);
	dataLength = pSample->GetActualDataLength();

	// Draw Pointer
	drawLowerLayer(data);
	drawUpperLayer(data);
	updatePinValid();

	return NOERROR;
	return NOERROR;
}
//------------------------------------------------------------------------------
// CTransformFilter pure virtual 
HRESULT
CPointerMixer::CheckInputType(const CMediaType *mtIn)
{
	if (IsEqualGUID(mtIn->majortype, MEDIATYPE_Video) &&
		IsEqualGUID(mtIn->subtype, MEDIASUBTYPE_RGB24) &&
		IsEqualGUID(mtIn->formattype, FORMAT_VideoInfo))
		return S_OK;

	return VFW_E_TYPE_NOT_ACCEPTED;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::GetClassID(CLSID *pClsid)
{
	*pClsid = CLSID_PointerMixer;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CPointerMixer::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	WRITEOUT(_windowPositionX);
	WRITEOUT(_windowPositionY);
	WRITEOUT(_windowWidth);
	WRITEOUT(_windowHeight);
	WRITEOUT(_mappingPositionX);
	WRITEOUT(_mappingPositionY);
	WRITEOUT(_mappingWidth);
	WRITEOUT(_mappingHeight);

	flInt numUserPointers = _pointerDraws.getSize() - IPointerPosition::NUM_POINTERSHAPE;
	assert(0 <= numUserPointers);
	WRITEOUT(numUserPointers);
	for(flInt i = 0, j = IPointerPosition::NUM_POINTERSHAPE; i < numUserPointers; i++, j++)
	{
		const flString& name = _pointerDraws[j]->getName();
		flUInt nameLen = name.length() + 1;
		WRITEOUT(nameLen);
		WRITENOUT(name.c_str(), nameLen);
		
		flInt orgX, orgY;
		_pointerDraws[j]->getOrigin(orgX, orgY);
		WRITEOUT(orgX);
		WRITEOUT(orgY);
	}
	
	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CPointerMixer::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	READIN(_windowPositionX);
	READIN(_windowPositionY);
	READIN(_windowWidth);
	READIN(_windowHeight);
	READIN(_mappingPositionX);
	READIN(_mappingPositionY);
	READIN(_mappingWidth);
	READIN(_mappingHeight);

	flInt numUserPointers;
	READIN(numUserPointers);
	for(flInt i = 0; i < numUserPointers; i++)
	{
		flUInt nameLen;
		flChar name[1024];

		READIN(nameLen);
		READNIN(name, nameLen);

		flInt orgX, orgY;
		READIN(orgX);
		READIN(orgY);

		flImageIO imageIO;
		flImage image = imageIO.load(flString(name));
		if (image.width() == 0 ||
			image.height() == 0 ||
			PM_MAXIMUM_POINTERSHAPEWIDTH < image.width() ||
			PM_MAXIMUM_POINTERSHAPEHEIGHT < image.height() ||
			image.components() != 3)
			continue;

		image.swapColor();

		_pointerDraws.add(new flPointerDraw(image, flString(name), orgX, orgY));
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CPointerMixer::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CPointerMixer::SizeMax()
{
	return	sizeof(flFloat) * 8;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages
STDMETHODIMP
CPointerMixer::GetPages(CAUUID *pPages)
{
	pPages->cElems = 2;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 2);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_PointerMixerPropertyPage;
	pPages->pElems[1] = CLSID_PointerMixerExtensionPropertyPage;

	return NOERROR;
}
//------------------------------------------------------------------------------
// IPointerMixer
STDMETHODIMP
CPointerMixer::get_WindowPosition(flFloat *x, flFloat *y)
{
	CAutoLock autoLock(&_filterLock);
	*x = _windowPositionX;
	*y = _windowPositionY;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::put_WindowPosition(flFloat x, flFloat y)
{
	CAutoLock autoLock(&_filterLock);
	_windowPositionX = x;
	_windowPositionY = y;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::get_WindowSize(flFloat *width, flFloat *height)
{
	CAutoLock autoLock(&_filterLock);
	*width = _windowWidth;
	*height = _windowHeight;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::put_WindowSize(flFloat width, flFloat height)
{
	CAutoLock autoLock(&_filterLock);
	_windowWidth = width;
	_windowHeight = height;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::get_MappingPosition(flFloat *x, flFloat *y)
{
	CAutoLock autoLock(&_filterLock);
	*x = _mappingPositionX;
	*y = _mappingPositionY;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::put_MappingPosition(flFloat x, flFloat y)
{
	CAutoLock autoLock(&_filterLock);
	_mappingPositionX = x;
	_mappingPositionY = y;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::get_MappingSize(flFloat *width, flFloat *height)
{
	CAutoLock autoLock(&_filterLock);
	*width = _mappingWidth;
	*height = _mappingHeight;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::put_MappingSize(flFloat width, flFloat height)
{
	CAutoLock autoLock(&_filterLock);
	_mappingWidth = width;
	_mappingHeight = height;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::get_NumPointerInput(flUInt* numPointerInput)
{
	CAutoLock autoLock(&_filterLock);
	*numPointerInput = _pointerInfoInputPins.getSize() - 1;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::get_NumPointerShape(flUInt *count)
{
	CAutoLock autoLock(&_filterLock);
	*count = _pointerDraws.getSize();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::add_PointerShape(const flChar * filename)
{
	CAutoLock autoLock(&_filterLock);

	flImageIO imageIO;
	flImage image = imageIO.load(flString(filename));
	if (image.width() == 0 ||
		image.height() == 0 ||
		PM_MAXIMUM_POINTERSHAPEWIDTH < image.width() ||
		PM_MAXIMUM_POINTERSHAPEHEIGHT < image.height() ||
		image.components() != 3)
		return E_FAIL;

	image.swapColor();

	_pointerDraws.add(new flPointerDraw(_videoWidth, _videoHeight, image, 
						flString(filename), 0, 0));
	
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::remove_PointerShape(flUInt index)
{
	CAutoLock autoLock(&_filterLock);
	if (index < IPointerPosition::NUM_POINTERSHAPE || _pointerDraws.getSize() <= index)
		return E_FAIL;
	_pointerDraws.removeIndex(index);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::get_PointerShapeName(flUInt index, flChar * shapename)
{
	CAutoLock autoLock(&_filterLock);
	if (shapename == NULL || _pointerDraws.getSize() <= index)
		return NOERROR;
	strcpy(shapename, _pointerDraws[index]->getName().c_str());
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::put_PointerShapeName(flUInt index, const flChar * shapename)
{
	CAutoLock autoLock(&_filterLock);
	if (shapename == NULL || _pointerDraws.getSize() <= index)
		return NOERROR;

	flString sname = flString(shapename);;

	if (PM_MAXIMUM_POINTERSHAPENAME_LENGTH - 1 <= sname.length())
		sname = sname.subString(0, PM_MAXIMUM_POINTERSHAPENAME_LENGTH - 1);

	_pointerDraws[index]->setName(sname);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::get_PointerShapeOrigin(flUInt index, flInt* originX, flInt* originY)
{
	CAutoLock autoLock(&_filterLock);
	if (_pointerDraws.getSize() <= index)
		return NOERROR;
	_pointerDraws[index]->getOrigin(*originX, *originY);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerMixer::put_PointerShapeOrigin(flUInt index, flInt originX, flInt originY)
{
	CAutoLock autoLock(&_filterLock);
	if (_pointerDraws.getSize() <= index)
		return NOERROR;
	_pointerDraws[index]->setOrigin(originX, originY);
	return NOERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
void
CPointerMixer::InitPointerInfoInputPins()
{
	_nextInputPinNumber = 0;
	for(flUInt i = 0; i < _pointerInfoInputPins.getSize(); i++)
	{
		CPointerInfoInputPin* pPin = _pointerInfoInputPins[i];
		pPin->Release();
	}
	_pointerInfoInputPins.setSize(0);
	_pointerInfoInputPins.setBufferSize(16);
	_pointerInfoInputPins.setGrowSize(16);
}
//------------------------------------------------------------------------------
void
CPointerMixer::CreateInputPin(CPointerMixer* owner)
{
	WCHAR szbuf[20];
	HRESULT hr = NOERROR;

	_nextInputPinNumber++;
	swprintf(szbuf, L"Input%d", _nextInputPinNumber);

	CPointerInfoInputPin *pPin = new CPointerInfoInputPin(
							NAME("PointerInfo Input"), owner, &hr, szbuf);

	if (FAILED(hr) || pPin == NULL)
	{
		delete pPin;
		return ;
	}

	pPin->AddRef();
	_pointerInfoInputPins.add(pPin);
	IncrementPinVersion();

	// Setup ShapeDraw
	_locusDraws.add(new flLocusDraw(_videoWidth, _videoHeight));
	_phraseDraws.add(new flTextDraw(_videoWidth, _videoHeight));
}
//------------------------------------------------------------------------------
void
CPointerMixer::DeleteInputPin(CPointerInfoInputPin* inputPin)
{
	for(flUInt i = 0; i < _pointerInfoInputPins.getSize(); i++)
	{
		CPointerInfoInputPin* pPin = _pointerInfoInputPins[i];

		if (pPin == inputPin)
		{
			delete pPin;
			_pointerInfoInputPins.removeIndex(i);

			IncrementPinVersion();

			delete _locusDraws[i];
			_locusDraws.removeIndex(i);

			delete _phraseDraws[i];
			_phraseDraws.removeIndex(i);

			break;
		}
	}
}
//------------------------------------------------------------------------------
flUInt
CPointerMixer::GetNumFreePins()
{
	flUInt n = 0;

	for(flUInt i = 0; i < _pointerInfoInputPins.getSize(); i++)
	{
		CPointerInfoInputPin* pPin = _pointerInfoInputPins[i];
		if (pPin->m_Connected == NULL)
			n++;
	}

    return n;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
void
CPointerMixer::updatePinValid()
{
	for(flUInt i = 0; i < _pointerInfoInputPins.getSize() - 1; i++)
	{
		CAutoLock autoLock(&_pointerInfoInputPins[i]->_pinLock);

		// Update Pin Valid
		flULong currentTime = timeGetTime();
		flULong receiveTime = _pointerInfoInputPins[i]->_pinReceiveTime;
		if (POINTERLOSTTIME < currentTime - receiveTime)
			_pointerInfoInputPins[i]->_pinValid = false;
	}
}
//------------------------------------------------------------------------------
void
CPointerMixer::drawUpperLayer(BYTE *data)
{
	// PointerShape, Phrase
	for(flUInt i = 0; i < _pointerInfoInputPins.getSize() - 1; i++)
	{
		flBool valid;

		flShort pointerShape;
		flByte pointerColor[3];
		flChar phrase[PP_MAXIMUM_PHRASE_LENGTH];
		flFloat cx, cy;
		flBool showPhrase;

		{
			CAutoLock autoLock(&_pointerInfoInputPins[i]->_pinLock);

			// Pin Valid
			valid = _pointerInfoInputPins[i]->_pinValid;

			const flPointerInfo& pointerInfo = _pointerInfoInputPins[i]->_pointerInfo;

			// Pointer Shape
			pointerShape = pointerInfo.getPointerShape();
			// Pointer Color
			memcpy(pointerColor, pointerInfo.getPointerColor(), sizeof(pointerColor));
			// Phrase
			strcpy(phrase, pointerInfo.getPhrase());
			// Normalized Pointer Position
			pointerInfo.getNormalizedPointerPosition(cx, cy);
			// Draw Status
			showPhrase = pointerInfo.getDrawStatusShowPhrase();
		}

		if (!valid)
			continue;

		// Compute Actual Pointer Coordinates
		flFloat nx = (cx - _windowPositionX) / _windowWidth; // nx, ny - Normalized Coordinate
		flFloat ny = (cy - _windowPositionY) / _windowHeight;
		flInt x = flInt(_videoWidth * nx);
		flInt y = flInt(_videoHeight * ny);

		flInt textX = x;
		flInt textY = y;

		// Draw
		if ((flUInt)pointerShape < _pointerDraws.getSize())
		{
			_pointerDraws[pointerShape]->setPosition(x, y);
			_pointerDraws[pointerShape]->setColor(pointerColor);
			_pointerDraws[pointerShape]->draw(data);

			const flImage& image = _pointerDraws[pointerShape]->getImage();
			flInt orgX, orgY;
			_pointerDraws[pointerShape]->getOrigin(orgX, orgY);
			textX += image.width();
			textX -= orgX;
			textY -= orgY;
		}

		if (showPhrase)
		{
			_phraseDraws[i]->setPosition(textX, textY);
			_phraseDraws[i]->setColor(pointerColor);
			_phraseDraws[i]->setText(phrase);
			_phraseDraws[i]->draw(data);
		}
	}
}
//------------------------------------------------------------------------------
void
CPointerMixer::drawLowerLayer(BYTE *data)
{
	ASSERT(0 < _pointerInfoInputPins.getSize());

	// Locus
	for(flUInt i = 0; i < _pointerInfoInputPins.getSize() - 1; i++)
	{
		flBool valid;

		flByte pointerColor[3];
		flFloat cx, cy;
		flBool drawLocusOn;
		flBool drawLocus;
		flBool eraseLocus;
		flBool locusEdgingOn;
		flBool locusBrokenLineOn;
		flUInt locusWidth;

		{
			CAutoLock autoLock(&_pointerInfoInputPins[i]->_pinLock);

			// Pin Valid
			valid = _pointerInfoInputPins[i]->_pinValid;

			const flPointerInfo& pointerInfo = _pointerInfoInputPins[i]->_pointerInfo;
			
			// Pointer Color
			memcpy(pointerColor, pointerInfo.getPointerColor(), sizeof(pointerColor));
			// Normalized Pointer Position
			pointerInfo.getNormalizedPointerPosition(cx, cy);
			// Draw Status
			drawLocusOn = pointerInfo.getDrawStatusDrawLocusOn();
			drawLocus = pointerInfo.getDrawStatusDrawLocus();
			eraseLocus = pointerInfo.getDrawStatusEraseLocus();
			locusEdgingOn = pointerInfo.getDrawStatusLocusEdgingOn();
			locusBrokenLineOn = pointerInfo.getDrawStatusLocusBrokenLineOn();
			locusWidth = pointerInfo.getDrawStatusLocusWidth();
		}

		if (!valid)
			continue;

		if (!drawLocusOn)
		{
			_locusDraws[i]->clearPoints();
			continue ;
		}

		// Compute Actual Pointer Coordinates
		flFloat nx = (cx - _windowPositionX) / _windowWidth; // nx, ny - Normalized Coordinate
		flFloat ny = (cy - _windowPositionY) / _windowHeight;
		flInt x = flInt(_videoWidth * nx);
		flInt y = flInt(_videoHeight * ny);

		// Draw
		if (eraseLocus)
			_locusDraws[i]->clearPoints();
		
		if (drawLocus)
			_locusDraws[i]->addPoint(x, y);
		else
			_locusDraws[i]->addDiscontinuousPoint();

		_locusDraws[i]->setColor(pointerColor);
		_locusDraws[i]->setEdgingOn(locusEdgingOn);
		_locusDraws[i]->setBrokenLineOn(locusBrokenLineOn);
		_locusDraws[i]->setLineWidth(locusWidth);
		_locusDraws[i]->draw(data);
	}
}
//------------------------------------------------------------------------------
void
CPointerMixer::createDefaultPointerDraws()
{
	// Standard Pointer
	_pointerDraws.add(
		new flPointerDraw(
			flImage(PSTANDARD_WIDTH, PSTANDARD_HEIGHT, 3, PStandard),
			PStandardName,
			PSTANDARD_ORGX,
			PSTANDARD_ORGY
			)
		);

	// Arrow Pointer
	_pointerDraws.add(
		new flPointerDraw(
			flImage(PARROW_WIDTH, PARROW_HEIGHT, 3, PArrow),
			PArrowName,
			PARROW_ORGX,
			PARROW_ORGY
			)
		);

	// Circle Pointer
	_pointerDraws.add(
		new flPointerDraw(
			flImage(PCIRCLE_WIDTH, PCIRCLE_HEIGHT, 3, PCircle),
			PCircleName,
			PCIRCLE_ORGX,
			PCIRCLE_ORGY
			)
		);

	// Rectangle Pointer
	_pointerDraws.add(
		new flPointerDraw(
			flImage(PRECTANGLE_WIDTH, PRECTANGLE_HEIGHT, 3, PRectangle),
			PRectangleName,
			PRECTANGLE_ORGX,
			PRECTANGLE_ORGY
			)
		);

	// Target Pointer
	_pointerDraws.add(
		new flPointerDraw(
			flImage(PTARGET_WIDTH, PTARGET_HEIGHT, 3, PTarget),
			PTargetName,
			PTARGET_ORGX,
			PTARGET_ORGY
			)
		);

	// Star Pointer
	_pointerDraws.add(
		new flPointerDraw(
			flImage(PSTAR_WIDTH, PSTAR_HEIGHT, 3, PStar),
			PStarName,
			PSTAR_ORGX,
			PSTAR_ORGY
			)
		);
}
//------------------------------------------------------------------------------
