/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flPointerPosition/PointerPosition.h>
#include <flPointerPosition/PointerPositionProp.h>
#include <flPointerPosition/PointerPositionExtProp.h>
#include <flPointerPosition/PointerPositionUserDataProp.h>
#include <flPointerPosition/PointerPositionStream.h>
#include <flBase/flNotify.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudOpPinTypes =
{
    &MEDIATYPE_NULL,              // MajorType
    &MEDIASUBTYPE_NULL            // MinorType
};
//------------------------------------------------------------------------------
const AMOVIESETUP_PIN sudOpPin =
{
	L"Out",								// The Pins name
	FALSE,								// Is rendered
	TRUE,								// Is an output pin
	FALSE,								// Allowed none
	FALSE,								// Allowed many
	&CLSID_NULL,						// Connects to filter
	NULL,								// Connects to pin
	1,									// Number of types
	&sudOpPinTypes						// Pin details
};
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudPointerPosition =
{
    &CLSID_PointerPosition,				// Filter CLSID
    L"NIME Pointer Position Capture",	// String name
    MERIT_DO_NOT_USE,					// Filter merit
    1,									// Number of pins
    &sudOpPin							// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[4] =
{
    { L"PointerPosition"
    , &CLSID_PointerPosition
    , CPointerPosition::CreateInstance
    , NULL
    , &sudPointerPosition }
  ,
    { L"PointerPositionProp"
    , &CLSID_PointerPositionPropertyPage
    , CPointerPositionProp::CreateInstance }
  ,
    { L"PointerPositionExtProp"
    , &CLSID_PointerPositionExtensionPropertyPage
    , CPointerPositionExtProp::CreateInstance }
  ,
    { L"PointerPositionUserDataProp"
    , &CLSID_PointerPositionUserDataPropertyPage
    , CPointerPositionUserDataProp::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CPointerPosition::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
    CPointerPosition *pNewObject = new CPointerPosition(NAME("PointerPosition"), punk, phr);
    if (pNewObject == NULL)
	{
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CPointerPosition::CPointerPosition(TCHAR *tszName, LPUNKNOWN pUnk, HRESULT *phr) :
CSource(tszName, pUnk, CLSID_PointerPosition, phr),
CPersistStream(pUnk, phr)
{
	ASSERT(tszName != NULL);
	ASSERT(phr != NULL);

	CAutoLock cAutoLock(&m_cStateLock);

	m_paStreams = new CSourceStream*[1];
	if (m_paStreams == NULL)
	{
		*phr = E_OUTOFMEMORY;
		return ;
	}

	m_paStreams[0] = new CPointerPositionStream(phr, this, L"Out");
	if (m_paStreams[0] == NULL)
	{
		*phr = E_OUTOFMEMORY;
		return;
	}
	_pointerPositionStream = (CPointerPositionStream *)m_paStreams[0];


	// Direct Input
	flDirectInput::initialize("PointerPosition.ax");
	_mouse = NULL;
	flMouseEnumerator enumerator;
	if (enumerator.hasMoreDevices())
		_mouse = (flMouse *)enumerator.nextDevice();
	if (_mouse != NULL)
	{
		_mouse->setForeground(false);
		_mouse->setExclusive(false);
	}

	// Filter Status
	_isRunning				= false;

	// Properties
	_outputMinimumPeriod = 30;
	_pointerShape			= PPPS_STANDARD;
	_pointerColor[0]		= 0xff;
	_pointerColor[1]		= 0xff;
	_pointerColor[2]		= 0xff;
	_normalizedOrigin[0]	= 0.0f;
	_normalizedOrigin[1]	= 0.0f;
	_normalizedSize[0]		= 1.0f;
	_normalizedSize[1]		= 1.0f;
	_drawLocusOn			= false;
	_showPhraseOn			= false;
	_phraseDatas.setSize(0);
	_phraseDatas.setGrowSize(16);
	_userDataSize			= 0;
	memset(_userData, 0, sizeof(_userData));
}
//------------------------------------------------------------------------------
CPointerPosition::~CPointerPosition()
{
	// Delete Mouse
	delete _mouse;

	// Shutdown Direct Input 
	flDirectInput::shutdown();
}
//------------------------------------------------------------------------------
// Reveals IPointerPosition & ISpecifyPropertyPages
STDMETHODIMP
CPointerPosition::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IPointerPosition)
	{
        return GetInterface((IPointerPosition *) this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    }
	else
	{
		return CSource::NonDelegatingQueryInterface(riid, ppv);
    }
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// IMediaFilter
STDMETHODIMP
CPointerPosition::Run(REFERENCE_TIME startTime)
{
	_isRunning			= true;

	HRESULT hr = CSource::Run(startTime);
	if (FAILED(hr))
		return hr;

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::Pause()
{
    return CSource::Pause();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::Stop()
{
	_isRunning			= false;

	HRESULT hr = CSource::Stop();
	if (FAILED(hr))
		return hr;

	return hr;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_PointerPosition;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CPointerPosition::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	flUInt size;
	
	WRITEOUT(_outputMinimumPeriod);
	WRITEOUT(_pointerShape);
	WRITENOUT(_pointerColor, sizeof(_pointerColor));
	WRITENOUT(_normalizedOrigin, sizeof(_normalizedOrigin));
	WRITENOUT(_normalizedSize, sizeof(_normalizedSize));
	WRITEOUT(_drawLocusOn);
	WRITEOUT(_showPhraseOn);
	WRITEOUT(_locusData);
	size = _phraseDatas.getSize();
	WRITEOUT(size);
	if (size != 0)
	{
		WRITENOUT(_phraseDatas.getPointer(), sizeof(flPhraseData) * size);
	}
	WRITEOUT(_userDataSize);
	if (_userDataSize != 0)
	{
		WRITENOUT(_userData, _userDataSize);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CPointerPosition::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	flUInt size;

	READIN(_outputMinimumPeriod);
	READIN(_pointerShape);
	READNIN(_pointerColor, sizeof(_pointerColor));
	READNIN(_normalizedOrigin, sizeof(_normalizedOrigin));
	READNIN(_normalizedSize, sizeof(_normalizedSize));
	READIN(_drawLocusOn);
	READIN(_showPhraseOn);
	READIN(_locusData);
	READIN(size);
	_phraseDatas.setSize(size);
	if (size != 0)
	{
		READNIN((void *)_phraseDatas.getPointer(), sizeof(flPhraseData) * size);
	}
	READIN(_userDataSize);
	if (_userDataSize != 0)
	{
		READNIN(_userData, _userDataSize);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CPointerPosition::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CPointerPosition::SizeMax()
{
	return	sizeof(flULong) +
		sizeof(flShort) +
		sizeof(flByte) * 3 +
		sizeof(flFloat) * 4 + 
		sizeof(flLocusData) + 
		sizeof(flUInt) + 
		sizeof(flPhraseData) * 100;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages method
STDMETHODIMP
CPointerPosition::GetPages(CAUUID *pPages)
{
	pPages->cElems = 3;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 3);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_PointerPositionPropertyPage;
	pPages->pElems[1] = CLSID_PointerPositionExtensionPropertyPage;
	pPages->pElems[2] = CLSID_PointerPositionUserDataPropertyPage;

	return NOERROR;
}
//------------------------------------------------------------------------------
// IPointerPosition
// Monitor
STDMETHODIMP
CPointerPosition::get_ScreenSize(flUShort *width, flUShort *height)
{
	CAutoLock autoLock(&_filterLock);
	_pointerInfo.getScreenSize(*width, *height);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_PointerPosition(flUShort *x, flUShort *y)
{
	CAutoLock autoLock(&_filterLock);
	_pointerInfo.getPointerPosition(*x, *y);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_ButtonStatus(flUShort *status)
{
	CAutoLock autoLock(&_filterLock);
	*status = _pointerInfo.getButtonStatus();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_WheelStatus(flUShort *status)
{
	CAutoLock autoLock(&_filterLock);
	*status = _pointerInfo.getWheelStatus();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_NormalizedPointerPosition(flFloat *x, flFloat *y)
{
	CAutoLock autoLock(&_filterLock);
	_pointerInfo.getNormalizedPointerPosition(*x, *y);
	return NOERROR;
}
//------------------------------------------------------------------------------
// Standard Settings
STDMETHODIMP
CPointerPosition::get_OutputMinimumPeriod(flULong *period)
{
	CAutoLock autoLock(&_filterLock);
	*period = _outputMinimumPeriod;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_OutputMinimumPeriod(flULong period)
{
	if (period < PP_MINIMUM_OUTPUTMINIMUM_PERIOD)
		period = PP_MINIMUM_OUTPUTMINIMUM_PERIOD;
	else if (PP_MAXIMUM_OUTPUTMINIMUM_PERIOD < period)
		period = PP_MAXIMUM_OUTPUTMINIMUM_PERIOD;
	CAutoLock autoLock(&_filterLock);
	_outputMinimumPeriod = period;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_PointerShape(flShort *pointerShape)
{
	CAutoLock autoLock(&_filterLock);
	*pointerShape = _pointerShape;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_PointerShape(flShort pointerShape)
{
	CAutoLock autoLock(&_filterLock);
	_pointerShape = pointerShape;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_PointerColor(flByte *color)
{
	CAutoLock autoLock(&_filterLock);
	memcpy(color, _pointerColor, sizeof(_pointerColor));
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_PointerColor(const flByte *color)
{
	CAutoLock autoLock(&_filterLock);
	memcpy(_pointerColor, color, sizeof(_pointerColor));
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_NormalizedOrigin(flFloat *ox, flFloat *oy)
{
	CAutoLock autoLock(&_filterLock);
	*ox = _normalizedOrigin[0];
	*oy = _normalizedOrigin[1];
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_NormalizedOrigin(flFloat ox, flFloat oy)
{
	CAutoLock autoLock(&_filterLock);
	_normalizedOrigin[0] = ox;
	_normalizedOrigin[1] = oy;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_NormalizedSize(flFloat *ow, flFloat *oh)
{
	CAutoLock autoLock(&_filterLock);
	*ow = _normalizedSize[0];
	*oh = _normalizedSize[1];
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_NormalizedSize(flFloat ow, flFloat oh)
{
	CAutoLock autoLock(&_filterLock);
	_normalizedSize[0] = ow;
	_normalizedSize[1] = oh;
	return NOERROR;
}
//------------------------------------------------------------------------------
// Locus Settings
STDMETHODIMP
CPointerPosition::get_DrawLocusOn(flBool *flag)
{
	CAutoLock autoLock(&_filterLock);
	*flag = _drawLocusOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_DrawLocusOn(flBool flag)
{
	CAutoLock autoLock(&_filterLock);
	_drawLocusOn = flag;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_LocusDrawButton(flUShort* buttonStatus)
{
	CAutoLock autoLock(&_filterLock);
	*buttonStatus = _locusData.DrawButton;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_LocusDrawButton(flUShort buttonStatus)
{
	CAutoLock autoLock(&_filterLock);
	_locusData.DrawButton = buttonStatus;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_LocusEraseButton(flUShort* buttonStatus)
{
	CAutoLock autoLock(&_filterLock);
	*buttonStatus = _locusData.EraseButton;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_LocusEraseButton(flUShort buttonStatus)
{
	CAutoLock autoLock(&_filterLock);
	_locusData.EraseButton = buttonStatus;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_LocusDrawPointerShape(flShort* shape)
{
	CAutoLock autoLock(&_filterLock);
	*shape = _locusData.DrawPointerShape;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_LocusDrawPointerShape(flShort shape)
{
	CAutoLock autoLock(&_filterLock);
	_locusData.DrawPointerShape = shape;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_LocusDrawPointerColor(flByte *color)
{
	CAutoLock autoLock(&_filterLock);
	memcpy(color, _locusData.DrawPointerColor, sizeof(_locusData.DrawPointerColor));
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_LocusDrawPointerColor(const flByte* color)
{
	CAutoLock autoLock(&_filterLock);
	memcpy(_locusData.DrawPointerColor, color, sizeof(_locusData.DrawPointerColor));
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_LocusEdgingOn(flBool *flag)
{
	CAutoLock autoLock(&_filterLock);
	*flag = _locusData.EdgingOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_LocusEdgingOn(flBool flag)
{
	CAutoLock autoLock(&_filterLock);
	_locusData.EdgingOn = flag;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_LocusBrokenLineOn(flBool *flag)
{
	CAutoLock autoLock(&_filterLock);
	*flag = _locusData.BrokenLineOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_LocusBrokenLineOn(flBool flag)
{
	CAutoLock autoLock(&_filterLock);
	_locusData.BrokenLineOn = flag;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_LocusWidth(flUShort *width)
{
	CAutoLock autoLock(&_filterLock);
	*width = _locusData.Width;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_LocusWidth(flUShort width)
{
	CAutoLock autoLock(&_filterLock);
	if (PP_MAXIMUM_LOCUS_WIDTH < width)
		width = PP_MAXIMUM_LOCUS_WIDTH;
	_locusData.Width = width;
	return NOERROR;
}
//------------------------------------------------------------------------------
// Phrase Settings
STDMETHODIMP
CPointerPosition::get_ShowPhraseOn(flBool *flag)
{
	CAutoLock autoLock(&_filterLock);
	*flag = _showPhraseOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_ShowPhraseOn(flBool flag)
{
	CAutoLock autoLock(&_filterLock);
	_showPhraseOn = flag;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_NumPhrases(flUInt *count)
{
	CAutoLock autoLock(&_filterLock);
	*count = _phraseDatas.getSize();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_NumPhrases(flUInt count)
{
	CAutoLock autoLock(&_filterLock);
	_phraseDatas.setSize(count);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::add_Phrase()
{
	CAutoLock autoLock(&_filterLock);
	_phraseDatas.add(flPhraseData());
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::remove_Phrase(flUInt index)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
		_phraseDatas.removeIndex(index);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_Phrase(flUInt index, flChar *phrase)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
		strcpy(phrase, _phraseDatas[index].Phrase);
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_Phrase(flUInt index, const flChar *phrase)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
	{
		if (phrase != NULL)
			strncpy(_phraseDatas[index].Phrase, phrase, sizeof(flChar) * PP_MAXIMUM_PHRASE_LENGTH);
		else
			_phraseDatas[index].Phrase[0] = 0;
	}
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_PhraseShowButton(flUInt index, flUShort* buttonStatus)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
		*buttonStatus = _phraseDatas[index].ShowButton;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_PhraseShowButton(flUInt index, flUShort buttonStatus)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
		_phraseDatas[index].ShowButton = buttonStatus;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_PhraseShowPointerShape(flUInt index, flShort* shape)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
		*shape = _phraseDatas[index].ShowPointerShape;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_PhraseShowPointerShape(flUInt index, flShort shape)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
		_phraseDatas[index].ShowPointerShape = shape;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::get_PhraseShowPointerColor(flUInt index, flByte *color)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
		memcpy(color, _phraseDatas[index].ShowPointerColor,
			sizeof(_phraseDatas[index].ShowPointerColor));
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_PhraseShowPointerColor(flUInt index, const flByte* color)
{
	CAutoLock autoLock(&_filterLock);
	if (index < _phraseDatas.getSize())
		memcpy(_phraseDatas[index].ShowPointerColor, color, 
			sizeof(_phraseDatas[index].ShowPointerColor));
	return NOERROR;
}
//------------------------------------------------------------------------------
// User Data
STDMETHODIMP
CPointerPosition::get_UserData(flByte *userData, flUInt *size)
{
	CAutoLock autoLock(&_filterLock);

	memcpy(userData, _userData, sizeof(flByte) * _userDataSize);
	*size = _userDataSize;

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CPointerPosition::put_UserData(const flByte *userData, flUInt size)
{
	CAutoLock autoLock(&_filterLock);

	if (PP_MAXIMUM_USERDATA_SIZE < size)
		size = PP_MAXIMUM_USERDATA_SIZE;

	memcpy(_userData, userData, size);
	_userDataSize = size;

	return NOERROR;
}
//------------------------------------------------------------------------------
flBool
CPointerPosition::updatePointerInfo()
{
	CAutoLock autoLock(&_filterLock);

	if (_mouse == NULL)
		return _pointerInfo.isChanged();

	_mouse->update();

	flUShort buttonStatus = _mouse->getButtons();
	flUShort wheelStatus = (flUShort)_mouse->getZ();

	RECT rect;
	GetWindowRect(GetDesktopWindow(), &rect);
	flUShort screenWidth = (flUShort)rect.right;
	flUShort screenHeight = (flUShort)rect.bottom;

	POINT pointer;
	GetCursorPos(&pointer);
	flUShort pointerX = (flUShort)pointer.x;
	flUShort pointerY = (flUShort)pointer.y;


	// PointerShape
	// PointerColor
	// Phrase
	// DrawStatus
	{
		// Default Settings
		flShort pointerShape		= _pointerShape;
		const flByte *pointerColor	= _pointerColor;
		const flChar *phrase		= "";
		flBool drawLocus			= false;
		flBool eraseLocus			= false;
		flBool showPhrase			= false;

		// Check Phrase
		if (_showPhraseOn)
		{
			for(flUInt i = 0; i < _phraseDatas.getSize(); i++)
			{
				if (_phraseDatas[i].ShowButton == buttonStatus)
				{
					pointerShape = _phraseDatas[i].ShowPointerShape;
					pointerColor = _phraseDatas[i].ShowPointerColor;
					phrase = _phraseDatas[i].Phrase;
					showPhrase = true;

					break;
				}
			}
		}

		// Check Locus
		if (_drawLocusOn)
		{
			if (_locusData.DrawButton == buttonStatus)
			{
				pointerShape = _locusData.DrawPointerShape;
				pointerColor = _locusData.DrawPointerColor;
				drawLocus = true;
			}
			if (_locusData.EraseButton == buttonStatus)
			{
				eraseLocus = true;
			}
		}
		else
		{
			eraseLocus = true;
		}

		// Setup
		_pointerInfo.setPointerShape(pointerShape);
		_pointerInfo.setPointerColor(pointerColor);
		_pointerInfo.setPhrase(phrase);
		_pointerInfo.setDrawStatusDrawLocusOn(_drawLocusOn);
		_pointerInfo.setDrawStatusShowPhraseOn(_showPhraseOn);
		_pointerInfo.setDrawStatusDrawLocus(drawLocus);
		_pointerInfo.setDrawStatusEraseLocus(eraseLocus);
		_pointerInfo.setDrawStatusShowPhrase(showPhrase);
		_pointerInfo.setDrawStatusLocusEdgingOn(_locusData.EdgingOn);
		_pointerInfo.setDrawStatusLocusBrokenLineOn(_locusData.BrokenLineOn);
		_pointerInfo.setDrawStatusLocusWidth(_locusData.Width);
	}

	// ScreenSize
	{
		_pointerInfo.setScreenSize(screenWidth, screenHeight);
	}

	// PointerPosition
	{
		_pointerInfo.setPointerPosition(pointerX, pointerY);
	}

	// ButtonStatus
	{
		_pointerInfo.setButtonStatus(buttonStatus);
	}

	// WheelStatus
	{
		_pointerInfo.setWheelStatus(wheelStatus);
	}

	// NormalizedOrigin
	{
		_pointerInfo.setNormalizedOrigin(
			_normalizedOrigin[0], _normalizedOrigin[1]);
	}

	// NormalizedSize
	{
		_pointerInfo.setNormalizedSize(
			_normalizedSize[0], _normalizedSize[1]);
	}

	// NormalizedPointerPosition
	{
		flFloat cx, cy;
		flFloat ox, oy, ow, oh;
		ox = _normalizedOrigin[0];
		oy = _normalizedOrigin[1];
		ow = _normalizedSize[0];
		oh = _normalizedSize[1];
		cx = ox + pointerX / flFloat(screenWidth) * ow;
		cy = oy + pointerY / flFloat(screenHeight) * oh;

		_pointerInfo.setNormalizedPointerPosition(cx, cy);
	}

	// UserData
	{
		_pointerInfo.setUserData(_userData, _userDataSize);
	}

	return _pointerInfo.isChanged();
}
//------------------------------------------------------------------------------
