/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flStillPicture/StillPicture.h>
#include <flStillPicture/StillPictureProp.h>
#include <flStillPicture/StillPictureStream.h>
#include <flBase/flSystemTime.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudOpPinTypes =
{
    &MEDIATYPE_Video,			// MajorType
    &MEDIASUBTYPE_NULL			// MinorType
};
//------------------------------------------------------------------------------
const AMOVIESETUP_PIN sudOpPin =
{
	L"Out",						// The Pins name
	FALSE,						// Is rendered
	TRUE,						// Is an output pin
	FALSE,						// Allowed none
	FALSE,						// Allowed many
	&CLSID_NULL,				// Connects to filter
	NULL,						// Connects to pin
	1,							// Number of types
	&sudOpPinTypes				// Pin details
};
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudStillPicture =
{
    &CLSID_StillPicture,		// Filter CLSID
    L"NIME Still Picture",		// String name
    MERIT_DO_NOT_USE,			// Filter merit
    1,							// Number of pins
    &sudOpPin					// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[2] = {
    { L"StillPicture"
    , &CLSID_StillPicture
    , CStillPicture::CreateInstance
    , NULL
    , &sudStillPicture }
  ,
    { L"StillPictureProp"
    , &CLSID_StillPicturePropertyPage
    , CStillPictureProp::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CStillPicture::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
    CStillPicture *pNewObject = new CStillPicture(NAME("StillPicture"), punk, phr);
    if (pNewObject == NULL)
	{
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CStillPicture::CStillPicture(TCHAR *tszName, LPUNKNOWN pUnk, HRESULT *phr) :
CFLSource(tszName, pUnk, CLSID_StillPicture, phr),
CPersistStream(pUnk, phr)
{
	ASSERT(tszName != NULL);
	ASSERT(phr != NULL);

	CAutoLock cAutoLock(&_stateLock);

	_streams = new CFLSourceStream*[1];
	if (_streams == NULL)
	{
		*phr = E_OUTOFMEMORY;
		return ;
	}

	_streams[0] = new CStillPictureStream(phr, this, L"Out");
	if (_streams[0] == NULL)
	{
		*phr = E_OUTOFMEMORY;
		return;
	}
	_stillPictureStream = (CStillPictureStream *)_streams[0];

	// Property
	_outVideoResolution				= SPVR_THROUGH;
	_outputMode						= SPOM_STRETCHTOVIDEO;
	_filename						= "";
	_frameRate						= 30;
	_noiseOn						= true;

	// Status
	_isRunning						= false;
	_orgImage.clear();
	_extImage.clear();
	setupImage();
}
//------------------------------------------------------------------------------
CStillPicture::~CStillPicture()
{
}
//------------------------------------------------------------------------------
// Reveals IStillPicture & ISpecifyPropertyPages
STDMETHODIMP
CStillPicture::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IStillPicture)
	{
        return GetInterface((IStillPicture *) this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    }
	else
	{
		return CFLSource::NonDelegatingQueryInterface(riid, ppv);
    }
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// IMediaFilter
STDMETHODIMP
CStillPicture::Run(REFERENCE_TIME startTime)
{
	HRESULT hr = CFLSource::Run(startTime);
	if (FAILED(hr))
		return hr;

	CAutoLock autoLock(&_filterLock);

	_isRunning			= true;

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::Pause()
{
    return CFLSource::Pause();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::Stop()
{
	{
		CAutoLock autoLock(&_filterLock);
		_isRunning			= false;
	}

	HRESULT hr = CFLSource::Stop();
	if (FAILED(hr))
		return hr;

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::GetState(DWORD dwMSecs, FILTER_STATE *State)
{
	HRESULT hr = CFLSource::GetState(dwMSecs, State);

	if (m_State == State_Paused)
		return VFW_S_CANT_CUE;
	else
		return hr;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_StillPicture;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CStillPicture::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	flUInt length;
	const flChar *str;

	WRITEOUT(_outVideoResolution);
	WRITEOUT(_outputMode);
	length = _filename.length();
	WRITEOUT(length);
	if (length != 0)
	{
		str = _filename.c_str();
		WRITENOUT(str, length);
	}
	WRITEOUT(_noiseOn);
	WRITEOUT(_frameRate);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CStillPicture::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	flUInt length;
	flChar str[SP_MAXIMUM_FILENAME_LENGTH];

	READIN(_outVideoResolution);
	READIN(_outputMode);
	READIN(length);
	if (length != 0)
	{
		READNIN(str, length);
		str[length] = 0;
		_filename = flString(str);

		flImageIO imageIO;
		_orgImage = imageIO.load(_filename);
	}
	READIN(_noiseOn);
	READIN(_frameRate);

	setupImage();

	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CStillPicture::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CStillPicture::SizeMax()
{
	return	1000;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages method
STDMETHODIMP
CStillPicture::GetPages(CAUUID *pPages)
{
	pPages->cElems = 1;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 1);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_StillPicturePropertyPage;
	return NOERROR;
}
//------------------------------------------------------------------------------
// IStillPicture
STDMETHODIMP
CStillPicture::get_OutputVideoResolution(flInt *videoResolution)
{
	//CAutoLock autoLock(&_filterLock);
	*videoResolution = _outVideoResolution;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::put_OutputVideoResolution(flInt videoResolution)
{
	CAutoLock autoLock(&_filterLock);

	if (_outVideoResolution == videoResolution)
		return NOERROR;

	_outVideoResolution = videoResolution;
	setupImage();

	ReconnectPin((IPin *)_streams[0], NULL);

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::get_OutputMode(flInt* mode)
{
	CAutoLock lock(&_filterLock);
	*mode = _outputMode;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::put_OutputMode(flInt mode)
{
	CAutoLock lock(&_filterLock);

	if (_outputMode == mode)
		return NOERROR;

	_outputMode = mode;
	setupImage();

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::get_Filename(flChar* filename)
{
	//CAutoLock autoLock(&_filterLock);

	if (filename == NULL)
		return NOERROR;

	if (_filename.length() == 0)
		filename[0] = 0;
	else
		strcpy(filename, _filename.c_str());

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::put_Filename(const flChar* filename)
{
	CAutoLock autoLock(&_filterLock);

	if (filename == NULL || strlen(filename) == 0)
		return NOERROR;

	if (_filename.length() != 0 && !strcmp(filename, _filename.c_str()))
		return NOERROR;

	_filename = flString(filename);

	flImageIO imageIO;
	_orgImage = imageIO.load(_filename);

	setupImage();

	ReconnectPin((IPin *)_streams[0], NULL);

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::get_FrameRate(flInt *frameRate)
{
	//CAutoLock autoLock(&_filterLock);
	*frameRate = _frameRate;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::put_FrameRate(flInt frameRate)
{
	CAutoLock autoLock(&_filterLock);
	_frameRate = frameRate;
	if (_frameRate == 0)
		_frameRate = 1;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::get_NoiseOn(flBool *noiseOn)
{
	//CAutoLock autoLock(&_filterLock);
	*noiseOn = _noiseOn;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPicture::put_NoiseOn(flBool noiseOn)
{
	CAutoLock autoLock(&_filterLock);
	_noiseOn = noiseOn;
	return NOERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
void
CStillPicture::setupImage()
{
	_extImage = _orgImage;

	if (_extImage.width() == 0 || _extImage.height() == 0)
	{
		flByte tmpData[3];
		memset(tmpData, 0, sizeof(flByte) * 3);
		_extImage.set(1, 1, 3, tmpData);
		_extImage.stretch(320, 240);
	}

	switch(_outVideoResolution)
	{
	case SPVR_SQCIF:		resize(128, 96);	break;
	case SPVR_QCIF:			resize(176, 144);	break;
	case SPVR_CIF:			resize(352, 288);	break;
	case SPVR_160x120:		resize(160, 120);	break;
	case SPVR_320x240:		resize(320, 240);	break;
	case SPVR_360x240:		resize(360, 240);	break;
	case SPVR_640x480:		resize(640, 480);	break;
	case SPVR_720x480:		resize(720, 480);	break;
	case SPVR_800x600:		resize(800, 600);	break;
	case SPVR_1024x768:		resize(1024, 768);	break;
	case SPVR_1280x1024:	resize(1280, 1024);	break;
	case SPVR_1600x1200:	resize(1600, 1200);	break;
	case SPVR_HDTV720P:		resize(1280, 720);	break;
	case SPVR_HDTV1080I:	resize(1920, 1080);	break;
	case SPVR_SDTV480P:		resize(852, 480);	break;
	case SPVR_SDTV480I:		resize(852, 480);	break;
	case SPVR_THROUGH:		resizeThrough();	break;
	}

	_extImage.swapColor();
}
//------------------------------------------------------------------------------
void
CStillPicture::resize(flUInt width, flUInt height)
{
	switch(_outputMode)
	{
	case SPOM_STRETCHTOVIDEO:
		{
			_extImage.stretch(width, height);
		}
		break;
	case SPOM_KEEPASPECTV:
		{
			flUInt oldWidth = _extImage.width();
			flUInt oldHeight = _extImage.height();

			flUInt newWidth = flUInt(oldHeight * flFloat(width) / flFloat(height));

			_extImage.resize(newWidth, oldHeight);
			_extImage.stretch(width, height);
		}
		break;
	case SPOM_KEEPASPECTH:
		{
			flUInt oldWidth = _extImage.width();
			flUInt oldHeight = _extImage.height();

			flUInt newHeight = flUInt(oldWidth * flFloat(height) / flFloat(width));

			_extImage.resize(oldWidth, newHeight);
			_extImage.stretch(width, height);
		}
		break;
	case SPOM_KEEPASPECTHV:
		{
			flUInt oldWidth = _extImage.width();
			flUInt oldHeight = _extImage.height();

			flUInt newWidth = flUInt(oldHeight * flFloat(width) / flFloat(height));
			flUInt newHeight = flUInt(oldWidth * flFloat(height) / flFloat(width));

			if (oldWidth < newWidth)
			{
				_extImage.resize(newWidth, oldHeight);
				_extImage.stretch(width, height);
			}
			else
			{
				_extImage.resize(oldWidth, newHeight);
				_extImage.stretch(width, height);
			}
		}
		break;
	}
}
//------------------------------------------------------------------------------
void
CStillPicture::resizeThrough()
{
	flUInt width, height;
	width = _extImage.width();
	height = _extImage.height();

	width -= (width % 4);

	_extImage.stretch(width, height);
}
//------------------------------------------------------------------------------
