/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN

#include <windows.h>
#include <streams.h>
#include <initguid.h>
#include <olectl.h>
#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif

#include <flTypes/flUIDs.h>
#include <flTypes/IStillPicture.h>
#include <flStillPicture/StillPicture.h>
#include <flStillPicture/StillPictureStream.h>
//------------------------------------------------------------------------------
CStillPictureStream::CStillPictureStream
					(HRESULT *phr, CStillPicture *pParent, LPCWSTR pPinName) :
CFLSourceStream(NAME("StillPictureStream"), phr, pParent, pPinName)
{
	_stillPicture	= pParent;
	_streamTime		= 0;
}
//------------------------------------------------------------------------------
CStillPictureStream::~CStillPictureStream()
{}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
HRESULT
CStillPictureStream::FillBuffer(IMediaSample *pms)
{
	// Stream Time
	REFERENCE_TIME interval = (10000000 / _stillPicture->_frameRate);
    REFERENCE_TIME timeStart = _streamTime;
    REFERENCE_TIME timeEnd  = _streamTime + interval;
	pms->SetTime(&timeStart, &timeEnd);
	_streamTime = timeEnd;

	// Media Time
	pms->SetMediaTime(NULL, NULL);

	// Preroll
	pms->SetPreroll(FALSE);

	// Discontinuity
	pms->SetDiscontinuity(FALSE);

	// SyncPoint
    pms->SetSyncPoint(TRUE);

	// Data
	BYTE* sampleData;
	flLong sampleDataLen;
	pms->GetPointer(&sampleData);
	sampleDataLen = pms->GetActualDataLength();

	// Copy Image data
	CAutoLock autoLock2(&_stillPicture->_filterLock);
	const flByte* imageData = _stillPicture->_extImage.image();
	flUInt imageDataLen = _stillPicture->_extImage.getSize();

	flUInt actualDataLen = flMin((flUInt)sampleDataLen, imageDataLen);

	memcpy(sampleData, imageData, actualDataLen);

	// Draw Noize
	if (_stillPicture->_noiseOn)
	{
		BYTE *ptr = sampleData;
		while(ptr < sampleData + actualDataLen)
		{
			BYTE val = *ptr;

			val *= (0.8f + 0.2f * (flFloat(rand()) / flFloat(RAND_MAX)));
			*ptr = val;

			ptr += BYTE((flFloat(rand()) / flFloat(RAND_MAX)) * 5000.0f);
		}
	}

	return S_OK;
}
//------------------------------------------------------------------------------
void
CStillPictureStream::AdjustRate(flInt64 deliverProcTime)
{
	if (_stillPicture->_frameRate == 0)
		return ;

	flInt64 interval = 10000000 / (flInt64)_stillPicture->_frameRate;
	flInt64 adjustTime = interval - (deliverProcTime + 1);

	if (30000 < adjustTime)
		Sleep((flULong)((adjustTime - 30000) / 10000));
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
HRESULT
CStillPictureStream::DecideBufferSize(IMemAllocator *pAlloc,
                            ALLOCATOR_PROPERTIES *pProperties)
{
    CAutoLock cAutoLock(_source->pStateLock());
    ASSERT(pAlloc);
    ASSERT(pProperties);
    HRESULT hr = NOERROR;

	flULong sampleSize = m_mt.GetSampleSize();
	pProperties->cBuffers = 1;
	pProperties->cbBuffer = sampleSize;

    ASSERT(pProperties->cbBuffer);

    ALLOCATOR_PROPERTIES Actual;
    hr = pAlloc->SetProperties(pProperties, &Actual);
    if (FAILED(hr))
        return hr;

    if (Actual.cbBuffer < pProperties->cbBuffer)
        return E_FAIL;

    ASSERT(Actual.cBuffers == 1);
    return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CStillPictureStream::SetMediaType(const CMediaType *pmt)
{
    CAutoLock cAutoLock(_source->pStateLock());
	
    // Pass the call up to my base class
    HRESULT hr = CFLSourceStream::SetMediaType(pmt);
	
	return hr;
}
//------------------------------------------------------------------------------
HRESULT
CStillPictureStream::GetMediaType(CMediaType *pmt)
{
	CAutoLock cAutoLock(_source->pStateLock());

	CheckPointer(pmt, E_POINTER);

	if (_stillPicture->_extImage.width() == 0 ||
		_stillPicture->_extImage.height() == 0)
		return E_FAIL;

	VIDEOINFOHEADER *pvi = (VIDEOINFOHEADER *)pmt->AllocFormatBuffer(sizeof(VIDEOINFOHEADER));
	if (pvi == 0) 
		return(E_OUTOFMEMORY);

	ZeroMemory(pvi, pmt->cbFormat);   
	pvi->AvgTimePerFrame = (10000000 / _stillPicture->_frameRate);

	pvi->bmiHeader.biSize			= sizeof(BITMAPINFOHEADER);
	pvi->bmiHeader.biWidth			= _stillPicture->_extImage.width();
	pvi->bmiHeader.biHeight			= _stillPicture->_extImage.height();
	pvi->bmiHeader.biPlanes			= 1;
	pvi->bmiHeader.biBitCount		= _stillPicture->_extImage.components() * 8;
	pvi->bmiHeader.biCompression	= BI_RGB;
	pvi->bmiHeader.biXPelsPerMeter	= 2952; // 75 DPI
	pvi->bmiHeader.biYPelsPerMeter	= 2952;
	pvi->bmiHeader.biClrUsed		= 0;
	pvi->bmiHeader.biClrImportant	= 0;
	pvi->bmiHeader.biSizeImage		= GetBitmapSize(&pvi->bmiHeader);

	SetRectEmpty(&(pvi->rcSource));
	SetRectEmpty(&(pvi->rcTarget));

	pmt->SetType(&MEDIATYPE_Video);
	pmt->SetFormatType(&FORMAT_VideoInfo);
	pmt->SetTemporalCompression(FALSE);

	const GUID SubTypeGUID = GetBitmapSubtype(&pvi->bmiHeader);
	pmt->SetSubtype(&SubTypeGUID);
	pmt->SetSampleSize(pvi->bmiHeader.biSizeImage);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CStillPictureStream::OnThreadCreate()
{
	_streamTime = 0;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CStillPictureStream::Notify(IBaseFilter * pSender, Quality q)
{
	return S_OK;
}
//------------------------------------------------------------------------------
