/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/

#include <windows.h>
#include <streams.h>
#include <initguid.h>
#include <olectl.h>
#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif

#include <stdio.h>
#include <limits.h>
#include <math.h>

#include <flFilterCommon/flPointerInfo.h>
#include <flTypes/flUIDs.h>
#include <flTypes/IVideoExtract.h>

#include "transfrmn2.h"

#include "videoextractprop.h"
#include "videoextract.h"

#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue

// setup data

const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
    &MEDIATYPE_Video,			// Major type
    &MEDIASUBTYPE_NULL			// Minor type
};

const AMOVIESETUP_MEDIATYPE sudPinTypesI =
{
    &MEDIATYPE_Stream,			// Major type
    &MEDIASUBTYPE_PointerInfo	// Minor type
};

const AMOVIESETUP_PIN psudPins[] =
{
	{
		L"Input",           // String pin name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Output",          // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Output",          // String pin name
		FALSE,              // Is it rendered
		TRUE,               // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Input",           // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Input Pointer",   // Pins string name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Are we allowed none
		FALSE,              // And allowed many
		&CLSID_NULL,        // Connects to filter
		NULL,               // Connects to pin
		1,                  // Number of types
		&sudPinTypesI       // Pin information
	}
};


const AMOVIESETUP_FILTER sudVideoExtract =
{
    &CLSID_VideoExtract,    // Filter CLSID
    L"NIME Video Extraction Filter", // Filter name
    MERIT_DO_NOT_USE,       // Its merit
    3,                      // Number of pins
    psudPins                // Pin details
};


// List of class IDs and creator functions for the class factory. This
// provides the link between the OLE entry point in the DLL and an object
// being created. The class factory will call the static CreateInstance

CFactoryTemplate g_Templates[2] = {

    { L"VideoExtract"
    , &CLSID_VideoExtract
    , CVideoExtract::CreateInstance
    , NULL
    , &sudVideoExtract }
  ,
    { L"VideoExtractProp"
    , &CLSID_VideoExtractPropertyPage
    , CVideoExtractProperties::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);



#define DEF_OUTPUT_VIDEORESOLUTION		VR_SQCIF
#define DEF_MASK						M_RECTANGLE
#define DEF_BGCOLOR						BGC_BLUE
#define DEF_CUTPOSX						0
#define DEF_CUTPOSY						0
#define DEF_POINTERINDISABLE			false
#define DEF_GRIDX						6
#define DEF_GRIDY						6


#define M_PI							3.14159265358979323846f

//
// Constructor
//
CVideoExtract::CVideoExtract(TCHAR *tszName,LPUNKNOWN punk,HRESULT *phr) :
CTransformN2Filter(tszName, punk, CLSID_VideoExtract),
CPersistStream(punk, phr),
_pPointer(NULL),
_outVR(DEF_OUTPUT_VIDEORESOLUTION),
_mask(DEF_MASK),
_bgColor(DEF_BGCOLOR),
_cutPosX(DEF_CUTPOSX),
_cutPosY(DEF_CUTPOSY),
_pointerInDisable(DEF_POINTERINDISABLE),
_gridX(DEF_GRIDX),
_gridY(DEF_GRIDY),
_baseBufSize(0),
_baseBuf(NULL),
_ellpBufSize(0),
_ellpBuf(NULL)
{
    ASSERT(tszName);
    ASSERT(phr);

} // VideoExtract

CVideoExtract::~CVideoExtract()
{
	delete _pPointer;
	delete _baseBuf;
	delete _ellpBuf;
}

//
// CreateInstance
//
// Provide the way for COM to create a CVideoExtract object
//
CUnknown * WINAPI CVideoExtract::CreateInstance(LPUNKNOWN punk, HRESULT *phr) {

    CVideoExtract *pNewObject = new CVideoExtract(NAME("VideoExtract"), punk, phr);
    if (pNewObject == NULL) {
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;

} // CreateInstance


//
// NonDelegatingQueryInterface
//
// Reveals IVideoExtract and ISpecifyPropertyPages
//
STDMETHODIMP CVideoExtract::NonDelegatingQueryInterface(REFIID riid, void **ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IVideoExtract) {
        return GetInterface((IVideoExtract *) this, ppv);
	} else if (riid == IID_IPersistStream) {
		return GetInterface((IPersistStream *) this, ppv);
    } else if (riid == IID_ISpecifyPropertyPages) {
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    } else {
        return CTransformN2Filter::NonDelegatingQueryInterface(riid, ppv);
    }

} // NonDelegatingQueryInterface

int
CVideoExtract::GetPinCount()
{
	return 3;
}

CBasePin*
CVideoExtract::GetPin(int n)
{
	HRESULT hr= S_OK;
    WCHAR szbuf[20];

	if(_pPointer == NULL)
	{
		swprintf(szbuf, L"PointerIn");
		_pPointer = new CPointerInputPin(NAME("Pointer Input Pin"), this, &hr, szbuf);

        ASSERT(SUCCEEDED(hr));
        if(_pPointer == NULL) {
            return NULL;
        }
	}

	if (n < 2)
	{
		return CTransformN2Filter::GetPin(n);
	}
	else if (n == 2)
	{
		return _pPointer;
	}
	else
	{
		return NULL;
	}
}

STDMETHODIMP
CVideoExtract::FindPin(LPCWSTR Id, IPin **ppPin)
{
	CheckPointer(ppPin, E_POINTER);
    ValidateReadWritePtr(ppPin, sizeof(IPin *));

    if (0 == lstrcmpW(Id, L"In"))
	{
        *ppPin = GetPin(0);
    }
    else if (0 == lstrcmpW(Id, L"Out"))
	{
        *ppPin = GetPin(1);
    }
	else if (0 == lstrcmpW(Id, L"PointerIn"))
	{
		*ppPin = GetPin(2);
	}
    else
	{
        *ppPin = NULL;
        return VFW_E_NOT_FOUND;
    }

    HRESULT hr = NOERROR;

    //  AddRef() returned pointer - but GetPin could fail if memory is low.
    if(*ppPin)
	{
        (*ppPin)->AddRef();
    }
    else
	{
        hr = E_OUTOFMEMORY;  // probably.  There's no pin anyway.
    }

	return hr;
}

//
// Transform
//
// Copy the input sample into the output sample
// Then transform the output sample 'in place'
//
HRESULT CVideoExtract::Transform(IMediaSample *pIn, IMediaSample *pOut)
{
    CAutoLock cAutoLock(&m_VideoExtractLock);

	HRESULT hr = Copy(pIn, pOut);
    if (FAILED(hr)) {
        return hr;
    }
	return NOERROR;

} // Transform



HRESULT
CVideoExtract::Copy(IMediaSample *pSource, IMediaSample *pDest) const
{

    // Copy the sample data
    BYTE *pSourceBuffer, *pDestBuffer;
	BYTE *pSB, *pDB;
	long lSourceSize, lDestSize;
	long sw, sh, dw, dh;
	int x1, y1, x2, y2, xdo, ydo, w, h;

    lSourceSize = pSource->GetActualDataLength();
    lDestSize = pDest->GetActualDataLength();

    pSource->GetPointer(&pSourceBuffer);
    pDest->GetPointer(&pDestBuffer);

	pSB = pSourceBuffer;
	pDB = pDestBuffer;

	CMediaType imt = m_pInput->CurrentMediaType();
	CMediaType omt = m_pOutput->CurrentMediaType();

	VIDEOINFO* ivinfo = (VIDEOINFO *)imt.Format();
	VIDEOINFO* ovinfo = (VIDEOINFO *)omt.Format();

	sw = ivinfo->bmiHeader.biWidth;
	sh = ivinfo->bmiHeader.biHeight;
	dw = ovinfo->bmiHeader.biWidth;
	dh = ovinfo->bmiHeader.biHeight;

	if (_baseBufSize == lDestSize)
		CopyMemory((PVOID)pDestBuffer, (PVOID)_baseBuf, lDestSize);

	if (_pointerInDisable || _pPointer->IsConnected() == FALSE)
	{
		x1 = _cutPosX;
		y1 = sh - dh - _cutPosY;
	} else
	{
		flFloat cx, cy;
		_pointerInfo.getNormalizedPointerPosition(cx, cy);
		x1 = (int)(float(sw) * cx);
		y1 = sh - dh - (int)(float(sh) * cy);
	}

	if (x1 + dw >= 0 && sw >= x1 &&
		y1 + dh >= 0 && sh >= y1)
	{

		xdo = ydo = 0;

		if (x1 < 0)
		{
			xdo = -x1;
			x1 = 0;
		}
		if (y1 < 0)
		{
			ydo = -y1;
			y1 = 0;
		}

		if (sw < x1 + dw - xdo)
		{
			x2 = sw;
		} else 
		{
			x2 = x1 + dw - xdo;
		}
		if (sh < y1 + dh - ydo)
		{
			y2 = sh;
		} else 
		{
			y2 = y1 + dh - ydo;
		}

		w = x2 - x1;
		h = y2 - y1;

		int bCnt = ivinfo->bmiHeader.biBitCount / 8;
		long bcSW = bCnt * sw;
		long bcDW = bCnt * dw;

		// Seek
		pSB += bcSW * y1;	// y dir
		pSB += bCnt * x1;	// x dir
		pDB += bcDW * ydo;	// y dir
		pDB += bCnt * xdo;	// x dir

		switch(_mask)
		{
		case M_RECTANGLE:
			{
				w *= bCnt;
				for(int i = 0; i < h; i++)
				{
					CopyMemory((PVOID)pDB, (PVOID)pSB, w);
					pSB += bcSW;
					pDB += bcDW;
				}
			}
			break;
		case M_ELLIPSE:
			{
				int beg, end;

				w *= bCnt;

				for(int i = 0; i < h; i++)
				{
					beg = (_ellpBuf[i + ydo] - xdo) * bCnt;
					if (beg < 0)
						beg = 0;
					else if (beg > w)
						beg = w;

					end = (_ellpBuf[i + ydo] + xdo) * bCnt - (bcDW - w);
					if (end < 0)
						end = 0;
					else if (end > w)
						end = w;

					pSB += beg;
					pDB += beg;
					CopyMemory((PVOID)pDB, (PVOID)pSB, w - beg - end);
					pSB += (bcSW - beg);
					pDB += (bcDW - beg);
				}
			}
			break;
		case M_CHECKERS:
			{
				w *= bCnt;

				for(int i = 0; i < h; i++)
				{
					CopyMemory((PVOID)pDB, (PVOID)pSB, w);
					pSB += bcSW;
					pDB += bcDW;
				}


				bool fl, fl2;
				int sx = dw / _gridX;
				int sy = dh / _gridY;

				BYTE *pDBt;
				pDB = pDestBuffer;

				sx *= bCnt;

				fl = false;
				for(int i = 0; i < dh; i++)
				{
					pDBt = pDB;

					if (i % sy == 0)
						fl = !fl;

					fl2 = false;
					for(unsigned int j = 0; j < _gridX; j++)
					{
						if ((fl && fl2) || (!fl && !fl2))
							CopyMemory((PVOID)pDBt, (PVOID)_baseBuf, sx);
						pDBt += sx;

						fl2 = !fl2;
					}

					pDB += bcDW;

					if (pDBt - pDB != 0)
					{
						if ((fl && fl2) || (!fl && !fl2))
							CopyMemory((PVOID)pDBt, (PVOID)_baseBuf, pDB - pDBt);
					}

				}
			}
			break;
		}
	}
		

    // Copy the sample times
    REFERENCE_TIME TimeStart, TimeEnd;
    if (NOERROR == pSource->GetTime(&TimeStart, &TimeEnd)) {
        pDest->SetTime(&TimeStart, &TimeEnd);
    }

    LONGLONG MediaStart, MediaEnd;
    if (pSource->GetMediaTime(&MediaStart,&MediaEnd) == NOERROR) {
        pDest->SetMediaTime(&MediaStart,&MediaEnd);
    }

    // Copy the Sync point property
    HRESULT hr = pSource->IsSyncPoint();
    if (hr == S_OK) {
        pDest->SetSyncPoint(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetSyncPoint(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

	// Copy the media type
	/*
    AM_MEDIA_TYPE *pMediaType;
    pSource->GetMediaType(&pMediaType);
    pDest->SetMediaType(pMediaType);
    DeleteMediaType(pMediaType);
	*/

    // Copy the preroll property
    hr = pSource->IsPreroll();
    if (hr == S_OK) {
        pDest->SetPreroll(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetPreroll(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

    // Copy the discontinuity property
    hr = pSource->IsDiscontinuity();
    if (hr == S_OK) {
	pDest->SetDiscontinuity(TRUE);
    }
    else if (hr == S_FALSE) {
        pDest->SetDiscontinuity(FALSE);
    }
    else {  // an unexpected error has occured...
        return E_UNEXPECTED;
    }

    // Copy the actual data length
	/*
    long lDataLength = pSource->GetActualDataLength();
    pDest->SetActualDataLength(lDataLength);
	*/

	long actLen = pDest->GetActualDataLength();

    return NOERROR;

}

HRESULT
CVideoExtract::AllocBaseBuffer(unsigned int size)
{
	delete [] _baseBuf;

	_baseBuf = new BYTE[size];
	if (!_baseBuf)
	{
		_baseBufSize = 0;
		return E_FAIL;
	}
	_baseBufSize = size;

	return NOERROR;
}

HRESULT
CVideoExtract::CreateEllipseMaskInfo()
{
	delete [] _ellpBuf;

	CMediaType mt = m_pOutput->CurrentMediaType();
	if (*(mt.FormatType()) != FORMAT_VideoInfo)
		return E_FAIL;

	VIDEOINFO* vinfo = (VIDEOINFO *)mt.Format();
	if (!vinfo)
		return E_FAIL;

	long w, h;
	float y;

	w = vinfo->bmiHeader.biWidth;
	h = vinfo->bmiHeader.biHeight;
	if (!w || !h)
		return E_FAIL;

	_ellpBufSize = h;
	_ellpBuf = new int[_ellpBufSize];

	w /= 2;
	
	for(int i = 0; i < h; i++)
	{
		y = float(i) / float(h - 1) * 2.0f - 1.0f;
		_ellpBuf[i] = int(float(w) * (1.0f - sqrt(1.0f - y * y)));
	}

	return NOERROR;
}

HRESULT
CVideoExtract::DrawBaseBufferInBGColor()
{
	if (!_baseBuf || !_baseBufSize)
		return E_FAIL;

	CMediaType cmt = m_pOutput->CurrentMediaType();

	if (*(cmt.FormatType()) != FORMAT_VideoInfo || _bgColor == BGC_BLACK)
	{
		ZeroMemory((PVOID)_baseBuf, _baseBufSize);
		return NOERROR;
	}

	if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB8))
	{
		const RGBQUAD colors [] = {
			{ 255, 0, 0, 0 },
			{ 0, 255, 0, 0 },
			{ 0, 0, 255, 0 }
		};

		VIDEOINFO* vinfo = (VIDEOINFO *)cmt.Format();
		int palCnt = PALETTE_ENTRIES(vinfo);

		if (palCnt < 255)
		{
			vinfo->bmiColors[palCnt] = colors[_bgColor - 1];
			FillMemory((PVOID)_baseBuf, _baseBufSize, (BYTE)palCnt);
		} else
		{
			ZeroMemory((PVOID)_baseBuf, _baseBufSize);
		}
	}
	else if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB555))
	{
		const BYTE colors[3][2] =
		{
			{ 0x00, 0xf8 },
			{ 0xe0, 0x07 },
			{ 0x1f, 0x00 }
		};

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 2; i++)
		{
			CopyMemory((PVOID)pBaseBuf, (PVOID)colors[_bgColor - 1], sizeof(BYTE) * 2);
			pBaseBuf += 2;
		}
	}
	else if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB565))
	{
		const BYTE colors[3][2] =
		{
			{ 0x00, 0x7c },
			{ 0xe0, 0x03 },
			{ 0x1f, 0x00 }
		};

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 2; i++)
		{
			CopyMemory((PVOID)pBaseBuf, (PVOID)colors[_bgColor - 1], sizeof(BYTE) * 2);
			pBaseBuf += 2;
		}
	}
	else if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB24))
	{
		const BYTE colors[3][3] =
		{
			{ 0x00, 0x00, 0xff },
			{ 0x00, 0xff, 0x00 },
			{ 0xff, 0x00, 0x00 }
		};

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 3; i++)
		{
			CopyMemory((PVOID)pBaseBuf, (PVOID)colors[_bgColor - 1], sizeof(BYTE) * 3);
			pBaseBuf += 3;
		}
	}
	else
	{
		ZeroMemory(_baseBuf, _baseBufSize);
	}

	return NOERROR;
}


//
// CheckInputType
//
// Check the input type is OK, return an error otherwise
//
HRESULT CVideoExtract::CheckInputType(const CMediaType *mtIn)
{
	HRESULT hr = E_INVALIDARG;

	if (*mtIn->FormatType() == FORMAT_VideoInfo)
	{

		if (!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB8) &&
			!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB555) &&
			!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB565) &&
			!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB24))
			return hr;

		return NOERROR;
	}

	return hr;

} // CheckInputType

HRESULT CVideoExtract::CheckOutputType(const CMediaType *mtOut)
{
	if (*mtOut->FormatType() != FORMAT_VideoInfo){
		return E_INVALIDARG;
	}

	VIDEOINFO* pvi = (VIDEOINFO *)mtOut->Format();
	if (pvi->rcSource.right != pvi->bmiHeader.biWidth ||
		pvi->rcSource.bottom != pvi->bmiHeader.biHeight)
		return E_INVALIDARG;

	return NOERROR;
}

//
// DecideBufferSize
//
// Tell the output pin's allocator what size buffers we
// require. Can only do this when the input is connected
//
HRESULT CVideoExtract::DecideBufferSize(IMemAllocator *pAlloc,ALLOCATOR_PROPERTIES *pProp)
{
	if (m_pInput->IsConnected() == FALSE)
		return E_UNEXPECTED;

	ASSERT(pAlloc);
	ASSERT(pProp);
	ASSERT(m_pInput->CurrentMediaType().bFixedSizeSamples);

	VIDEOINFOHEADER *pvi = (VIDEOINFOHEADER *)m_pOutput->CurrentMediaType().pbFormat;
	pvi->bmiHeader.biSizeImage = GetBitmapSize(&pvi->bmiHeader);
	ASSERT(pvi->bmiHeader.biSizeImage != 0);

	pProp->cBuffers = 1;
	pProp->cbBuffer = pvi->bmiHeader.biSizeImage;

	ALLOCATOR_PROPERTIES Actual;
	HRESULT hr = pAlloc->SetProperties(pProp, &Actual);
	if (FAILED(hr))
		return hr;

	ASSERT(Actual.cBuffers == 1);

	if (Actual.cBuffers < pProp->cBuffers ||
		Actual.cbBuffer < pProp->cbBuffer)
		return E_FAIL;


    /// Allocate BASE BUFFER
	hr = AllocBaseBuffer(Actual.cbBuffer);
	if (FAILED(hr))
		return E_FAIL;

	/// Create ELLIPSE MASK INFO
	hr = CreateEllipseMaskInfo();
	if (FAILED(hr))
		return E_FAIL;

	/// Draw BASE BUFFER in BG COLOR
	hr = DrawBaseBufferInBGColor();
	if (FAILED(hr))
		return E_FAIL;

    return NOERROR;

} // DecideBufferSize


HRESULT
CVideoExtract::SetMediaType(PIN_DIRECTION direction,const CMediaType *pmt)
{
    // Reconnect where necessary.
    if( m_pInput->IsConnected() && m_pOutput->IsConnected() ){
        FILTER_INFO fInfo;

        QueryFilterInfo( &fInfo );

        if (direction == PINDIR_INPUT && *pmt != m_pOutput->CurrentMediaType() )
            fInfo.pGraph->Reconnect( m_pOutput );

        QueryFilterInfoReleaseGraph( fInfo );
    }

	return NOERROR;
}


//
// GetMediaType
//
// I support one type, namely the type of the input pin
// We must be connected to support the single output type
//
HRESULT CVideoExtract::GetMediaType(int iPosition, CMediaType *pMediaType)
{
    CAutoLock cAutoLock(&m_VideoExtractLock);
    if (iPosition < 0) {
        return E_INVALIDARG;
    }
    if (iPosition > 0) {
        return VFW_S_NO_MORE_ITEMS;
    }
	
	ASSERT(m_pInput != NULL);
	if (m_pInput->IsConnected() == FALSE)
        return E_UNEXPECTED;
	CMediaType iMediaType = m_pInput->CurrentMediaType();

	*pMediaType = iMediaType;
	
	VIDEOINFO *pvi = (VIDEOINFO *)pMediaType->AllocFormatBuffer(sizeof(VIDEOINFO));
    if (pvi == NULL) {
		return(E_OUTOFMEMORY);
    }
    ZeroMemory(pvi, sizeof(VIDEOINFO));

	VIDEOINFO *pivi = (VIDEOINFO *)iMediaType.Format();

	pvi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	pvi->bmiHeader.biBitCount = pivi->bmiHeader.biBitCount;
	pvi->bmiHeader.biXPelsPerMeter = pivi->bmiHeader.biXPelsPerMeter;
	pvi->bmiHeader.biYPelsPerMeter = pivi->bmiHeader.biYPelsPerMeter;
	pvi->bmiHeader.biClrImportant = pivi->bmiHeader.biClrImportant;
	pvi->bmiHeader.biClrUsed = pivi->bmiHeader.biClrUsed;
	for(DWORD i = 0; i < pvi->bmiHeader.biClrUsed; i++)
		pvi->bmiColors[i] = pivi->bmiColors[i];

	switch(_outVR){
	case VR_SQCIF:
		pvi->bmiHeader.biWidth		= 128;
		pvi->bmiHeader.biHeight		= 96;
		break;
	case VR_QCIF:
		pvi->bmiHeader.biWidth		= 176;
		pvi->bmiHeader.biHeight		= 144;
		break;
	case VR_CIF:
		pvi->bmiHeader.biWidth		= 352;
		pvi->bmiHeader.biHeight		= 288;
		break;
	case VR_160x120:
		pvi->bmiHeader.biWidth		= 160;
		pvi->bmiHeader.biHeight		= 120;
		break;
	case VR_320x240:
		pvi->bmiHeader.biWidth		= 320;
		pvi->bmiHeader.biHeight		= 240;
		break;
	case VR_360x240:
		pvi->bmiHeader.biWidth		= 360;
		pvi->bmiHeader.biHeight		= 240;
		break;
	case VR_640x480:
		pvi->bmiHeader.biWidth		= 640;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_720x480:
		pvi->bmiHeader.biWidth		= 720;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_800x600:
		pvi->bmiHeader.biWidth		= 800;
		pvi->bmiHeader.biHeight		= 600;
		break;
	case VR_1024x768:
		pvi->bmiHeader.biWidth		= 1024;
		pvi->bmiHeader.biHeight		= 768;
		break;
	case VR_1280x1024:
		pvi->bmiHeader.biWidth		= 1280;
		pvi->bmiHeader.biHeight		= 1024;
		break;
	case VR_1600x1200:
		pvi->bmiHeader.biWidth		= 1600;
		pvi->bmiHeader.biHeight		= 1200;
		break;
	case VR_HDTV720P:
		pvi->bmiHeader.biWidth		= 1280;
		pvi->bmiHeader.biHeight		= 720;
		break;
	case VR_HDTV1080I:
		pvi->bmiHeader.biWidth		= 1920;
		pvi->bmiHeader.biHeight		= 1080;
		break;
	case VR_SDTV480P:
		pvi->bmiHeader.biWidth		= 852;
		pvi->bmiHeader.biHeight		= 480;
		break;
	case VR_SDTV480I:
		pvi->bmiHeader.biWidth		= 852;
		pvi->bmiHeader.biHeight		= 480;
		break;
	}

	pMediaType->SetTemporalCompression(FALSE);
    pvi->bmiHeader.biPlanes		= 1;
	pvi->bmiHeader.biSizeImage	= GetBitmapSize(&pvi->bmiHeader);

	SetRectEmpty(&(pvi->rcSource));	// we want the whole image area rendered.
    SetRectEmpty(&(pvi->rcTarget));	// no particular destination rectangle
	pvi->rcSource.right = pvi->bmiHeader.biWidth;
	pvi->rcSource.bottom = pvi->bmiHeader.biHeight;

	const GUID SubTypeGUID = GetBitmapSubtype(&pvi->bmiHeader);
	pMediaType->SetSampleSize(pvi->bmiHeader.biSizeImage);
	pMediaType->SetType(&MEDIATYPE_Video);
	pMediaType->SetFormatType(&FORMAT_VideoInfo);


	return NOERROR;

} // GetMediaType


STDMETHODIMP
CVideoExtract::get_OutputVideoResolution(unsigned int *videoResolution)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	*videoResolution = (unsigned int)_outVR;

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::put_OutputVideoResolution(unsigned int videoResolution)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	if (_outVR != videoResolution)
	{
		_outVR = (VideoResolutionEnum)videoResolution;
		SetDirty(TRUE);

		ReconnectPin((IPin *)m_pOutput, NULL);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::get_Mask(unsigned int *mask)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	*mask = (unsigned int)_mask;

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::put_Mask(unsigned int mask)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	if (_mask != mask)
	{
		_mask = (MaskEnum)mask;
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::get_BackgroundColor(unsigned int *bgColor)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	*bgColor = (unsigned int)_bgColor;

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::put_BackgroundColor(unsigned int bgColor)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	if (_bgColor != bgColor)
	{
		_bgColor = (BackgroundColorEnum)bgColor;
		SetDirty(TRUE);

		DrawBaseBufferInBGColor();
	}

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::get_CutPosition(int* x, int* y)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	*x = _cutPosX;
	*y = _cutPosY;

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::put_CutPosition(int x, int y)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	if (_cutPosX != x || _cutPosY != y)
	{
		_cutPosX = x;
		_cutPosY = y;
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::get_PointerInDisable(bool* flag)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	*flag = _pointerInDisable;

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::put_PointerInDisable(bool flag)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	if (_pointerInDisable != flag)
	{
		_pointerInDisable = flag;
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::get_CheckerGridXY(unsigned int* x, unsigned int* y)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	*x = _gridX;
	*y = _gridY;

	return NOERROR;
}

STDMETHODIMP
CVideoExtract::put_CheckerGridXY(unsigned int x, unsigned int y)
{
	CAutoLock cAutoLock(&m_VideoExtractLock);
	if (_gridX != x || _gridY != y)
	{
		_gridX = x;
		_gridY = y;

		if (_gridX < 2)
			_gridX = 2;
		else if (_gridX > 20)
			_gridX = 20;

		if (_gridY < 2)
			_gridY = 2;
		else if (_gridY > 20)
			_gridY = 20;

		SetDirty(TRUE);
	}

	return NOERROR;
}

#define WRITEOUT(var)  hr = pStream->Write(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define WRITENOUT(var,n)  hr = pStream->Write(var, n, NULL); \
if (FAILED(hr)) return hr;
#define READIN(var)    hr = pStream->Read(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define READNIN(var,n)    hr = pStream->Read(var, n, NULL); \
if (FAILED(hr)) return hr;


// IPersitStream
STDMETHODIMP
CVideoExtract::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_VideoExtract;
	return S_OK;
} // GetClassID

HRESULT
CVideoExtract::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	WRITEOUT(_outVR);
	WRITEOUT(_mask);
	WRITEOUT(_bgColor);
	WRITEOUT(_cutPosX);
	WRITEOUT(_cutPosY);
	WRITEOUT(_pointerInDisable);
	WRITEOUT(_gridX);
	WRITEOUT(_gridY);
	return NOERROR;
	
} // WriteToStream

HRESULT
CVideoExtract::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	READIN(_outVR);
	READIN(_mask);
	READIN(_bgColor);
	READIN(_cutPosX);
	READIN(_cutPosY);
	READIN(_pointerInDisable);
	READIN(_gridX);
	READIN(_gridY);
	return NOERROR;
	
} // ReadFromStream

DWORD
CVideoExtract::GetSoftwareVersion(void)
{
	return 1;
} // GetSoftwareVersion

int
CVideoExtract::SizeMax()
{
	return sizeof(unsigned int) * 7 + sizeof(bool) * 1;
} // SizeMax


//
// GetPages
//
// This is the sole member of ISpecifyPropertyPages
// Returns the clsid's of the property pages we support
//
STDMETHODIMP CVideoExtract::GetPages(CAUUID *pPages)
{
    pPages->cElems = 1;
    pPages->pElems = (GUID *) CoTaskMemAlloc(sizeof(GUID));
    if (pPages->pElems == NULL) {
        return E_OUTOFMEMORY;
    }
    *(pPages->pElems) = CLSID_VideoExtractPropertyPage;
    return NOERROR;

} // GetPages

HRESULT
CVideoExtract::ReceivePointer(IMediaSample *pSource, CPointerInputPin *ppin)
{
	// Get Media Sample data
	flLong dataLength;
	BYTE *data;
	dataLength = pSource->GetActualDataLength();
	pSource->GetPointer(&data);

	// Setup New PointerInfo
	CAutoLock autoLock(&m_VideoExtractLock);
	_pointerInfo.setChanged(data, dataLength);

	return NO_ERROR;
}

//////////////////////////////////////////////////////////////////////////////

// =================================================================
// Implements the CPointerInputPin class
// =================================================================

// Check streaming status
HRESULT
CPointerInputPin::CheckStreaming()
{

	//  Shouldn't be able to get any data if we're not connected!
	ASSERT(IsConnected());

	//  we're flushing
	if(m_bFlushing)
	{
		return S_FALSE;
	}
	//  Don't process stuff in Stopped state
	if(IsStopped())
	{
		return VFW_E_WRONG_STATE;
	}
	if(m_bRunTimeError)
	{
		return VFW_E_RUNTIME_ERROR;
	}
	return S_OK;
}


// constructor
CPointerInputPin::CPointerInputPin(TCHAR *pObjectName,  CTransformN2Filter *pTransformFilter,
								   HRESULT * phr, LPCWSTR pName)
: CBaseInputPin(pObjectName, pTransformFilter, &((CVideoExtract *)pTransformFilter)->getCSFilter(), phr, pName)
{
    DbgLog((LOG_TRACE,2,TEXT("CPointerInputPin::CPointerInputPin")));
    m_pTransformFilter = (CVideoExtract *)pTransformFilter;

}

// provides derived filter a chance to grab extra interfaces
HRESULT
CPointerInputPin::CheckConnect(IPin *pPin)
{
    HRESULT hr = m_pTransformFilter->CheckConnect(PINDIR_INPUT,pPin);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseInputPin::CheckConnect(pPin);
}


// provides derived filter a chance to release it's extra interfaces
HRESULT
CPointerInputPin::BreakConnect()
{
    ASSERT(IsStopped());
    m_pTransformFilter->BreakConnect(PINDIR_INPUT);
    return CBaseInputPin::BreakConnect();
}


// Let derived class know when the input pin is connected
HRESULT
CPointerInputPin::CompleteConnect(IPin *pReceivePin)
{
    HRESULT hr = m_pTransformFilter->CompleteConnect(PINDIR_INPUT,pReceivePin);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseInputPin::CompleteConnect(pReceivePin);
}


// check that we can support a given media type
HRESULT
CPointerInputPin::CheckMediaType(const CMediaType* pmt)
{
 	if(*pmt->Type() == MEDIATYPE_Stream &&
	   *pmt->Subtype() == MEDIASUBTYPE_PointerInfo){
        return NOERROR;
	}else {
		return S_FALSE;
	}
}


// set the media type for this connection
HRESULT
CPointerInputPin::SetMediaType(const CMediaType* mtIn)
{
    HRESULT hr = CBasePin::SetMediaType(mtIn);
    if(FAILED(hr)) {
        return hr;
    }

	if(*mtIn->Type() == MEDIATYPE_Stream &&
	   *mtIn->Subtype() == MEDIASUBTYPE_PointerInfo)
		return S_OK;
	else
		return S_FALSE;
}


// =================================================================
// Implements IMemInputPin interface
// =================================================================


// provide EndOfStream that passes straight downstream
// (there is no queued data)
STDMETHODIMP
CPointerInputPin::EndOfStream(void)
{
    CAutoLock lck(&m_pTransformFilter->getCSReceive());
    HRESULT hr = CheckStreaming();
    return hr;
}


// enter flushing state. Call default handler to block Receives, then
// pass to overridable method in filter
STDMETHODIMP
CPointerInputPin::BeginFlush(void)
{
	CAutoLock lck(&m_pTransformFilter->getCSFilter());
    HRESULT hr = CBaseInputPin::BeginFlush();
    if(FAILED(hr)) {
        return hr;
    }

	return S_OK;
}


// leave flushing state.
// Pass to overridable method in filter, then call base class
// to unblock receives (finally)
STDMETHODIMP
CPointerInputPin::EndFlush(void)
{
    CAutoLock lck(&m_pTransformFilter->getCSFilter());
    return CBaseInputPin::EndFlush();
}


// here's the next block of data from the stream.
// AddRef it yourself if you need to hold it beyond the end
// of this call.
HRESULT
CPointerInputPin::Receive(IMediaSample * pSample)
{
    HRESULT hr;
    CAutoLock lck(&m_pTransformFilter->getCSReceive());
    ASSERT(pSample);

    hr = CBaseInputPin::Receive(pSample);
    if(S_OK == hr)
	{
        hr = ((CVideoExtract *)m_pTransformFilter)->ReceivePointer(pSample, this);
    }
    return hr;
}


// override to pass downstream
STDMETHODIMP
CPointerInputPin::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate)
{
    return CBasePin::NewSegment(tStart, tStop, dRate);
}

//////////////////////////////////////////////////////////////////////////////


//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2( TRUE );

} // DllRegisterServer

//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2( FALSE );

} // DllUnregisterServer

