/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/

#include <streams.h>
#include <measure.h>

#include "cpndbase3.h"

#define DEF_REFSTREAM		0
#define DEF_INTERVAL		33
#define DEF_TIMESTAMPZERO	false
#define DEF_TIMEOFFSET		1

// =================================================================
// Implements the CCpndBase3Filter class
// =================================================================

CCpndBase3Filter::CCpndBase3Filter(TCHAR     *pName,
                                   LPUNKNOWN pUnk,
                                   REFCLSID  clsid,
								   unsigned int inputPinCnt) :
CBaseFilter(pName,pUnk,&m_csFilter, clsid),
m_iInputPinCnt(inputPinCnt),
_refStream(DEF_REFSTREAM),
_interval(DEF_INTERVAL),
_timeStampZero(DEF_TIMESTAMPZERO),
_timeOffset(DEF_TIMEOFFSET),
_startTime(0),
_streamTime(0),
_streamTimeIsValid(false),
m_pOutput(NULL),
m_bEOSDelivered(FALSE),
m_bQualityChanged(FALSE),
m_bSampleSkipped(FALSE)
{
#ifdef PERF
    RegisterPerfId();
#endif //  PERF

	ASSERT(m_iInputPinCnt != 0);
	m_ppInputs = new CCpndBase3InputPin *[m_iInputPinCnt];
	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
		m_ppInputs[i] = NULL;
	}

	_workerThread = new CCpndBase3Worker(this);
}

// destructor

CCpndBase3Filter::~CCpndBase3Filter() {

	delete _workerThread;

	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
		delete m_ppInputs[i];
	delete [] m_ppInputs;

	delete m_pOutput;

}


HRESULT
CCpndBase3Filter::SetReferenceStream(int refStream)
{
	CAutoLock lock(&m_csProp);

	if (refStream != -1 && m_iInputPinCnt <= (unsigned int)refStream)
		return E_INVALIDARG;

	_refStream = refStream;
	return NOERROR;
}

int
CCpndBase3Filter::GetReferenceStream()
{
	CAutoLock lock(&m_csProp);

	return _refStream;
}

HRESULT
CCpndBase3Filter::SetFrameRate(int frameRate)
{
	CAutoLock lock(&m_csProp);

	if (frameRate < 1)
		frameRate = 1;
	
	if (60 < frameRate)
		frameRate = 60;

	_interval = (long)(1000 / frameRate);
	return NOERROR;
}

int
CCpndBase3Filter::GetFrameRate()
{
	CAutoLock lock(&m_csProp);
	int frameRate = (1000 / (int)_interval);
	return frameRate;
}

HRESULT
CCpndBase3Filter::SetTimeStampZero(bool flag)
{
	CAutoLock lock(&m_csProp);
	_timeStampZero = flag;
	return NOERROR;
}

bool
CCpndBase3Filter::GetTimeStampZero()
{
	CAutoLock lock(&m_csProp);
	return _timeStampZero;
}

HRESULT
CCpndBase3Filter::SetTimeOffset(DWORD timeOffset)
{
	CAutoLock lock(&m_csProp);
	_timeOffset = timeOffset;
	return NOERROR;
}

DWORD
CCpndBase3Filter::GetTimeOffset()
{
	CAutoLock lock(&m_csProp);
	return _timeOffset;
}

CCpndBase3InputPin*
CCpndBase3Filter::GetConnectedInputPin(int priIdx)
{
	ASSERT(m_ppInputs[0]);

	CCpndBase3InputPin* pin = NULL;

	if (priIdx != -1 && priIdx < (int)m_iInputPinCnt)
	{
		if (m_ppInputs[priIdx]->IsConnected())
			return m_ppInputs[priIdx];
	}

	for(unsigned int i = 0; i < 4; i++)
	{
		pin = m_ppInputs[i];
		if (pin->IsConnected())
			break;
	}

	if (i == 4)
		return NULL;
	else
		return pin;
}


HRESULT CCpndBase3Filter::Transform(IMediaSample **ppInputs, IMediaSample *pOut) {
    UNREFERENCED_PARAMETER(ppInputs);
    UNREFERENCED_PARAMETER(pOut);
    DbgBreak("CCpndBase3Filter::Transform() should never be called");
    return E_UNEXPECTED;
}


// return the number of pins we provide

int CCpndBase3Filter::GetPinCount() {
    return m_iInputPinCnt + 1;
}


// return a non-addrefed CBasePin * for the user to addref if he holds onto it
// for longer than his pointer to us. We create the pins dynamically when they
// are asked for rather than in the constructor. This is because we want to
// give the derived class an oppportunity to return different pin objects

// We return the objects as and when they are needed. If either of these fails
// then we return NULL, the assumption being that the caller will realise the
// whole deal is off and destroy us - which in turn will delete everything.

CBasePin *
CCpndBase3Filter::GetPin(int n) {
    HRESULT hr = S_OK;

    // Create an input pin if necessary

    if(m_ppInputs[0] == NULL) {

		for(unsigned int i = 0; i < m_iInputPinCnt; i++)
		{
			char name[64];
			WCHAR pname[20];

			sprintf(name, "Transform(N) input pin %d", i);
			swprintf(pname, L"Input %d", i);

			m_ppInputs[i] = new CCpndBase3InputPin(
						NAME(name), this, &hr, pname, i);
		}
		
        //  Can't fail
        ASSERT(SUCCEEDED(hr));
        if(m_ppInputs[0] == NULL) {
            return NULL;
        }


		m_pOutput = (CCpndBase3OutputPin *)
            new CCpndBase3OutputPin(NAME("Transform(N) output pin"),
            this,            // Owner filter
            &hr,             // Result code
            L"Output");   // Pin name
        // Can't fail
        ASSERT(SUCCEEDED(hr));
        if(m_pOutput == NULL) {
			for(unsigned int i = 0; i < m_iInputPinCnt; i++)
			{
				delete m_ppInputs[i];
				m_ppInputs[i] = NULL;
			}
        }
    }

    // Return the appropriate pin
	if (n < (int)m_iInputPinCnt)
	{
		return m_ppInputs[n];
	} else if (n == m_iInputPinCnt)
	{
		return m_pOutput;
	} else
	{
		return NULL;
	}
}


//
// FindPin
//

STDMETHODIMP CCpndBase3Filter::FindPin(LPCWSTR Id, IPin **ppPin) {
    CheckPointer(ppPin,E_POINTER);
    ValidateReadWritePtr(ppPin,sizeof(IPin *));

	*ppPin = NULL;

	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
	    WCHAR szbuf[20];
		swprintf(szbuf, L"In%d", i);

		if (!lstrcmpW(Id, szbuf))
		{
			*ppPin = GetPin(i);
			break;
		}
	}

	if (!(*ppPin))
	{
		if (!lstrcmpW(Id, L"Out"))
		{
			*ppPin = GetPin(m_iInputPinCnt + 0);
		} else 
		{
			return VFW_E_NOT_FOUND;
		}
	}

    HRESULT hr = NOERROR;
    if(*ppPin) {
        (*ppPin)->AddRef();
    }
    else {
        hr = E_OUTOFMEMORY;
    }
    return hr;
}


// override these two functions if you want to inform something
// about entry to or exit from streaming state.

HRESULT
CCpndBase3Filter::StartStreaming() {
    return NOERROR;
}


HRESULT
CCpndBase3Filter::StopStreaming() {
    return NOERROR;
}


// override this to grab extra interfaces on connection

HRESULT
CCpndBase3Filter::CheckConnect(PIN_DIRECTION dir,
							   IPin *pPin, unsigned int pinIdx)
{
    UNREFERENCED_PARAMETER(dir);
    UNREFERENCED_PARAMETER(pPin);
	UNREFERENCED_PARAMETER(pinIdx);
    return NOERROR;
}


// place holder to allow derived classes to release any extra interfaces

HRESULT
CCpndBase3Filter::BreakConnect(PIN_DIRECTION dir, unsigned int pinIdx)
{
    UNREFERENCED_PARAMETER(dir);
	UNREFERENCED_PARAMETER(pinIdx);
    return NOERROR;
}


// Let derived classes know about connection completion

HRESULT
CCpndBase3Filter::CompleteConnect(PIN_DIRECTION direction,
								  IPin *pReceivePin, unsigned int pinIdx)
{
    UNREFERENCED_PARAMETER(direction);
    UNREFERENCED_PARAMETER(pReceivePin);
	UNREFERENCED_PARAMETER(pinIdx);
    return NOERROR;
}


// override this to know when the media type is really set

HRESULT
CCpndBase3Filter::SetMediaType(PIN_DIRECTION direction,
							   const CMediaType *pmt, unsigned int pinIdx)
{
    UNREFERENCED_PARAMETER(direction);
    UNREFERENCED_PARAMETER(pmt);
	UNREFERENCED_PARAMETER(pinIdx);
    return NOERROR;
}

// Return S_FALSE to mean "pass the note on upstream"
// Return NOERROR (Same as S_OK)
// to mean "I've done something about it, don't pass it on"
HRESULT CCpndBase3Filter::AlterQuality(Quality q)
{
    UNREFERENCED_PARAMETER(q);
    return S_FALSE;
}


// EndOfStream received. Default behaviour is to deliver straight
// downstream, since we have no queued data. If you overrode Receive
// and have queue data, then you need to handle this and deliver EOS after
// all queued data is sent
HRESULT
CCpndBase3Filter::EndOfStream(unsigned int pinIdx)
{
	unsigned int i;
	for(i = 0; i < m_iInputPinCnt; i++)
	{
		if (pinIdx != i &&
			m_ppInputs[i]->IsConnected() &&
			!m_ppInputs[i]->IsEndOfStream())
			break;
	}

	HRESULT hr = NOERROR;
	if (i == m_iInputPinCnt && m_pOutput != NULL)
	{
#ifdef DEBUG
		OutputDebugString("  Deliver End Of Stream.\n");
#endif
        hr = m_pOutput->DeliverEndOfStream();
    }

    return hr;
}


// enter flush state. Receives already blocked
// must override this if you have queued data or a worker thread
HRESULT
CCpndBase3Filter::BeginFlush(unsigned int pinIdx)
{
    HRESULT hr = NOERROR;
	if(m_pOutput != NULL)
	{
        // block receives -- done by caller (CBaseInputPin::BeginFlush)

        // discard queued data -- we have no queued data

        // free anyone blocked on receive - not possible in this filter

        // call downstream
#ifdef DEBUG
		OutputDebugString("  Deliver Begin Flush.\n");
#endif
        hr = m_pOutput->DeliverBeginFlush();
    }
    return hr;
}


// leave flush state. must override this if you have queued data
// or a worker thread
HRESULT
CCpndBase3Filter::EndFlush(unsigned int pinIdx)
{
	// sync with pushing thread -- we have no worker thread
	//CAutoLock lock2(&_workerThread->m_WorkerLock);

    // ensure no more data to go downstream -- we have no queued data

    // call EndFlush on downstream pins
    ASSERT(m_pOutput != NULL);

#ifdef DEBUG
	OutputDebugString("  Deliver End Flush.\n");
#endif
	
    return m_pOutput->DeliverEndFlush();

    // caller (the input pin's method) will unblock Receives
}


// override these so that the derived filter can catch them

STDMETHODIMP
CCpndBase3Filter::Stop()
{
	// Stop Worker Thread
	ASSERT(_workerThread->ThreadExists());
	_workerThread->Stop();
	_workerThread->Exit();
	_workerThread->Close();

	// Check Stop
	CAutoLock lck1(&m_csFilter);
    if(m_State == State_Stopped)
        return NOERROR;

    // Succeed the Stop if we are not completely connected
    ASSERT(m_ppInputs[0] == NULL || m_pOutput != NULL);

    if(m_ppInputs[0] == NULL || InputIsAllDisconnected() ||
        m_pOutput->IsConnected() == FALSE)
	{
        m_State = State_Stopped;
        m_bEOSDelivered = FALSE;
        return NOERROR;
    }

    ASSERT(m_ppInputs[0]);
    ASSERT(m_pOutput);

    // decommit the input pin before locking or we can deadlock
	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	    m_ppInputs[i]->Inactive();

    // synchronize with Receive calls
    CAutoLock lck2(&m_csReceive);
    m_pOutput->Inactive();


    // allow a class derived from CCpndBase3Filter
    // to know about starting and stopping streaming
    HRESULT hr = StopStreaming();
    if(SUCCEEDED(hr)) {
        // complete the state transition
        m_State = State_Stopped;
        m_bEOSDelivered = FALSE;
    }
    return hr;
}

STDMETHODIMP
CCpndBase3Filter::Pause()
{
    CAutoLock lck(&m_csFilter);
    HRESULT hr = NOERROR;

    if (m_State == State_Paused)
	{
        // (This space left deliberately blank)
    }

    // If we have no input pin or it isn't yet connected then when we are
    // asked to pause we deliver an end of stream to the downstream filter.
    // This makes sure that it doesn't sit there forever waiting for
    // samples which we cannot ever deliver without an input connection.

    else if(m_ppInputs[0] == NULL || InputIsAllDisconnected())
	{
        if(m_pOutput && m_bEOSDelivered == FALSE)
		{
            m_pOutput->DeliverEndOfStream();
            m_bEOSDelivered = TRUE;
        }
        m_State = State_Paused;
    }

    // We may have an input connection but no output connection
    // However, if we have an input pin we do have an output pin

    else if(m_pOutput->IsConnected() == FALSE)
	{
        m_State = State_Paused;
    }
    else
	{
        if(m_State == State_Stopped)
		{
            // allow a class derived from CCpndBase3Filter
            // to know about starting and stopping streaming
            CAutoLock lck2(&m_csReceive);
            hr = StartStreaming();
        }
        if(SUCCEEDED(hr))
		{
            hr = CBaseFilter::Pause();
        }
    }

    m_bSampleSkipped = FALSE;
    m_bQualityChanged = FALSE;
    return hr;
}

HRESULT
CCpndBase3Filter::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate, unsigned int pinIdx)
{
	CAutoLock lck(&m_csProp);

    if (m_pOutput != NULL && pinIdx == _refStream)
	{
#ifdef DEBUG
		char buffer[256];
		sprintf(buffer, "  Deliver NewSegment.  MediaStart:%d  MediaEnd:%d\n",
			(int)(tStart / (UNITS / MILLISECONDS)), (int)(tStop / (UNITS / MILLISECONDS)));
		OutputDebugString(buffer);
#endif

        return m_pOutput->DeliverNewSegment(tStart, tStop, dRate);
    }
    return S_OK;
}


BOOL
CCpndBase3Filter::InputIsAllDisconnected()
{
	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
		if (m_ppInputs[i]->IsConnected() == TRUE)
			return false;
	}
	return true;
}


HRESULT
CCpndBase3Filter::InitializeOutputSample(int pinIdx, 
										 IMediaSample *pSample,
										 IMediaSample **ppOutSample)
{
	CAutoLock lock(&m_csProp);

	HRESULT hr;
	IMediaSample *pOutSample;

	//
	// Get Output Media Sample
	//
	DWORD dwFlags = m_bSampleSkipped ? AM_GBF_PREVFRAMESKIPPED : 0;

	ASSERT(m_pOutput->m_pAllocator != NULL);
	hr = m_pOutput->m_pAllocator->GetBuffer(&pOutSample, NULL, NULL, dwFlags);
	if (FAILED(hr))
		return hr;

	*ppOutSample = pOutSample;
	ASSERT(pOutSample != NULL);


	//
	// Setup Media Sample property
	//
	if (_refStream != -1 && _refStream == pinIdx)
	{
		AM_SAMPLE2_PROPERTIES * const pProps = m_ppInputs[pinIdx]->SampleProps();

		// Stream Time
		if(pProps->dwSampleFlags & AM_SAMPLE_TIMEVALID)
		{
			_streamTimeIsValid = true;
			pOutSample->SetTime(&pProps->tStart, &pProps->tStop);
		}
		else
		{
			pOutSample->SetTime(NULL, NULL);
		}

		// Media Time
		LONGLONG MediaStart, MediaEnd;
		if(pSample->GetMediaTime(&MediaStart,&MediaEnd) == NOERROR)
		{
			pOutSample->SetMediaTime(&MediaStart,&MediaEnd);
		}
		else
		{
			pOutSample->SetMediaTime(NULL, NULL);
		}

		// Sync Point
		if(pProps->dwSampleFlags & AM_SAMPLE_SPLICEPOINT)
		{
			pOutSample->SetSyncPoint(TRUE);
		}
		else
		{
			pOutSample->SetSyncPoint(FALSE);
		}

		// Preroll
		if (pProps->dwSampleFlags & AM_SAMPLE_PREROLL)
		{
			pOutSample->SetPreroll(TRUE);
		}
		else
		{
			pOutSample->SetPreroll(FALSE);
		}

		// Discontinuity
		if(pProps->dwSampleFlags & AM_SAMPLE_DATADISCONTINUITY)
		{
			pOutSample->SetDiscontinuity(TRUE);
			m_bSampleSkipped = FALSE;
		}
		else
		{
			pOutSample->SetDiscontinuity(FALSE);
		}

	}

	return hr;
}

HRESULT
CCpndBase3Filter::Receive(unsigned int pinIdx, IMediaSample* pSample)
{
    ASSERT(pSample != NULL);
    ASSERT(m_pOutput != NULL);

    HRESULT hr;
	IMediaSample* pOutSample;

	{
		CAutoLock lock(&_workerThread->m_WorkerLock);

		// Get output Sample
		hr = InitializeOutputSample(pinIdx, pSample, &pOutSample);
		if (FAILED(hr))
			return hr;

		// Compound Input Media sample
		Compound(pinIdx, pSample, pOutSample);
	}

	pOutSample->Release();
    return hr;
}

HRESULT
CCpndBase3Filter::DeliverCompoundSample()
{
    HRESULT hr;
	IMediaSample* pOutSample;

    ASSERT(m_pOutput != NULL);

	{
		CAutoLock lock(&_workerThread->m_WorkerLock);

		// Get output Sample
		hr = InitializeOutputSample(-1, NULL, &pOutSample);
		if (FAILED(hr))
			return hr;

		// Original Compound for Inherited Filter
		FilterCompound(pOutSample);
		_streamTimeIsValid = false;

		// Setup Output sample property
		SetupMediaSampleProperty(pOutSample);
	}

	// Deliver Output Media sample
    hr = m_pOutput->m_pInputPin->Receive(pOutSample);
    m_bSampleSkipped = FALSE;

	pOutSample->Release();
    return hr;
}

void
CCpndBase3Filter::AdjustFrameRate()
{
	int refStream;
	bool timeStampZero;
	long interval;
	{
		CAutoLock cAutoLock(&m_csProp);
		refStream = _refStream;
		timeStampZero = _timeStampZero;
		interval = _interval;
	}

#if 1
	if ((refStream == -1) ||
		(refStream != -1 && !_streamTimeIsValid))
		Sleep(interval);
#else
	REFERENCE_TIME currentTime;

	IReferenceClock *pClock;
	GetSyncSource(&pClock);
	pClock->GetTime(&currentTime);
	pClock->Release();

	_streamTime = currentTime - _startTime;

	if ((refStream == -1 && timeStampZero) ||
		(refStream != -1 && !_streamTimeIsValid))
		Sleep(interval);
#endif
}

void
CCpndBase3Filter::SetupMediaSampleProperty(IMediaSample *pSample)
{
	CAutoLock cAutoLock(&m_csProp);

	if (_refStream != -1)
		return ;
#if 1
	pSample->SetTime(NULL, NULL);
#else
	if (_timeStampZero)
	{
		pSample->SetTime(NULL, NULL);
	}
	else
	{
		REFERENCE_TIME endTime;
		endTime = _streamTime + Int32x32To64(UNITS / MILLISECONDS, _interval);
		pSample->SetTime(&_streamTime, &endTime);
	}
#endif

	pSample->SetMediaTime(NULL, NULL);
	pSample->SetSyncPoint(FALSE);
	pSample->SetPreroll(FALSE);
	pSample->SetDiscontinuity(FALSE);
}


///////////////////////////////////////////////////////////////

CCpndBase3Worker::CCpndBase3Worker(CCpndBase3Filter* owner) :
_owner(owner)
{
}

CCpndBase3Worker::~CCpndBase3Worker()
{
}

DWORD CCpndBase3Worker::ThreadProc(void)
{
    Command com;

    do {
        com = (Command)GetRequest();
        if (com != CMD_INIT)
		{
            Reply((DWORD)E_UNEXPECTED);
        }
    } while (com != CMD_INIT);

    // Initialisation suceeded
    Reply(NOERROR);

    Command cmd;
    do {
        cmd = (Command)GetRequest();

        switch (cmd) {

        case CMD_EXIT:
            Reply(NOERROR);
            break;

        case CMD_RUN:
        case CMD_PAUSE:
            Reply(NOERROR);
            DoBufferProcessingLoop();
            break;

        case CMD_STOP:
            Reply(NOERROR);
            break;

        default:
            Reply((DWORD)E_NOTIMPL);
            break;
        }
    } while (cmd != CMD_EXIT);

    return 0;
}

HRESULT
CCpndBase3Worker::DoBufferProcessingLoop(void)
{
	Command com;

	do {
        while (!CheckRequest((DWORD*)&com))
		{
			if (_owner->IsActive())
			{
				_owner->AdjustFrameRate();
				_owner->DeliverCompoundSample();
			}
			else 
			{
				Sleep(10);
			}
		}

        if (com == CMD_RUN || com == CMD_PAUSE)
		{
            Reply(NOERROR);
        }
		else if (com != CMD_STOP)
		{
            Reply((DWORD)E_UNEXPECTED);
        }
    } while (com != CMD_STOP);

    return S_FALSE;
}

///////////////////////////////////////////////////////////////


// Check streaming status
HRESULT
CCpndBase3InputPin::CheckStreaming() {
    ASSERT(m_pTransformFilter->m_pOutput != NULL);
    if(!m_pTransformFilter->m_pOutput->IsConnected()) {
        return VFW_E_NOT_CONNECTED;
    }
    else {
        //  Shouldn't be able to get any data if we're not connected!
        ASSERT(IsConnected());

        //  we're flushing
        if(m_bFlushing) {
            return S_FALSE;
        }
        //  Don't process stuff in Stopped state
        if(IsStopped()) {
            return VFW_E_WRONG_STATE;
        }
        if(m_bRunTimeError) {
            return VFW_E_RUNTIME_ERROR;
        }
        return S_OK;
    }
}


// =================================================================
// Implements the CCpndBase3InputPin class
// =================================================================


// constructor

CCpndBase3InputPin::CCpndBase3InputPin(TCHAR *pObjectName, CCpndBase3Filter *pTransformFilter,
									HRESULT * phr, LPCWSTR pName, unsigned int pinIdx) :
CBaseInputPin(pObjectName, pTransformFilter, &pTransformFilter->m_csFilter, phr, pName)
{
	DbgLog((LOG_TRACE,2,TEXT("CCpndBase3InputPin::CCpndBase3InputPin")));
	m_pTransformFilter = pTransformFilter;
	m_bPinIdx = pinIdx;
	m_bEOS = FALSE;
}

#ifdef UNICODE
CCpndBase3InputPin::CCpndBase3InputPin(CHAR *pObjectName, CCpndBase3Filter *pTransformFilter,
									 HRESULT * phr, LPCWSTR pName, unsigned int pinIdx) :
CBaseInputPin(pObjectName, pTransformFilter, &pTransformFilter->m_csFilter, phr, pName)
{
    DbgLog((LOG_TRACE,2,TEXT("CCpndBase3InputPin::CCpndBase3InputPin")));
    m_pTransformFilter = pTransformFilter;
	m_bPinIdx = pinIdx;
	m_bEOS = FALSE;
}
#endif


BOOL
CCpndBase3InputPin::IsEndOfStream()
{
	return m_bEOS;
}

// provides derived filter a chance to grab extra interfaces
HRESULT
CCpndBase3InputPin::CheckConnect(IPin *pPin)
{
    HRESULT hr = m_pTransformFilter->CheckConnect(PINDIR_INPUT, pPin, m_bPinIdx);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseInputPin::CheckConnect(pPin);
}


// provides derived filter a chance to release it's extra interfaces
HRESULT
CCpndBase3InputPin::BreakConnect()
{
    //  Can't disconnect unless stopped
    ASSERT(IsStopped());
    m_pTransformFilter->BreakConnect(PINDIR_INPUT, m_bPinIdx);
    return CBaseInputPin::BreakConnect();
}


// Let derived class know when the input pin is connected
HRESULT
CCpndBase3InputPin::CompleteConnect(IPin *pReceivePin)
{
    HRESULT hr = m_pTransformFilter->CompleteConnect(PINDIR_INPUT, pReceivePin, m_bPinIdx);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseInputPin::CompleteConnect(pReceivePin);
}


// check that we can support a given media type
HRESULT
CCpndBase3InputPin::CheckMediaType(const CMediaType* pmt)
{
    HRESULT hr = m_pTransformFilter->CheckInputType(pmt, m_bPinIdx);
	return hr;
}


// set the media type for this connection
HRESULT
CCpndBase3InputPin::SetMediaType(const CMediaType* mtIn)
{
    // Set the base class media type (should always succeed)
    HRESULT hr = CBasePin::SetMediaType(mtIn);
    if(FAILED(hr)) {
        return hr;
    }

    // check the transform can be done (should always succeed)
    ASSERT(SUCCEEDED(m_pTransformFilter->CheckInputType(mtIn, m_bPinIdx)));

    return m_pTransformFilter->SetMediaType(PINDIR_INPUT, mtIn, m_bPinIdx);
}


HRESULT
CCpndBase3InputPin::Active()
{
	{
	    CAutoLock lck(&m_pTransformFilter->m_csFilter);
		m_bEOS = FALSE;
	}
	return CBaseInputPin::Active();
}

// =================================================================
// Implements IMemInputPin interface
// =================================================================


// here's the next block of data from the stream.
// AddRef it yourself if you need to hold it beyond the end
// of this call.
HRESULT
CCpndBase3InputPin::Receive(IMediaSample * pSample)
{
    HRESULT hr;
    CAutoLock lck(&m_pTransformFilter->m_csReceive);
    ASSERT(pSample);

    // check all is well with the base class
	hr = CBaseInputPin::Receive(pSample);
    if(S_OK == hr)
		hr = m_pTransformFilter->Receive(m_bPinIdx, pSample);
	
    return hr;
}

STDMETHODIMP
CCpndBase3InputPin::ReceiveCanBlock()
{
	return S_OK;
}


// provide EndOfStream that passes straight downstream
// (there is no queued data)
STDMETHODIMP
CCpndBase3InputPin::EndOfStream(void)
{
	CAutoLock lck(&m_pTransformFilter->m_csReceive);

	{
	    CAutoLock lck(&m_pTransformFilter->m_csFilter);
		m_bEOS = TRUE;
	}

    HRESULT hr = CheckStreaming();
	if(S_OK == hr)
	{
		hr = m_pTransformFilter->EndOfStream(m_bPinIdx);
	}
	return hr;
}


// override to pass downstream
STDMETHODIMP
CCpndBase3InputPin::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate)
{
	//  Save the values in the pin
    CBasePin::NewSegment(tStart, tStop, dRate);
	return m_pTransformFilter->NewSegment(tStart, tStop, dRate, m_bPinIdx);
}

// enter flushing state. Call default handler to block Receives, then
// pass to overridable method in filter
STDMETHODIMP
CCpndBase3InputPin::BeginFlush(void)
{
	CAutoLock lck(&m_pTransformFilter->m_csFilter);
    //  Are we actually doing anything?
    ASSERT(m_pTransformFilter->m_pOutput != NULL);

	if (!IsConnected() || !m_pTransformFilter->m_pOutput->IsConnected())
	{
        return VFW_E_NOT_CONNECTED;
    }
    HRESULT hr = CBaseInputPin::BeginFlush();
    if(FAILED(hr)) {
        return hr;
    }

    return m_pTransformFilter->BeginFlush(m_bPinIdx);
}


// leave flushing state.
// Pass to overridable method in filter, then call base class
// to unblock receives (finally)
STDMETHODIMP
CCpndBase3InputPin::EndFlush(void)
{
	CAutoLock lck(&m_pTransformFilter->m_csFilter);
    //  Are we actually doing anything?
    ASSERT(m_pTransformFilter->m_pOutput != NULL);
    if(!IsConnected() || !m_pTransformFilter->m_pOutput->IsConnected())
	{
        return VFW_E_NOT_CONNECTED;
    }

	HRESULT hr = m_pTransformFilter->EndFlush(m_bPinIdx);
	if(FAILED(hr)) {
		return hr;
	}

	return CBaseInputPin::EndFlush();
}





// =================================================================
// Implements the CCpndBase3OutputPin class
// =================================================================


// constructor

CCpndBase3OutputPin::CCpndBase3OutputPin(TCHAR *pObjectName, CCpndBase3Filter *pTransformFilter,
									   HRESULT * phr, LPCWSTR pPinName) :
CBaseOutputPin(pObjectName, pTransformFilter, &pTransformFilter->m_csFilter, phr, pPinName),
m_pPosition(NULL)
{
    DbgLog((LOG_TRACE,2,TEXT("CCpndBase3OutputPin::CCpndBase3OutputPin")));
    m_pTransformFilter = pTransformFilter;
}

#ifdef UNICODE
CCpndBase3OutputPin::CCpndBase3OutputPin(CHAR *pObjectName, CCpndBase3Filter *pTransformFilter,
									   HRESULT * phr, LPCWSTR pPinName) :
CBaseOutputPin(pObjectName, pTransformFilter, &pTransformFilter->m_csFilter, phr, pPinName),
m_pPosition(NULL)
{
    DbgLog((LOG_TRACE,2,TEXT("CCpndBase3OutputPin::CCpndBase3OutputPin")));
    m_pTransformFilter = pTransformFilter;
}
#endif

// destructor

CCpndBase3OutputPin::~CCpndBase3OutputPin()
{
    DbgLog((LOG_TRACE,2,TEXT("CCpndBase3OutputPin::~CCpndBase3OutputPin")));
    if(m_pPosition) m_pPosition->Release();
}


// overriden to expose IMediaPosition and IMediaSeeking control interfaces

STDMETHODIMP
CCpndBase3OutputPin::NonDelegatingQueryInterface(REFIID riid, void **ppv)
{
    CheckPointer(ppv,E_POINTER);
    ValidateReadWritePtr(ppv,sizeof(PVOID));
    *ppv = NULL;

    if(riid == IID_IMediaPosition || riid == IID_IMediaSeeking) {

        // we should have an input pin by now
        ASSERT(m_pTransformFilter->m_ppInputs[0] != NULL);

        if(m_pPosition == NULL) {

            HRESULT hr = CreatePosPassThru(GetOwner(),
                FALSE,
                (IPin *)m_pTransformFilter->m_ppInputs[0],
                &m_pPosition);
            if(FAILED(hr)) {
                return hr;
            }
        }
        return m_pPosition->QueryInterface(riid, ppv);
    }
    else {
        return CBaseOutputPin::NonDelegatingQueryInterface(riid, ppv);
    }
}



// provides derived filter a chance to grab extra interfaces
HRESULT
CCpndBase3OutputPin::CheckConnect(IPin *pPin)
{
    // we should have an input connection first

    ASSERT(m_pTransformFilter->m_ppInputs[0] != NULL);
	if (m_pTransformFilter->InputIsAllDisconnected())
	{
        return E_UNEXPECTED;
    }

    HRESULT hr = m_pTransformFilter->CheckConnect(PINDIR_OUTPUT, pPin, 0);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseOutputPin::CheckConnect(pPin);
}


// provides derived filter a chance to release it's extra interfaces
HRESULT
CCpndBase3OutputPin::BreakConnect()
{
    //  Can't disconnect unless stopped
    ASSERT(IsStopped());
    m_pTransformFilter->BreakConnect(PINDIR_OUTPUT, 0);
    return CBaseOutputPin::BreakConnect();
}


// Let derived class know when the output pin is connected

HRESULT
CCpndBase3OutputPin::CompleteConnect(IPin *pReceivePin)
{
    HRESULT hr = m_pTransformFilter->CompleteConnect(PINDIR_OUTPUT, pReceivePin, 0);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseOutputPin::CompleteConnect(pReceivePin);
}


// check a given transform - must have selected input type first

HRESULT
CCpndBase3OutputPin::CheckMediaType(const CMediaType* pmtOut)
{
    HRESULT hr = m_pTransformFilter->CheckOutputType(pmtOut);
	return hr;
}


// called after we have agreed a media type to actually set it in which case
// we run the CheckTransform function to get the output format type again

HRESULT
CCpndBase3OutputPin::SetMediaType(const CMediaType* pmtOut)
{
	HRESULT hr;
    hr = CBasePin::SetMediaType(pmtOut);
    if(FAILED(hr)) {
        return hr;
    }
    return m_pTransformFilter->SetMediaType(PINDIR_OUTPUT, pmtOut, 0);
}


// pass the buffer size decision through to the main transform class

HRESULT
CCpndBase3OutputPin::DecideBufferSize(IMemAllocator * pAllocator, ALLOCATOR_PROPERTIES* pProp)
{
    return m_pTransformFilter->DecideBufferSize(pAllocator, pProp);
}


HRESULT
CCpndBase3OutputPin::Active()
{
	if (!IsConnected())
		return NOERROR;

	HRESULT hr = CBaseOutputPin::Active();
	if (FAILED(hr))
		return hr;

	ASSERT(!m_pTransformFilter->_workerThread->ThreadExists());

    m_pTransformFilter->_workerThread->Create();
    m_pTransformFilter->_workerThread->Init();
	m_pTransformFilter->_workerThread->Pause();

    return hr;
}

// return a specific media type indexed by iPosition

HRESULT
CCpndBase3OutputPin::GetMediaType(int iPosition, CMediaType *pMediaType)
{
	return m_pTransformFilter->GetMediaType(iPosition, pMediaType);
}


// Override this if you can do something constructive to act on the
// quality message.  Consider passing it upstream as well

// Pass the quality mesage on upstream.
STDMETHODIMP
CCpndBase3OutputPin::Notify(IBaseFilter * pSender, Quality q)
{
    UNREFERENCED_PARAMETER(pSender);
    ValidateReadPtr(pSender,sizeof(IBaseFilter));

    // First see if we want to handle this ourselves
    HRESULT hr = m_pTransformFilter->AlterQuality(q);
    if(hr!=S_FALSE) {
        return hr;        // either S_OK or a failure
    }

    // S_FALSE means we pass the message on.
    // Find the quality sink for our input pin and send it there

    ASSERT(m_pTransformFilter->m_ppInputs[0] != NULL);

	for(unsigned int i = 0; i < m_pTransformFilter->m_iInputPinCnt; i++)
	{
		if (m_pTransformFilter->m_ppInputs[i]->PassNotify(q) == S_OK)
			return S_OK;
	}
    return VFW_E_NOT_FOUND;

} // Notify


// the following removes a very large number of level 4 warnings from the microsoft
// compiler output, which are not useful at all in this case.
#pragma warning(disable:4514)

