/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/

#ifndef __CPNDBASE3__
#define __CPNDBASE3__

// ======================================================================
// This is the com object that represents a simple transform filter. It
// supports IBaseFilter, IMediaFilter and two pins through nested interfaces
// ======================================================================

#include <stdio.h>

class CCpndBase3Filter;

// ==================================================
// Implements the input pin
// ==================================================

class CCpndBase3InputPin : public CBaseInputPin
{
    friend class CCpndBase3Filter;

protected:
    CCpndBase3Filter *m_pTransformFilter;
	unsigned int 		m_bPinIdx;
	BOOL				m_bEOS;

public:

    CCpndBase3InputPin(
        TCHAR *pObjectName,
        CCpndBase3Filter *pTransformFilter,
        HRESULT * phr,
        LPCWSTR pName, 
		unsigned int pinIdx);
#ifdef UNICODE
    CCpndBase3InputPin(
        char *pObjectName,
        CCpndBase3Filter *pTransformFilter,
        HRESULT * phr,
        LPCWSTR pName,
		unsigned int pinIdx);
#endif

    STDMETHODIMP QueryId(LPWSTR * Id)
    {
	    WCHAR szbuf[20];
		swprintf(szbuf, L"In%d", m_bPinIdx);
        return AMGetWideString(szbuf, Id);
    }

	// --- Public methods -----

	BOOL IsEndOfStream();
    HRESULT CheckConnect(IPin *pPin);
    HRESULT BreakConnect();
    HRESULT CompleteConnect(IPin *pReceivePin);
    HRESULT CheckMediaType(const CMediaType* mtIn);
    HRESULT SetMediaType(const CMediaType* mt);

	// --- CBasePin ----
	virtual HRESULT Active();

    // --- IMemInputPin -----

    // here's the next block of data from the stream.
    // AddRef it yourself if you need to hold it beyond the end
    // of this call.
    STDMETHODIMP Receive(IMediaSample * pSample);
	STDMETHODIMP ReceiveCanBlock();

	// --- IBasePin ----
    STDMETHODIMP EndOfStream(void);
    STDMETHODIMP NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate);

	// --- IPin ---
    STDMETHODIMP BeginFlush(void);
    STDMETHODIMP EndFlush(void);

    // Check if it's OK to process samples
    virtual HRESULT CheckStreaming();

public:
    CMediaType& CurrentMediaType() { return m_mt; };

};

// ==================================================
// Implements the output pin
// ==================================================

class CCpndBase3OutputPin : public CBaseOutputPin
{
    friend class CCpndBase3Filter;

protected:
    CCpndBase3Filter *m_pTransformFilter;

public:

    // implement IMediaPosition by passing upstream
    IUnknown * m_pPosition;

    CCpndBase3OutputPin(
        TCHAR *pObjectName,
        CCpndBase3Filter *pTransformFilter,
        HRESULT * phr,
        LPCWSTR pName);
#ifdef UNICODE
    CCpndBase3OutputPin(
        CHAR *pObjectName,
        CCpndBase3Filter *pTransformFilter,
        HRESULT * phr,
        LPCWSTR pName);
#endif
    ~CCpndBase3OutputPin();

    // override to expose IMediaPosition
    STDMETHODIMP NonDelegatingQueryInterface(REFIID riid, void **ppv);

    // --- CBaseOutputPin ------------
    STDMETHODIMP QueryId(LPWSTR * Id)
    {
        return AMGetWideString(L"Out", Id);
    }

	// --- Public methods ----
    HRESULT CheckConnect(IPin *pPin);
    HRESULT BreakConnect();
    HRESULT CompleteConnect(IPin *pReceivePin);
    HRESULT CheckMediaType(const CMediaType* mtOut);
    HRESULT SetMediaType(const CMediaType *pmt);
    HRESULT DecideBufferSize(IMemAllocator * pAlloc, ALLOCATOR_PROPERTIES *pProp);

	// --- CBasePin ----
	virtual HRESULT Active();
    HRESULT GetMediaType(int iPosition,CMediaType *pMediaType);

    // inherited from IQualityControl via CBasePin
    STDMETHODIMP Notify(IBaseFilter * pSender, Quality q);

public:
    CMediaType& CurrentMediaType() { return m_mt; };
};



class CCpndBase3Worker : public CAMThread 
{
public:
	CCpndBase3Worker(CCpndBase3Filter* owner);
	virtual ~CCpndBase3Worker();

public:
    enum Command {CMD_INIT, CMD_PAUSE, CMD_RUN, CMD_STOP, CMD_EXIT};
    HRESULT Init(void)	{ return CallWorker(CMD_INIT); }
    HRESULT Exit(void)	{ return CallWorker(CMD_EXIT); }
    HRESULT Run(void)	{ return CallWorker(CMD_RUN); }
    HRESULT Pause(void)	{ return CallWorker(CMD_PAUSE); }
    HRESULT Stop(void)	{ return CallWorker(CMD_STOP); }

protected:
	DWORD	ThreadProc(void);
	HRESULT	DoBufferProcessingLoop(void);

protected:
	CCpndBase3Filter*		_owner;
};


class AM_NOVTABLE CCpndBase3Filter : public CBaseFilter
{
	friend class CCpndBase3InputPin;
	friend class CCpndBase3OutputPin;
	friend class CCpndBase3Worker;

public:

    // map getpin/getpincount for base enum of pins to owner
    // override this to return more specialised pin objects
    virtual int GetPinCount();
    virtual CBasePin * GetPin(int n);
    STDMETHODIMP FindPin(LPCWSTR Id, IPin **ppPin);

    // override state changes to allow derived transform filter
    // to control streaming start/stop
    STDMETHODIMP Stop();
    STDMETHODIMP Pause();

public:

    CCpndBase3Filter(TCHAR *, LPUNKNOWN, REFCLSID clsid, unsigned int inputPinCnt);
#ifdef UNICODE
    CCpndBase3Filter(CHAR *, LPUNKNOWN, REFCLSID clsid, unsigned int inputPinCnt);
#endif
    ~CCpndBase3Filter();


public:
	HRESULT				SetReferenceStream(int refStream);
	int					GetReferenceStream();

	HRESULT				SetFrameRate(int frameRate);
	int					GetFrameRate();

	HRESULT				SetTimeStampZero(bool flag);
	bool				GetTimeStampZero();

	HRESULT				SetTimeOffset(DWORD timeOffset);
	DWORD				GetTimeOffset();

	CCpndBase3InputPin*	GetConnectedInputPin(int priIdx = -1);


    // =================================================================
    // ----- override these bits ---------------------------------------
    // =================================================================

	virtual HRESULT		Transform(IMediaSample **ppInputs, IMediaSample *pOut);
    virtual HRESULT		CheckInputType(const CMediaType* mtIn, unsigned int pinIdx) PURE;
	virtual HRESULT		CheckOutputType(const CMediaType* mtOut) PURE;
    virtual HRESULT		GetMediaType(int iPosition, CMediaType *pMediaType) PURE;
	virtual HRESULT		DecideBufferSize(IMemAllocator * pAllocator, ALLOCATOR_PROPERTIES *pprop) PURE;

    // this goes in the factory template table to create new instances
    // static CCOMObject * CreateInstance(LPUNKNOWN, HRESULT *);


    // =================================================================
    // ----- Optional Override Methods           -----------------------
    // =================================================================

    // you can also override these if you want to know about streaming
    virtual HRESULT		StartStreaming();
    virtual HRESULT		StopStreaming();

    // override if you can do anything constructive with quality notifications
    virtual HRESULT		AlterQuality(Quality q);

    // override this to know when the media type is actually set
    virtual HRESULT		SetMediaType(PIN_DIRECTION direction, const CMediaType *pmt, unsigned int pinIdx);

    // chance to grab extra interfaces on connection
    virtual HRESULT		CheckConnect(PIN_DIRECTION dir,IPin *pPin, unsigned int pinIdx);
    virtual HRESULT		BreakConnect(PIN_DIRECTION dir, unsigned int pinIdx);
    virtual HRESULT		CompleteConnect(PIN_DIRECTION direction,IPin *pReceivePin, unsigned int pinIdx);

    // if you override Receive, you may need to override these three too
    virtual HRESULT		EndOfStream(unsigned int pinIdx);
    virtual HRESULT		BeginFlush(unsigned int pinIdx);
    virtual HRESULT		EndFlush(unsigned int pinIdx);
    virtual HRESULT		NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate, unsigned int pinIdx);

    // chance to customize the transform process
	HRESULT				InitializeOutputSample(int pinIdx, IMediaSample *pSample, IMediaSample **ppOutSample);

	// For Input pin Thread
	virtual HRESULT		Receive(unsigned int pinIdx, IMediaSample* pSample);
	virtual HRESULT		Compound(unsigned int pinIdx, IMediaSample* pInputSample, IMediaSample* pOutputSample) = 0;

	// For Filter Worker Thread
	virtual HRESULT		DeliverCompoundSample();
	virtual HRESULT		FilterCompound(IMediaSample* pOutputSample) = 0;

	void				AdjustFrameRate();
	void				SetupMediaSampleProperty(IMediaSample* pOutSample);


#ifdef PERF
    // Override to register performance measurement with a less generic string
    // You should do this to avoid confusion with other filters
    virtual void RegisterPerfId()
         {m_idTransform = MSR_REGISTER(TEXT("Transform"));}
#endif // PERF


private:
	BOOL		InputIsAllDisconnected();


// implementation details

protected:

#ifdef PERF
    int m_idTransform;                 // performance measuring id
#endif
    BOOL m_bEOSDelivered;              // have we sent EndOfStream
    BOOL m_bSampleSkipped;             // Did we just skip a frame
    BOOL m_bQualityChanged;            // Have we degraded?

    // critical section protecting filter state.
    CCritSec m_csFilter;

    // critical section stopping state changes (ie Stop) while we're
    // processing a sample.
    //
    // This critical section is held when processing
    // events that occur on the receive thread - Receive() and EndOfStream().
    //
    // If you want to hold both m_csReceive and m_csFilter then grab
    // m_csFilter FIRST - like CCpndBase3Filter::Stop() does.
    CCritSec m_csReceive;

	CCritSec m_csProp;

    // these hold our input and output pins
	CCpndBase3InputPin			**m_ppInputs;
    CCpndBase3OutputPin			*m_pOutput;

	unsigned int				m_iInputPinCnt;

	int							_refStream;
	DWORD						_interval;
	bool						_timeStampZero;
	DWORD						_timeOffset;

	REFERENCE_TIME				_startTime;
	REFERENCE_TIME				_streamTime;
	bool						_streamTimeIsValid;

private:
	CCpndBase3Worker			*_workerThread;
};

#endif /* __CPNDBASE__ */


