/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/

#include <windows.h>
#include <streams.h>
#include <initguid.h>
#include <olectl.h>
#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif

#include <stdio.h>
#include <limits.h>
#include <math.h>

#include <flFilterCommon/flPointerInfo.h>
#include <flTypes/flUIDs.h>
#include <flTypes/IVideoOverlayMixer.h>

#include "cpndbase3.h"

#include "videooverlaymixerprop.h"
#include "videooverlaymixer.h"


#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue


const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
    &MEDIATYPE_Video,			// Major type
    &MEDIASUBTYPE_NULL			// Minor type
};

const AMOVIESETUP_MEDIATYPE sudPinTypesI =
{
    &MEDIATYPE_Stream,			// Major type
    &MEDIASUBTYPE_PointerInfo	// Minor type
};

const AMOVIESETUP_PIN psudPins[] =
{
	{
		L"Input1",          // String pin name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Output",          // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Input2",          // String pin name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Output",          // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Output",          // String pin name
		FALSE,              // Is it rendered
		TRUE,               // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Input",           // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Input Pointer",   // Pins string name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Are we allowed none
		FALSE,              // And allowed many
		&CLSID_NULL,        // Connects to filter
		NULL,               // Connects to pin
		1,                  // Number of types
		&sudPinTypesI       // Pin information
	}
};


const AMOVIESETUP_FILTER sudVideoOverlayMixer =
{
    &CLSID_VideoOverlayMixer,   // Filter CLSID
    L"NIME 2-input Video Overlay Mixer",   // Filter name
    MERIT_DO_NOT_USE,       // Its merit
    4,                      // Number of pins
    psudPins                // Pin details
};


// List of class IDs and creator functions for the class factory. This
// provides the link between the OLE entry point in the DLL and an object
// being created. The class factory will call the static CreateInstance

CFactoryTemplate g_Templates[2] = {

    { L"VideoOverlayMixer"
    , &CLSID_VideoOverlayMixer
    , CVideoOverlayMixer::CreateInstance
    , NULL
    , &sudVideoOverlayMixer }
  ,
    { L"VideoOverlayMixerProp"
    , &CLSID_VideoOverlayMixerPropertyPage
    , CVideoOverlayMixerProperties::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);


#define ERASETIME				5000

#define DEF_TIMESTAMP			TS_BASE
#define DEF_MASK				M_RECTANGLE
#define DEF_BLENDING			BL_SIMPLE
#define DEF_CUTPOSX				0
#define DEF_CUTPOSY				0
#define DEF_POINTERINDISABLE	false
#define DEF_GRIDX				6
#define DEF_GRIDY				6

//
// Constructor
//
CVideoOverlayMixer::CVideoOverlayMixer(TCHAR *tszName,LPUNKNOWN punk,HRESULT *phr) :
CCpndBase3Filter(tszName, punk, CLSID_VideoOverlayMixer, 2),
CPersistStream(punk, phr),
_pPointer(NULL),
_timeStamp(DEF_TIMESTAMP),
_mask(DEF_MASK),
_blending(DEF_BLENDING),
_compPosX(DEF_CUTPOSX),
_compPosY(DEF_CUTPOSY),
_pointerInDisable(DEF_POINTERINDISABLE),
_gridX(DEF_GRIDX),
_gridY(DEF_GRIDY),
_baseBufSize(0),
_baseBuf(NULL),
_maskCreator(NULL),
_maskBuf(NULL),
_maskBufSize(0),
_eraseBase(false),
_eraseAdd(false),
_baseReceiveTime(0),
_addReceiveTime(0),
_srcDataLen(0),
_srcData(NULL)
{
    ASSERT(tszName);
    ASSERT(phr);

} // VideoOverlayMixer

CVideoOverlayMixer::~CVideoOverlayMixer()
{
	delete _pPointer;
	delete _baseBuf;
	delete _maskCreator;
	delete _srcData;
}

//
// CreateInstance
//
// Provide the way for COM to create a CVideoOverlayMixer object
//
CUnknown * WINAPI CVideoOverlayMixer::CreateInstance(LPUNKNOWN punk, HRESULT *phr) {

    CVideoOverlayMixer *pNewObject = new CVideoOverlayMixer(NAME("VideoOverlayMixer"), punk, phr);
    if (pNewObject == NULL) {
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;

} // CreateInstance


//
// NonDelegatingQueryInterface
//
// Reveals IVideoOverlayMixer and ISpecifyPropertyPages
//
STDMETHODIMP CVideoOverlayMixer::NonDelegatingQueryInterface(REFIID riid, void **ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IVideoOverlayMixer){
        return GetInterface((IVideoOverlayMixer *) this, ppv);
	} else if (riid == IID_IPersistStream){
		return GetInterface((IPersistStream *) this, ppv);
    } else if (riid == IID_ISpecifyPropertyPages){
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    } else {
        return CCpndBase3Filter::NonDelegatingQueryInterface(riid, ppv);
    }

} // NonDelegatingQueryInterface

int
CVideoOverlayMixer::GetPinCount()
{
	return 4;
}

CBasePin*
CVideoOverlayMixer::GetPin(int n)
{
	HRESULT hr= S_OK;
    WCHAR szbuf[20];

	if(_pPointer == NULL)
	{
		swprintf(szbuf, L"Pointer Input");
		_pPointer = new CPointerInputPin(NAME("Pointer Input Pin"), this, &hr, szbuf);

        ASSERT(SUCCEEDED(hr));
        if(_pPointer == NULL) {
            return NULL;
        }
	}

	if (n < 3)
	{
		return CCpndBase3Filter::GetPin(n);
	}
	else if (n == 3)
	{
		return _pPointer;
	}
	else
	{
		return NULL;
	}
}

STDMETHODIMP
CVideoOverlayMixer::FindPin(LPCWSTR Id, IPin **ppPin)
{
	*ppPin = NULL;

	if (!lstrcmpW(Id, L"In0"))
	{
		*ppPin = GetPin(0);
	} else if (!lstrcmpW(Id, L"In1"))
	{
		*ppPin = GetPin(1);
	} else if (!lstrcmpW(Id, L"Out"))
	{
		*ppPin = GetPin(2);
	} else if (!lstrcmpW(Id, L"Pointer Input"))
	{
		*ppPin = GetPin(3);
	}

	HRESULT hr = NOERROR;
	if(*ppPin)
	{
		(*ppPin)->AddRef();
	}
	else
	{
		hr = E_OUTOFMEMORY;
	}
	return hr;
}

// Called input pin Thread
HRESULT
CVideoOverlayMixer::Compound(unsigned int pinIdx, IMediaSample* pInputSample, IMediaSample* pOutputSample)
{
	if (pinIdx == 0)
	{
		_eraseBase = (_baseReceiveTime == 0) ? true : (ERASETIME < GetTickCount() - _baseReceiveTime);
		_baseReceiveTime = GetTickCount();

		Copy(pInputSample, pOutputSample);
	}
	else if (pinIdx == 1)
	{
		_eraseAdd = (_addReceiveTime == 0) ? true : (ERASETIME < GetTickCount() - _addReceiveTime);
		_addReceiveTime = GetTickCount();

		long dataLen = pInputSample->GetActualDataLength();
		if (_srcDataLen < dataLen)
		{
			delete _srcData;
			_srcData = new BYTE[dataLen];
		}
		_srcDataLen = dataLen;
        
		BYTE *ptr;
		pInputSample->GetPointer(&ptr);
		CopyMemory(_srcData, ptr, _srcDataLen);
	}
	return NOERROR;
}

// Called Filter Worker Thread
HRESULT
CVideoOverlayMixer::FilterCompound(IMediaSample* pOutputSample)
{
	long currentTime = GetTickCount();

	if (_eraseBase || ERASETIME < currentTime - _baseReceiveTime)
		Fill(pOutputSample);

	if (_eraseAdd || ERASETIME < currentTime - _addReceiveTime || _srcData == NULL)
		return NOERROR;

	VIDEOINFO* ivinfo = (VIDEOINFO *)m_ppInputs[1]->CurrentMediaType().Format();
	if (_srcDataLen < (long)ivinfo->bmiHeader.biSizeImage)
		return NOERROR;

	long destDataLen = pOutputSample->GetActualDataLength();
	BYTE *destData;
	pOutputSample->GetPointer(&destData);

	Compound(_srcData, _srcDataLen, destData, destDataLen);

	return NOERROR;
}


HRESULT
CVideoOverlayMixer::Copy(IMediaSample *pSource, IMediaSample *pDest)
{
    BYTE *pSourceBuffer, *pDestBuffer;

	pSource->GetPointer(&pSourceBuffer);
    pDest->GetPointer(&pDestBuffer);

    long lSourceSize = pSource->GetActualDataLength();
    long lDestSize = pDest->GetActualDataLength();

	if (lSourceSize != lDestSize)
	{
		ASSERT(FALSE);
		return E_FAIL;
	}

	CopyMemory((PVOID)pDestBuffer, (PVOID)pSourceBuffer, lDestSize);
    return NOERROR;
}


HRESULT	
CVideoOverlayMixer::Fill(IMediaSample *pDest)
{
    BYTE *pDestBuffer;

    pDest->GetPointer(&pDestBuffer);

    long lDestSize = pDest->GetActualDataLength();

	if (_baseBufSize != lDestSize)
	{
		ASSERT(FALSE);
		return E_FAIL;
	}

	CopyMemory((PVOID)pDestBuffer, (PVOID)_baseBuf, lDestSize);
    return NOERROR;
}

HRESULT
CVideoOverlayMixer::Compound(IMediaSample *pSource, IMediaSample *pDest)
{
	long srcDataLen = pSource->GetActualDataLength();
	long destDataLen = pDest->GetActualDataLength();

	BYTE *srcData, *destData;
	pSource->GetPointer(&srcData);
	pDest->GetPointer(&destData);

	return Compound(srcData, srcDataLen, destData, destDataLen);
}

HRESULT
CVideoOverlayMixer::Compound(BYTE* pSourceBuffer, long lSourceSize,
						 BYTE* pDestBuffer, long lDestSize)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);

	BYTE *pSB, *pDB;
	long sw, sh, dw, dh;
	int x1, y1;

	pSB = pSourceBuffer;
	pDB = pDestBuffer;

	CMediaType imt = m_ppInputs[1]->CurrentMediaType();
	CMediaType omt = m_pOutput->CurrentMediaType();

	VIDEOINFO* ivinfo = (VIDEOINFO *)imt.Format();
	VIDEOINFO* ovinfo = (VIDEOINFO *)omt.Format();

	sw = ivinfo->bmiHeader.biWidth;
	sh = ivinfo->bmiHeader.biHeight;
	dw = ovinfo->bmiHeader.biWidth;
	dh = ovinfo->bmiHeader.biHeight;


	if (_pointerInDisable || _pPointer->IsConnected() == FALSE)
	{
		x1 = _compPosX;
		y1 = dh - sh - _compPosY;
	} else
	{
		flFloat cx, cy;
		_pointerInfo.getNormalizedPointerPosition(cx, cy);
		x1 = (int)(float(dw) * cx);
		y1 = dh -sh - (int)(float(dh) * cy);
	}


	if (0 <= x1 + sw && x1 <= dw &&
		0 <= y1 + sh && y1 <= dh)
	{

		int bCnt = ivinfo->bmiHeader.biBitCount / 8;
		long bcSW = bCnt * sw;
		long bcDW = bCnt * dw;

		x1 *= bCnt;
		y1 *= bCnt;

		// Seek
		pDB += dw * y1;
		pDB += x1;

		{
			int size, tsize, inc, ofst;
			float alph, ialph;
			BYTE *pDBt, *pSBt;

			inc = 0;

			switch(bCnt)
			{
			case 2:
				// RGB555, RGB565
				for(unsigned int i = 0; i < _maskBufSize; i+=2)
				{
					size = _maskBuf[i];
					alph = (float)_maskBuf[i + 1];

					tsize = size;
					ofst = 0;

					if (x1 + inc < 0)
					{
						ofst = -(x1 + inc);
						tsize -= ofst;
						if (tsize < 0)
							tsize = 0;
					}

					if (bcDW < tsize + inc + x1)
					{
						tsize = bcDW - (inc + x1);
						if (tsize < 0)
							tsize = 0;
					}

					if (tsize > bcDW)
						tsize = bcDW;

					pDBt = pDB + ofst;
					pSBt = pSB + ofst;

					if (pDestBuffer <= pDBt &&
						pDBt + tsize <= pDestBuffer + lDestSize)
					{
						if (alph >= 255.0f)
						{
							CopyMemory(pDBt, pSBt, tsize);
						}
						else if (alph != 0.0f)
						{
							alph /= 255.0f;
							ialph = 1.0f - alph;

							for(int j = 0; j < tsize; j += 2)
							{
								short *sd = (short *)(pDBt + j);
								short *ss = (short *)(pSBt + j);

								*sd = ( (short)( ( (float)((*sd & 0xf800) >> 11) * ialph) +
												( (float)((*ss & 0xf800) >> 11) * alph) ) << 11 ) +
									  ( (short)( ( (float)((*sd & 0x3e0) >> 5) * ialph) +
												( (float)((*ss & 0x3e0) >> 5) * alph) ) << 5) +
									  ( (short)( ( (float)(*sd & 0x1f) * ialph ) +
												( (float)(*ss & 0x1f) * alph ) ) );
							}
						}
					}

					pSB += size;
					pDB += size;
					inc += size;

					if (inc >= sw * 2)
					{
						pDB += (bcDW - bcSW);
						inc = 0;
					}
				} // for ..
				break;

			case 3:
				// RGB24
				for(unsigned int i = 0; i < _maskBufSize; i+=2)
				{
					size = _maskBuf[i];
					alph = (float)_maskBuf[i + 1];

					tsize = size;
					ofst = 0;

					if (x1 + inc < 0)
					{
						ofst = -(x1 + inc);
						tsize -= ofst;
						if (tsize < 0)
							tsize = 0;
					}

					if (bcDW < tsize + inc + x1)
					{
						tsize = bcDW - (inc + x1);
						if (tsize < 0)
							tsize = 0;
					}

					if (tsize > bcDW)
						tsize = bcDW;

					pDBt = pDB + ofst;
					pSBt = pSB + ofst;

					if (pDestBuffer <= pDBt &&
						pDBt + tsize <= pDestBuffer + lDestSize)
					{
						if (alph >= 255.0f)
						{
							CopyMemory(pDBt, pSBt, tsize);
						}
						else if (alph != 0.0f)
						{
							alph /= 255.0f;
							ialph = 1.0f - alph;

							for(int j = 0; j < tsize; j++)
							{
								pDBt[j] = BYTE(pDBt[j] * ialph) + BYTE(pSBt[j] * alph);
							}
						}
					}

					pSB += size;
					pDB += size;
					inc += size;

					if (inc >= sw * 3)
					{
						pDB += (bcDW - bcSW);
						inc = 0;
					}
				} // for ..
				break;
			}
		}

	}

	return NOERROR;
}

HRESULT
CVideoOverlayMixer::AllocBaseBuffer(unsigned int size)
{
	delete [] _baseBuf;

	_baseBuf = new BYTE[size];
	if (!_baseBuf)
	{
		_baseBufSize = 0;
		return E_FAIL;
	}
	_baseBufSize = size;

	return NOERROR;
}

HRESULT
CVideoOverlayMixer::DrawBaseBufferInBGColor()
{
	if (!_baseBuf || !_baseBufSize)
		return E_FAIL;

	CMediaType cmt = m_pOutput->CurrentMediaType();

	if (*(cmt.FormatType()) != FORMAT_VideoInfo)
	{
		ZeroMemory((PVOID)_baseBuf, _baseBufSize);
		return NOERROR;
	}

	if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB555))
	{
		const BYTE color[2] = { 0x1f, 0x00 };

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 2; i++)
		{
			CopyMemory((PVOID)pBaseBuf, (PVOID)color, sizeof(BYTE) * 2);
			pBaseBuf += 2;
		}
	}
	else if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB565))
	{
		const BYTE color[2] = { 0x1f, 0x00 };

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 2; i++)
		{
			CopyMemory((PVOID)pBaseBuf, (PVOID)color, sizeof(BYTE) * 2);
			pBaseBuf += 2;
		}
	}
	else if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB24))
	{
		const BYTE color[3] = { 0xff, 0x00, 0x00 };

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 3; i++)
		{
			CopyMemory((PVOID)pBaseBuf, color, sizeof(BYTE) * 3);
			pBaseBuf += 3;
		}
	}
	else
	{
		ZeroMemory(_baseBuf, _baseBufSize);
	}

	return NOERROR;
}

HRESULT
CVideoOverlayMixer::AllocMaskBuffer(int width, int height, WORD bitCnt)
{
	if (width == 0 || height == 0)
		return E_FAIL;

	delete _maskCreator;

	_maskCreator = new MaskCreator(width, height, (unsigned int)bitCnt / 8);
	_maskCreator->createAllMasks(_gridX, _gridY);
	_maskCreator->getMask(_mask, _blending, &_maskBuf, &_maskBufSize);

	return NOERROR;
}

//
// CheckInputType
//
// Check the input type is OK, return an error otherwise
//
HRESULT CVideoOverlayMixer::CheckInputType(const CMediaType *mtIn, unsigned int pinIdx)
{
	ASSERT(m_ppInputs[0]);

	if (*mtIn->FormatType() == FORMAT_VideoInfo)
	{
		if (!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB555) &&
			!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB565) &&
			!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB24))
			return E_INVALIDARG;
		else
			return NOERROR;
	}

	return E_INVALIDARG;

} // CheckInputType

HRESULT CVideoOverlayMixer::CheckOutputType(const CMediaType *mtOut)
{
	ASSERT(m_ppInputs[0]);

	if (*mtOut->FormatType() != FORMAT_VideoInfo)
		return E_INVALIDARG;

	if (!m_ppInputs[0]->IsConnected())
		return E_INVALIDARG;

	const CMediaType& mtIn = m_ppInputs[0]->CurrentMediaType();
	if (!IsEqualGUID(*mtIn.Subtype(), *mtOut->Subtype()))
		return E_INVALIDARG;

	VIDEOINFO* pivi = (VIDEOINFO *)mtIn.Format();
	VIDEOINFO* povi = (VIDEOINFO *)mtOut->Format();

	if (pivi->bmiHeader.biWidth != povi->bmiHeader.biWidth ||
		pivi->bmiHeader.biHeight != povi->bmiHeader.biHeight)
		return E_INVALIDARG;

	return NOERROR;
}

//
// DecideBufferSize
//
// Tell the output pin's allocator what size buffers we
// require. Can only do this when the input is connected
//
HRESULT CVideoOverlayMixer::DecideBufferSize(IMemAllocator *pAlloc,ALLOCATOR_PROPERTIES *pProp)
{
    if (m_ppInputs[0]->IsConnected() == FALSE)
        return E_UNEXPECTED;

	ASSERT(pAlloc);
	ASSERT(pProp);
	ASSERT(m_ppInputs[0]->CurrentMediaType().bFixedSizeSamples);

	VIDEOINFOHEADER *pvi = (VIDEOINFOHEADER *)m_pOutput->CurrentMediaType().pbFormat;
	pvi->bmiHeader.biSizeImage = GetBitmapSize(&pvi->bmiHeader);
	ASSERT(pvi->bmiHeader.biSizeImage != 0);

	pProp->cBuffers = 1;
	pProp->cbBuffer = pvi->bmiHeader.biSizeImage;

	ALLOCATOR_PROPERTIES Actual;
	HRESULT hr = pAlloc->SetProperties(pProp, &Actual);
	if (FAILED(hr))
		return hr;

	ASSERT(Actual.cBuffers == 1);

	if (Actual.cBuffers < pProp->cBuffers ||
		Actual.cbBuffer < pProp->cbBuffer)
		return E_FAIL;


    /// Allocate BASE BUFFER
	hr = AllocBaseBuffer(Actual.cbBuffer);
	if (FAILED(hr))
		return E_FAIL;

	/// Draw BASE BUFFER in BG COLOR
	hr = DrawBaseBufferInBGColor();
	if (FAILED(hr))
		return E_FAIL;

    return NOERROR;
}



HRESULT
CVideoOverlayMixer::SetMediaType(PIN_DIRECTION direction, const CMediaType *pmt, unsigned int pinIdx)
{
	ASSERT(m_ppInputs[0]);

	HRESULT hr = NOERROR;

	// Check tow input pin VideoResolution
	if (direction == PINDIR_INPUT)
	{
		if (pinIdx == 0 && m_ppInputs[1]->IsConnected())
		{
			CMediaType cmt = m_ppInputs[1]->CurrentMediaType();
			if (!IsEqualGUID(*pmt->Subtype(), *cmt.Subtype()))
			{
				hr = E_FAIL;
			}
		}
		else if (pinIdx == 1 && m_ppInputs[0]->IsConnected())
		{
			CMediaType cmt = m_ppInputs[0]->CurrentMediaType();
			if (!IsEqualGUID(*pmt->Subtype(), *cmt.Subtype()))
			{
				hr = E_FAIL;
			}
		}
	}

	if (hr == E_FAIL)
		return hr;

	// Create Mask
	if (direction == PINDIR_INPUT && pinIdx == 1)
	{
		VIDEOINFO* vinfo = (VIDEOINFO *)pmt->Format();
		if (vinfo)
		{
			hr = AllocMaskBuffer(
						vinfo->bmiHeader.biWidth,
						vinfo->bmiHeader.biHeight,
						vinfo->bmiHeader.biBitCount);
		} else
		{
			hr = E_FAIL;
		}
	}

	// Reconnect 
    if(pinIdx == 0 && m_ppInputs[pinIdx]->IsConnected() && m_pOutput->IsConnected())
	{
        FILTER_INFO fInfo;
        QueryFilterInfo( &fInfo );

        if (direction == PINDIR_INPUT && *pmt != m_pOutput->CurrentMediaType())
            fInfo.pGraph->Reconnect( m_pOutput );

        QueryFilterInfoReleaseGraph( fInfo );
    }

	return hr;
}


//
// GetMediaType
//
// I support one type, namely the type of the input pin
// We must be connected to support the single output type
//
HRESULT CVideoOverlayMixer::GetMediaType(int iPosition, CMediaType *pMediaType)
{
    CAutoLock cAutoLock(&_videoOverlayMixerLock);
    if (iPosition < 0)
	{
        return E_INVALIDARG;
    }
    if (iPosition > 0)
	{
        return VFW_S_NO_MORE_ITEMS;
    }
	
	ASSERT(m_ppInputs[0] != NULL);

	if (m_ppInputs[0]->IsConnected() == FALSE)
        return E_UNEXPECTED;

	CMediaType iMediaType = m_ppInputs[0]->CurrentMediaType();
	*pMediaType = iMediaType;

	VIDEOINFO *vinfo = (VIDEOINFO *)pMediaType->Format();
	pMediaType->SetSampleSize(GetBitmapSize(&vinfo->bmiHeader));

	return NOERROR;

} // GetMediaType


STDMETHODIMP
CVideoOverlayMixer::get_TimeStamp(unsigned int *timeStamp)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*timeStamp = (unsigned int)_timeStamp;

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_TimeStamp(unsigned int timeStamp)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	if (_timeStamp != timeStamp)
	{
		_timeStamp = (TimeStampEnum)timeStamp;
		SetDirty(TRUE);

		if (_timeStamp == TS_ORIGINAL)
			_refStream = -1;
		else
			_refStream = (int)_timeStamp;

		ReconnectPin((IPin *)m_pOutput, NULL);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::get_TimeStampZero(bool *flag)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*flag = GetTimeStampZero();
	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_TimeStampZero(bool flag)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);

	if (GetTimeStampZero() != flag)
	{
		SetTimeStampZero(flag);
		SetDirty(TRUE);
	}
	
	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::get_TimeOffset(int *timeOffset)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*timeOffset = (int)GetTimeOffset();
	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_TimeOffset(int timeOffset)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);

	if (GetTimeOffset() != timeOffset)
	{
		SetTimeOffset(timeOffset);
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::get_FrameRate(int *rate)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*rate = GetFrameRate();

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_FrameRate(int rate)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	if (GetFrameRate() != rate)
	{
		SetFrameRate(rate);
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::get_Mask(unsigned int *mask)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*mask = (unsigned int)_mask;

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_Mask(unsigned int mask)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	if (_mask != mask)
	{
		_mask = (MaskEnum)mask;
		SetDirty(TRUE);

		if (_maskCreator != NULL)
	        _maskCreator->getMask(_mask, _blending, &_maskBuf, &_maskBufSize);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::get_Blending(unsigned int *blending)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*blending = (unsigned int)_blending;

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_Blending(unsigned int blending)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	if (_blending != blending)
	{
		_blending = (BlendingEnum)blending;
		SetDirty(TRUE);

		if (_maskCreator != NULL)
			_maskCreator->getMask(_mask, _blending, &_maskBuf, &_maskBufSize);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::get_CompPosition(int* x, int* y)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*x = _compPosX;
	*y = _compPosY;

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_CompPosition(int x, int y)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	if (_compPosX != x || _compPosY != y)
	{
		_compPosX = x;
		_compPosY = y;
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::get_PointerInDisable(bool* flag)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*flag = _pointerInDisable;

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_PointerInDisable(bool flag)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	if (_pointerInDisable != flag)
	{
		_pointerInDisable = flag;
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::get_CheckerGridXY(unsigned int* x, unsigned int* y)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	*x = _gridX;
	*y = _gridY;

	return NOERROR;
}

STDMETHODIMP
CVideoOverlayMixer::put_CheckerGridXY(unsigned int x, unsigned int y)
{
	CAutoLock cAutoLock(&_videoOverlayMixerLock);
	if (_gridX != x || _gridY != y)
	{
		_gridX = x;
		_gridY = y;

		if (_gridX < 2)
			_gridX = 2;
		else if (_gridX > 20)
			_gridX = 20;

		if (_gridY < 2)
			_gridY = 2;
		else if (_gridY > 20)
			_gridY = 20;

		if (_maskCreator != NULL)
		{
			_maskCreator->recreateCheckerMask(_gridX, _gridY);
			_maskCreator->getMask(_mask, _blending, &_maskBuf, &_maskBufSize);
		}

		SetDirty(TRUE);
	}

	return NOERROR;
}

#define WRITEOUT(var)  hr = pStream->Write(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define WRITENOUT(var,n)  hr = pStream->Write(var, n, NULL); \
if (FAILED(hr)) return hr;
#define READIN(var)    hr = pStream->Read(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define READNIN(var,n)    hr = pStream->Read(var, n, NULL); \
if (FAILED(hr)) return hr;


// IPersitStream
STDMETHODIMP
CVideoOverlayMixer::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_VideoOverlayMixer;
	return S_OK;
} // GetClassID

HRESULT
CVideoOverlayMixer::WriteToStream(IStream *pStream)
{
	int frameRate = GetFrameRate();
	bool timeStampZero = GetTimeStampZero();
	int timeOffset = GetTimeOffset();

	HRESULT hr;
	WRITEOUT(_timeStamp);
	WRITEOUT(timeStampZero);
	WRITEOUT(timeOffset);
	WRITEOUT(frameRate);
	WRITEOUT(_mask);
	WRITEOUT(_blending);
	WRITEOUT(_compPosX);
	WRITEOUT(_compPosY);
	WRITEOUT(_pointerInDisable);
	WRITEOUT(_gridX);
	WRITEOUT(_gridY);
	return NOERROR;
	
} // WriteToStream

HRESULT
CVideoOverlayMixer::ReadFromStream(IStream *pStream)
{
	int frameRate;
	bool timeStampZero;
	int timeOffset;

	HRESULT hr;
	READIN(_timeStamp);
	READIN(timeStampZero);
	READIN(timeOffset);
	READIN(frameRate);
	READIN(_mask);
	READIN(_blending);
	READIN(_compPosX);
	READIN(_compPosY);
	READIN(_pointerInDisable);
	READIN(_gridX);
	READIN(_gridY);

	SetFrameRate(frameRate);
	SetTimeStampZero(timeStampZero);
	SetTimeOffset(timeOffset);

	if (_timeStamp == TS_ORIGINAL)
		_refStream = -1;
	else
		_refStream = (int)_timeStamp;

	return NOERROR;
	
} // ReadFromStream

DWORD
CVideoOverlayMixer::GetSoftwareVersion(void)
{
	return 1;
} // GetSoftwareVersion

int
CVideoOverlayMixer::SizeMax()
{
	return sizeof(unsigned int) * 7 + sizeof(bool) * 1;
} // SizeMax


//
// GetPages
//
// This is the sole member of ISpecifyPropertyPages
// Returns the clsid's of the property pages we support
//
STDMETHODIMP CVideoOverlayMixer::GetPages(CAUUID *pPages)
{
    pPages->cElems = 1;
    pPages->pElems = (GUID *) CoTaskMemAlloc(sizeof(GUID));
    if (pPages->pElems == NULL) {
        return E_OUTOFMEMORY;
    }
    *(pPages->pElems) = CLSID_VideoOverlayMixerPropertyPage;
    return NOERROR;

} // GetPages

HRESULT
CVideoOverlayMixer::ReceivePointer(IMediaSample *pSource, CPointerInputPin *ppin)
{
	// Get Media Sample data
	flLong dataLength;
	BYTE *data;
	dataLength = pSource->GetActualDataLength();
	pSource->GetPointer(&data);

	// Setup New PointerInfo
	CAutoLock autoLock(&_videoOverlayMixerLock);
	_pointerInfo.setChanged(data, dataLength);

	return NO_ERROR;
}

//////////////////////////////////////////////////////////////////////////////

#define NUM_MASKTYPE			6
#define MIDX(_mask, _blending)	((_mask) + CVideoOverlayMixer::NUM_M * (_blending))

MaskCreator::MaskCreator(unsigned int width, unsigned int height, unsigned int bCnt) :
_width(width),
_height(height),
_bCnt(bCnt)
{
	_maskBufSize = new int[NUM_MASKTYPE];
	_maskBuf = new int *[NUM_MASKTYPE];
	for(unsigned int i = 0; i < NUM_MASKTYPE; i++)
	{
		_maskBufSize[i] = 0;
		_maskBuf[i] = NULL;
	}
}

MaskCreator::~MaskCreator()
{
	for(unsigned int i = 0; i < NUM_MASKTYPE; i++)
		delete _maskBuf[i];
	delete _maskBufSize;
	delete [] _maskBuf;
}

void
MaskCreator::createAllMasks(unsigned int cgX, unsigned int cgY)
{
#ifdef DEBUG
	OutputDebugString("MaskCreator::createAllMask() called.\n");
#endif

	unsigned int i;
	unsigned int index;

	// Clear All Buffer Table
	for(i = 0; i < NUM_MASKTYPE; i++)
	{
		delete _maskBuf[i];
		_maskBuf[i] = NULL;
		_maskBufSize[i] = 0;
	}


	// Make Mask Map
	BYTE **map;
	map = new BYTE *[_width];
	for(i = 0; i < _width; i++)
		map[i] = new BYTE[_height];


	// Rectangle
	{
		index = MIDX(CVideoOverlayMixer::M_RECTANGLE, CVideoOverlayMixer::BL_SIMPLE);
		getRectMap(&map);
		createMask(&_maskBufSize[index], &_maskBuf[index], map);
	}	

	// Blend Rectangle
	{
		index = MIDX(CVideoOverlayMixer::M_RECTANGLE, CVideoOverlayMixer::BL_VAGUELY);
		getBlendRectMap(&map);
		createMask(&_maskBufSize[index], &_maskBuf[index], map);
	}

	// Ellipse
	{
		index = MIDX(CVideoOverlayMixer::M_ELLIPSE, CVideoOverlayMixer::BL_SIMPLE);
        getEllipMap(&map);
		createMask(&_maskBufSize[index], &_maskBuf[index], map);
	}

	// Blend Ellipse
	{
		index = MIDX(CVideoOverlayMixer::M_ELLIPSE, CVideoOverlayMixer::BL_VAGUELY);
		getBlendEllipMap(&map);
		createMask(&_maskBufSize[index], &_maskBuf[index], map);
	}

	// Checker
	{
		index = MIDX(CVideoOverlayMixer::M_CHECKERS, CVideoOverlayMixer::BL_SIMPLE);
		getCheckMap(&map, cgX, cgY);
		createMask(&_maskBufSize[index], &_maskBuf[index], map);
	}

	// Blend Checker
	{
		index = MIDX(CVideoOverlayMixer::M_CHECKERS, CVideoOverlayMixer::BL_VAGUELY);
		getBlendCheckMap(&map, cgX, cgY);
		createMask(&_maskBufSize[index], &_maskBuf[index], map);
	}

	// Delete Mask Map
	for(i = 0; i < _width; i++)
		delete map[i];
	delete [] map;
}

void
MaskCreator::recreateCheckerMask(unsigned int cgX, unsigned int cgY)
{
#ifdef DEBUG
	OutputDebugString("MaskCreator::recreateCheckerMask() called.\n");
#endif

	unsigned int index;
	unsigned int i;

	// Make Mask Map
	BYTE **map;
	map = new BYTE *[_width];
	for(i = 0; i < _width; i++)
		map[i] = new BYTE[_height];

	// Checker
	{
		index = MIDX(CVideoOverlayMixer::M_CHECKERS, CVideoOverlayMixer::BL_SIMPLE);

		delete _maskBuf[index];

		getCheckMap(&map, cgX, cgY);
		createMask(&_maskBufSize[index], &_maskBuf[index], map);
	}

	// Blend Checker
	{
		index = MIDX(CVideoOverlayMixer::M_CHECKERS, CVideoOverlayMixer::BL_VAGUELY);

		delete _maskBuf[index];

		getBlendCheckMap(&map, cgX, cgY);
		createMask(&_maskBufSize[index], &_maskBuf[index], map);
	}

	// Delete Mask Map
	for(i = 0; i < _width; i++)
		delete map[i];
	delete [] map;

}

void
MaskCreator::getMask(CVideoOverlayMixer::MaskEnum mask,
					 CVideoOverlayMixer::BlendingEnum blending,
					 int** maskBuf, unsigned int* maskBufSize)
{
	unsigned int index = MIDX(mask, blending);

	if (index >= NUM_MASKTYPE)
	{
		maskBuf = NULL;
		*maskBufSize = 0;
		return ;
	}

	*maskBuf = _maskBuf[index];
	*maskBufSize = _maskBufSize[index];

#ifdef DEBUG
/*
	{
		char buffer[256];
		sprintf(buffer, "MaskCreator::getMask() called. Mask Buffer Size:%d\n", _maskBufSize[index]);
		OutputDebugString(buffer);

		for(int i = 0; i < _maskBufSize[index]; i += 2)
		{
			sprintf(buffer, "%d  ", _maskBuf[index][i]);
			OutputDebugString(buffer);
			if (i % 10 == 0)
				OutputDebugString("\n");
		}
	}
*/
#endif
}

void
MaskCreator::createMask(int* maskBufSize, int** maskBuf, BYTE** map)
{
	unsigned int i, j;

	int size, tbufSize;
	int *tbuf;
	int val, oVal;

	tbufSize = 1024;
	tbuf = new int[tbufSize];
	size = 0;

	for(j = 0; j < _height; j++)
	{
		oVal = 256;

		for(i = 0; i < _width; i++)
		{
			val = (int)map[i][j];
			if (val != oVal)
			{
				size += 2;
				
				if (size > tbufSize)
				{
					int* ntbuf = new int[tbufSize * 2];
					CopyMemory(ntbuf, tbuf, sizeof(int) * tbufSize);
					tbufSize *= 2;

					delete tbuf;
					tbuf = ntbuf;
				}

				tbuf[size - 2] = _bCnt;		// count of val
				tbuf[size - 1] = val;		// val
			} else 
			{
				tbuf[size - 2] += _bCnt;	// count of val
			}

			oVal = val;
		}
	}


	*maskBufSize = size;
	*maskBuf = new int[size];
	CopyMemory(*maskBuf, tbuf, sizeof(int) * size);

	delete tbuf;
}

void
MaskCreator::getRectMap(BYTE ***map)
{
	unsigned int i, j;

	for(j = 0; j < _height; j++)
		for(i = 0; i < _width; i++)
			(*map)[i][j] = 255;
}

void
MaskCreator::getEllipMap(BYTE ***map)
{
	unsigned int i, j;
	int x, x2;
	float y;

	for(j = 0; j < _height; j++)
		for(i = 0; i < _width; i++)
			(*map)[i][j] = 0;

	for(j = 0; j < _height; j++)
	{
		y = float(j) / float(_height - 1) * 2.0f - 1.0f;
		x = int(float(_width / 2) * (1.0f - sqrt(1.0f - y * y)));
		x2 = _width - x;

		for(i = (unsigned int)x; i < (unsigned int)x2; i++)
			(*map)[i][j] = 255;
	}
}

void
MaskCreator::getCheckMap(BYTE ***map, unsigned int cgX, unsigned int cgY)
{
	unsigned int i, j;
	bool f1, f2;

	int gw = _width / cgX;
	int gh = _height / cgY;

	f1 = true;

	for(j = 0; j < _height; j++)
	{
		f2 = true;

		if (j % gh == 0)
			f1 = !f1;

		for(i = 0; i < _width; i++)
		{
			if (i % gw == 0)
				f2 = !f2;

			if ((f1 && f2) || (!f1 && !f2))
				(*map)[i][j] = 255;
			else
				(*map)[i][j] = 0;
		}

	}
}

void
MaskCreator::getBlendRectMap(BYTE ***map)
{
	const int BWIDTH = 10;

	unsigned int i, j, cnt;

	for(j = 0; j < _height; j++)
		for(i = 0; i < _width; i++)
			(*map)[i][j] = 255;

	if (BWIDTH > _width || BWIDTH > _height)
		return ;

	// upper
	for(j = 0; j < BWIDTH; j++)
		for(i = 0; i < _width; i++)
			(*map)[i][j] = BYTE(float((*map)[i][j]) * float(j) / float(BWIDTH));

	// left & right
	for(j = 0; j < _height; j++)
	{
		for(i = 0; i < BWIDTH; i++)
			(*map)[i][j] = BYTE(float((*map)[i][j]) * float(i) / float(BWIDTH));

		for(cnt = BWIDTH, i = _width - BWIDTH; i < _width; cnt--, i++)
			(*map)[i][j] = BYTE(float((*map)[i][j]) * float(cnt) / float(BWIDTH));
	}

	// bottom
	for(cnt = BWIDTH, j = _height - BWIDTH; j < _height; cnt--, j++)
		for(i = 0; i < _width; i++)
			(*map)[i][j] = BYTE(float((*map)[i][j]) * float(cnt) / float(BWIDTH));

}

void
MaskCreator::getBlendEllipMap(BYTE ***map)
{
	const int BWIDTH = 10;
	const int VAL = 255 / BWIDTH;

	unsigned int i, j, h, w;
	int x, x2;
	float y;

	for(j = 0; j < _height; j++)
		for(i = 0; i < _width; i++)
			(*map)[i][j] = 0;

	if (BWIDTH > _width || BWIDTH > _height)
		return ;

	for(j = 0; j < BWIDTH; j++)
	{
		h = _height - j * 2;
		w = _width - j * 2;

		for(i = 0; i < h; i++)
		{
			y = float(i) / float(h - 1) * 2.0f - 1.0f;
			x = int(float(w / 2) * (1.0f - sqrt(1.0f - y * y)));
			x2 = w - x;

			for(int k = x; k < x2; k++)
				(*map)[k + j][i + j] += VAL;
		}
	}

	for(j = 0; j < _height; j++)
		for(i = 0; i < _width; i++)
			if ((*map)[i][j] >= 250)
				(*map)[i][j] = 255;
}

void
MaskCreator::getBlendCheckMap(BYTE ***map, unsigned int cgX, unsigned int cgY)
{
	unsigned int i, j;
	bool f1, f2;

	BYTE** omap;
	omap = new BYTE *[_width];
	for(i = 0; i < _width; i++)
		omap[i] = new BYTE[_height];


	int gw = _width / cgX;
	int gh = _height / cgY;

	f1 = true;

	for(j = 0; j < _height; j++)
	{
		f2 = true;

		if (j % gh == 0)
			f1 = !f1;

		for(i = 0; i < _width; i++)
		{
			if (i % gw == 0)
				f2 = !f2;

			if ((f1 && f2) || (!f1 && !f2))
				(*map)[i][j] = 255;
			else
				(*map)[i][j] = 0;
		}

	}

	const int MCNT = 4;
	int total, cnt, mcnt, s;

	for(j = 0; j < _height; j++)
	{
		for(i = 0; i < _width; i++)
		{
			cnt = 0;
			total = 0;

			/// left
			s = i;
			mcnt = 0;
			while(s >= 0 && mcnt < MCNT)
			{
				total += (*map)[s][j];	cnt++;
				s--;
				mcnt++;
			}

			// center
			cnt++;
			total += (*map)[i][j];

			/// right
			s = i;
			mcnt = 0;
			while(s < (int)_width && mcnt < MCNT)
			{
				total += (*map)[s][j];	cnt++;
				s++;
				mcnt++;
			}

			omap[i][j] = (BYTE)(total / cnt);
		}

	}

	for(j = 0; j < _height; j++)
	{
		for(i = 0; i < _width; i++)
		{
			cnt = 0;
			total = 0;

			/// top
			s = j;
			mcnt = 0;
			while(s >= 0 && mcnt < MCNT)
			{
				total += omap[i][s];	cnt++;
				s--;
				mcnt++;
			}

			// center
			cnt++;
			total += omap[i][j];

			/// bottom
			s = j;
			mcnt = 0;
			while(s < (int)_height && mcnt < MCNT)
			{
				total += omap[i][s];	cnt++;
				s++;
				mcnt++;
			}

			(*map)[i][j] = (BYTE)(total / cnt);
		}

	}

	const int BWIDTH = 10;

	// upper
	for(j = 0; j < BWIDTH; j++)
		for(i = 0; i < _width; i++)
			(*map)[i][j] = BYTE(float((*map)[i][j]) * float(j) / float(BWIDTH));

	// left & right
	for(j = 0; j < _height; j++)
	{
		for(i = 0; i < BWIDTH; i++)
			(*map)[i][j] = BYTE(float((*map)[i][j]) * float(i) / float(BWIDTH));

		for(cnt = BWIDTH, i = _width - BWIDTH; i < _width; cnt--, i++)
			(*map)[i][j] = BYTE(float((*map)[i][j]) * float(cnt) / float(BWIDTH));
	}

	// bottom
	for(cnt = BWIDTH, j = _height - BWIDTH; j < _height; cnt--, j++)
		for(i = 0; i < _width; i++)
			(*map)[i][j] = BYTE(float((*map)[i][j]) * float(cnt) / float(BWIDTH));


	for(i = 0; i < _width; i++)
		delete omap[i];
	delete [] omap;
}

//////////////////////////////////////////////////////////////////////////////

	
// =================================================================
// Implements the CPointerInputPin class
// =================================================================

// Check streaming status
HRESULT
CPointerInputPin::CheckStreaming() {

	//  Shouldn't be able to get any data if we're not connected!
	ASSERT(IsConnected());

	//  we're flushing
	if(m_bFlushing) {
		return S_FALSE;
	}
	//  Don't process stuff in Stopped state
	if(IsStopped()) {
		return VFW_E_WRONG_STATE;
	}
	if(m_bRunTimeError) {
		return VFW_E_RUNTIME_ERROR;
	}
	return S_OK;
}


// constructor
CPointerInputPin::CPointerInputPin(TCHAR *pObjectName,  CCpndBase3Filter *pTransformFilter,
								   HRESULT * phr, LPCWSTR pName)
: CBaseInputPin(pObjectName, pTransformFilter, &((CVideoOverlayMixer *)pTransformFilter)->getCSFilter(), phr, pName)
{
    DbgLog((LOG_TRACE,2,TEXT("CPointerInputPin::CPointerInputPin")));
    m_pTransformFilter = (CVideoOverlayMixer *)pTransformFilter;

}

// provides derived filter a chance to grab extra interfaces
HRESULT
CPointerInputPin::CheckConnect(IPin *pPin) {
    return CBaseInputPin::CheckConnect(pPin);
}


// provides derived filter a chance to release it's extra interfaces
HRESULT
CPointerInputPin::BreakConnect() {
    ASSERT(IsStopped());
    return CBaseInputPin::BreakConnect();
}


// Let derived class know when the input pin is connected
HRESULT
CPointerInputPin::CompleteConnect(IPin *pReceivePin)
{
    return CBaseInputPin::CompleteConnect(pReceivePin);
}


// check that we can support a given media type
HRESULT
CPointerInputPin::CheckMediaType(const CMediaType* pmt)
{
 	if(*pmt->Type() == MEDIATYPE_Stream &&
	   *pmt->Subtype() == MEDIASUBTYPE_PointerInfo){
        return NOERROR;
	}else {
		return S_FALSE;
	}
}


// set the media type for this connection
HRESULT
CPointerInputPin::SetMediaType(const CMediaType* mtIn)
{
    HRESULT hr = CBasePin::SetMediaType(mtIn);
    if(FAILED(hr)) {
        return hr;
    }

	if(*mtIn->Type() == MEDIATYPE_Stream &&
	   *mtIn->Subtype() == MEDIASUBTYPE_PointerInfo)
		return S_OK;
	else
		return S_FALSE;
}


// =================================================================
// Implements IMemInputPin interface
// =================================================================


// provide EndOfStream that passes straight downstream
// (there is no queued data)
STDMETHODIMP
CPointerInputPin::EndOfStream(void)
{
    CAutoLock lck(&m_pTransformFilter->getCSReceive());
    HRESULT hr = CheckStreaming();
    return hr;
}


// enter flushing state. Call default handler to block Receives, then
// pass to overridable method in filter
STDMETHODIMP
CPointerInputPin::BeginFlush(void)
{
	CAutoLock lck(&m_pTransformFilter->getCSFilter());
    HRESULT hr = CBaseInputPin::BeginFlush();
    if(FAILED(hr)) {
        return hr;
    }

	return S_OK;
}


// leave flushing state.
// Pass to overridable method in filter, then call base class
// to unblock receives (finally)
STDMETHODIMP
CPointerInputPin::EndFlush(void)
{
    CAutoLock lck(&m_pTransformFilter->getCSFilter());
    return CBaseInputPin::EndFlush();
}


// here's the next block of data from the stream.
// AddRef it yourself if you need to hold it beyond the end
// of this call.
HRESULT
CPointerInputPin::Receive(IMediaSample * pSample)
{
    HRESULT hr;
    CAutoLock lck(&m_pTransformFilter->getCSReceive());
    ASSERT(pSample);

    hr = CBaseInputPin::Receive(pSample);
    if(S_OK == hr)
	{
        hr = ((CVideoOverlayMixer *)m_pTransformFilter)->ReceivePointer(pSample, this);
    }
    return hr;
}


// override to pass downstream
STDMETHODIMP
CPointerInputPin::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate)
{
    return CBasePin::NewSegment(tStart, tStop, dRate);
}

//////////////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2( TRUE );

} // DllRegisterServer

//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2( FALSE );

} // DllUnregisterServer

