/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/

#include <streams.h>
#include <measure.h>

#include "cpndbase2.h"

#define DEF_M_IREFSTREAM		0
#define DEF_M_DWINTERVAL		33
#define DEF_M_BTIMESTAMPZERO	false
#define DEF_M_DWTIMEOFFSET		1

// =================================================================
// Implements the CCpndBase2Filter class
// =================================================================

CCpndBase2Filter::CCpndBase2Filter(TCHAR     *pName,
                                   LPUNKNOWN pUnk,
                                   REFCLSID  clsid,
								   unsigned int inputPinCnt) :
CBaseFilter(pName,pUnk,&m_csFilter, clsid),
m_iInputPinCnt(inputPinCnt),
m_iRefStream(DEF_M_IREFSTREAM),
m_dwInterval(DEF_M_DWINTERVAL),
m_bTimeStampZero(DEF_M_BTIMESTAMPZERO),
m_dwTimeOffset(DEF_M_DWTIMEOFFSET),
m_dwPrevTime(0),
m_tStartOrg(0),
m_tStopOrg(0),
m_llMediaStart(0),
m_llMediaEnd(0),
m_pOutput(NULL),
m_bEOSDelivered(FALSE),
m_bQualityChanged(FALSE),
m_bSampleSkipped(FALSE)
{
#ifdef PERF
    RegisterPerfId();
#endif //  PERF

	ASSERT(m_iInputPinCnt != 0);
	m_ppInputs = new CCpndBase2InputPin *[m_iInputPinCnt];
	m_ppInputSamples = new IMediaSample *[m_iInputPinCnt];
	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
		m_ppInputs[i] = NULL;
		m_ppInputSamples[i] = NULL;
	}

	m_pWorker = new CCpndBase2Worker(this);
	m_pWorker->Create();
	m_pWorker->Init();
	m_pWorker->Pause();
}

#ifdef UNICODE
CCpndBase2Filter::CCpndBase2Filter(char     *pName,
                                   LPUNKNOWN pUnk,
                                   REFCLSID  clsid,
								   unsigned int inputPinCnt) :
CBaseFilter(pName,pUnk,&m_csFilter, clsid),
m_iInputPinCnt(inputPinCnt),
m_dwInterval(DEF_M_DWINTERVAL),
m_iRefStream(0),
m_bTimeStampZero(DEF_M_BTIMESTAMPZERO),
m_dwTimeOffset(DEF_M_DWPREVTIMEOFFSE),
m_dwPrevTime(0),
m_tStartOrg(0),
m_tStopOrg(0),
m_llMediaStart(0),
m_llMediaEnd(0),
m_pOutput(NULL),
m_bEOSDelivered(FALSE),
m_bQualityChanged(FALSE),
m_bSampleSkipped(FALSE)
{
#ifdef PERF
    RegisterPerfId();
#endif //  PERF

	ASSERT(m_iInputPinCnt != 0);
	m_ppInputs = new CCpndBase2InputPin *[m_iInputPinCnt];
	m_ppInputSamples = new IMediaSample *[m_iInputPinCnt];
	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
		m_ppInputs[i] = NULL;
		m_ppInputSamples[i] = NULL;
	}

	m_pWorker = new CCpndBase2Worker(this);
	m_pWorker->Create();
	m_pWorker->Init();
	m_pWorker->Pause();

}
#endif

// destructor

CCpndBase2Filter::~CCpndBase2Filter() {

	m_pWorker->Stop();
	m_pWorker->Exit();
	m_pWorker->Close();

	delete m_pWorker;

	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
		delete m_ppInputs[i];
		ASSERT(!m_ppInputSamples[i]);
	}
	delete [] m_ppInputs;
	delete [] m_ppInputSamples;

	delete m_pOutput;

}


HRESULT
CCpndBase2Filter::SetReferenceStream(int refStream)
{
	CAutoLock lock(&m_csProp);

	if (refStream != -1 && m_iInputPinCnt <= (unsigned int)refStream)
		return E_INVALIDARG;

	m_iRefStream = refStream;
	return NOERROR;
}

int
CCpndBase2Filter::GetReferenceStream()
{
	CAutoLock lock(&m_csProp);

	return m_iRefStream;
}

HRESULT
CCpndBase2Filter::SetFrameRate(int frameRate)
{
	CAutoLock lock(&m_csProp);

	if (frameRate < 1)
		frameRate = 1;
	
	if (60 < frameRate)
		frameRate = 60;

	m_dwInterval = (long)(1000 / frameRate);
	return NOERROR;
}

int
CCpndBase2Filter::GetFrameRate()
{
	CAutoLock lock(&m_csProp);
	int frameRate = (1000 / (int)m_dwInterval);
	return frameRate;
}

HRESULT
CCpndBase2Filter::SetTimeStampZero(bool flag)
{
	CAutoLock lock(&m_csProp);
	m_bTimeStampZero = flag;
	return NOERROR;
}

bool
CCpndBase2Filter::GetTimeStampZero()
{
	CAutoLock lock(&m_csProp);
	return m_bTimeStampZero;
}

HRESULT
CCpndBase2Filter::SetTimeOffset(DWORD timeOffset)
{
	CAutoLock lock(&m_csProp);
	m_dwTimeOffset = timeOffset;
	return NOERROR;
}

DWORD
CCpndBase2Filter::GetTimeOffset()
{
	CAutoLock lock(&m_csProp);
	return m_dwTimeOffset;
}

CCpndBase2InputPin*
CCpndBase2Filter::GetConnectedInputPin(int priIdx)
{
	ASSERT(m_ppInputs[0]);

	CCpndBase2InputPin* pin = NULL;

	if (priIdx != -1 && priIdx < (int)m_iInputPinCnt)
	{
		if (m_ppInputs[priIdx]->IsConnected())
			return m_ppInputs[priIdx];
	}

	for(unsigned int i = 0; i < 4; i++)
	{
		pin = m_ppInputs[i];
		if (pin->IsConnected())
			break;
	}

	if (i == 4)
		return NULL;
	else
		return pin;
}


HRESULT CCpndBase2Filter::Transform(IMediaSample **ppInputs, IMediaSample *pOut) {
    UNREFERENCED_PARAMETER(ppInputs);
    UNREFERENCED_PARAMETER(pOut);
    DbgBreak("CCpndBase2Filter::Transform() should never be called");
    return E_UNEXPECTED;
}


// return the number of pins we provide

int CCpndBase2Filter::GetPinCount() {
    return m_iInputPinCnt + 1;
}


// return a non-addrefed CBasePin * for the user to addref if he holds onto it
// for longer than his pointer to us. We create the pins dynamically when they
// are asked for rather than in the constructor. This is because we want to
// give the derived class an oppportunity to return different pin objects

// We return the objects as and when they are needed. If either of these fails
// then we return NULL, the assumption being that the caller will realise the
// whole deal is off and destroy us - which in turn will delete everything.

CBasePin *
CCpndBase2Filter::GetPin(int n) {
    HRESULT hr = S_OK;

    // Create an input pin if necessary

    if(m_ppInputs[0] == NULL) {

		for(unsigned int i = 0; i < m_iInputPinCnt; i++)
		{
			char name[64];
			WCHAR pname[20];

			sprintf(name, "Transform(N) input pin %d", i);
			swprintf(pname, L"Input %d", i);

			m_ppInputs[i] = new CCpndBase2InputPin(
						NAME(name), this, &hr, pname, i);
		}
		
        //  Can't fail
        ASSERT(SUCCEEDED(hr));
        if(m_ppInputs[0] == NULL) {
            return NULL;
        }


		m_pOutput = (CCpndBase2OutputPin *)
            new CCpndBase2OutputPin(NAME("Transform(N) output pin"),
            this,            // Owner filter
            &hr,             // Result code
            L"Output");   // Pin name
        // Can't fail
        ASSERT(SUCCEEDED(hr));
        if(m_pOutput == NULL) {
			for(unsigned int i = 0; i < m_iInputPinCnt; i++)
			{
				delete m_ppInputs[i];
				m_ppInputs[i] = NULL;
			}
        }
    }

    // Return the appropriate pin
	if (n < (int)m_iInputPinCnt)
	{
		return m_ppInputs[n];
	} else if (n == m_iInputPinCnt)
	{
		return m_pOutput;
	} else
	{
		return NULL;
	}
}


//
// FindPin
//

STDMETHODIMP CCpndBase2Filter::FindPin(LPCWSTR Id, IPin **ppPin) {
    CheckPointer(ppPin,E_POINTER);
    ValidateReadWritePtr(ppPin,sizeof(IPin *));

	*ppPin = NULL;

	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
	    WCHAR szbuf[20];
		swprintf(szbuf, L"In%d", i);

		if (!lstrcmpW(Id, szbuf))
		{
			*ppPin = GetPin(i);
			break;
		}
	}

	if (!(*ppPin))
	{
		if (!lstrcmpW(Id, L"Out"))
		{
			*ppPin = GetPin(m_iInputPinCnt + 0);
		} else 
		{
			return VFW_E_NOT_FOUND;
		}
	}

    HRESULT hr = NOERROR;
    if(*ppPin) {
        (*ppPin)->AddRef();
    }
    else {
        hr = E_OUTOFMEMORY;
    }
    return hr;
}


// override these two functions if you want to inform something
// about entry to or exit from streaming state.

HRESULT
CCpndBase2Filter::StartStreaming() {
    return NOERROR;
}


HRESULT
CCpndBase2Filter::StopStreaming() {
    return NOERROR;
}


// override this to grab extra interfaces on connection

HRESULT
CCpndBase2Filter::CheckConnect(PIN_DIRECTION dir,IPin *pPin, unsigned int pinIdx) {
    UNREFERENCED_PARAMETER(dir);
    UNREFERENCED_PARAMETER(pPin);
	UNREFERENCED_PARAMETER(pinIdx);
    return NOERROR;
}


// place holder to allow derived classes to release any extra interfaces

HRESULT
CCpndBase2Filter::BreakConnect(PIN_DIRECTION dir, unsigned int pinIdx) {
    UNREFERENCED_PARAMETER(dir);
	UNREFERENCED_PARAMETER(pinIdx);
    return NOERROR;
}


// Let derived classes know about connection completion

HRESULT
CCpndBase2Filter::CompleteConnect(PIN_DIRECTION direction,IPin *pReceivePin, unsigned int pinIdx) {
    UNREFERENCED_PARAMETER(direction);
    UNREFERENCED_PARAMETER(pReceivePin);
	UNREFERENCED_PARAMETER(pinIdx);
    return NOERROR;
}


// override this to know when the media type is really set

HRESULT
CCpndBase2Filter::SetMediaType(PIN_DIRECTION direction, const CMediaType *pmt, unsigned int pinIdx) {
    UNREFERENCED_PARAMETER(direction);
    UNREFERENCED_PARAMETER(pmt);
	UNREFERENCED_PARAMETER(pinIdx);
    return NOERROR;
}

HRESULT
CCpndBase2Filter::SetupTimeStamp()
{
	CAutoLock lck(&m_csProp);

	if (m_bTimeStampZero)
	{
		m_tStartOrg = Int32x32To64(UNITS / MILLISECONDS, 0);
		m_tStopOrg = Int32x32To64(UNITS / MILLISECONDS, 0);

		if (m_dwPrevTime == 0)
		{
			m_llMediaStart = 1;
			m_llMediaEnd = 2;
		} else 
		{
			m_llMediaStart++;
			m_llMediaEnd++;
		}

		m_dwPrevTime += m_dwInterval;

	} else
	{
		if (m_dwPrevTime == 0)
		{
			m_tStartOrg = Int32x32To64(UNITS / MILLISECONDS, 0);
			m_llMediaStart = 1;
			m_llMediaEnd = 2;
		} else
		{
			m_tStartOrg = Int32x32To64(UNITS / MILLISECONDS, m_dwPrevTime + 1);
			m_llMediaStart++;
			m_llMediaEnd++;
		}

		m_dwPrevTime += m_dwInterval;
		m_tStopOrg = Int32x32To64(UNITS / MILLISECONDS, m_dwPrevTime + m_dwTimeOffset);
	}

	return NOERROR;
}

// Set up our output sample
HRESULT
CCpndBase2Filter::InitializeOutputSample(IMediaSample **ppSample, IMediaSample **ppOutSample)
{
	CAutoLock lock(&m_csProp);

	IMediaSample *pOutSample;

	if (m_iRefStream != -1 && ppSample[m_iRefStream])
	{
		AM_SAMPLE2_PROPERTIES * const pProps = m_ppInputs[m_iRefStream]->SampleProps();

		DWORD dwFlags = m_bSampleSkipped ? AM_GBF_PREVFRAMESKIPPED : 0;

		// This will prevent the image renderer from switching us to DirectDraw
		// when we can't do it without skipping frames because we're not on a
		// keyframe.  If it really has to switch us, it still will, but then we
		// will have to wait for the next keyframe
		if (!(pProps->dwSampleFlags & AM_SAMPLE_SPLICEPOINT))
		{
			dwFlags |= AM_GBF_NOTASYNCPOINT;
		}

		ASSERT(m_pOutput->m_pAllocator != NULL);
		HRESULT hr = m_pOutput->m_pAllocator->GetBuffer(&pOutSample,
			pProps->dwSampleFlags & AM_SAMPLE_TIMEVALID ? &pProps->tStart : NULL,
			pProps->dwSampleFlags & AM_SAMPLE_STOPVALID ? &pProps->tStop : NULL,
			dwFlags);
		*ppOutSample = pOutSample;

		if(FAILED(hr))
		{
			return hr;
		}

		ASSERT(pOutSample);


		IMediaSample2 *pOutSample2;
		if(SUCCEEDED(pOutSample->QueryInterface(IID_IMediaSample2, (void **)&pOutSample2)))
		{
			AM_SAMPLE2_PROPERTIES OutProps;

			EXECUTE_ASSERT(SUCCEEDED(pOutSample2->GetProperties(
				FIELD_OFFSET(AM_SAMPLE2_PROPERTIES, tStart), (PBYTE)&OutProps)));

			OutProps.dwTypeSpecificFlags = pProps->dwTypeSpecificFlags;
			OutProps.dwSampleFlags =
				(OutProps.dwSampleFlags & AM_SAMPLE_TYPECHANGED) |
				(pProps->dwSampleFlags & ~AM_SAMPLE_TYPECHANGED);

			OutProps.tStart = pProps->tStart;
			OutProps.tStop  = pProps->tStop;
			OutProps.cbData = FIELD_OFFSET(AM_SAMPLE2_PROPERTIES, dwStreamId);

			hr = pOutSample2->SetProperties(
				FIELD_OFFSET(AM_SAMPLE2_PROPERTIES, dwStreamId), (PBYTE)&OutProps);

			if(pProps->dwSampleFlags & AM_SAMPLE_DATADISCONTINUITY)
			{
				m_bSampleSkipped = FALSE;
			}
			pOutSample2->Release();
		}
		else
		{
			if (pProps->dwSampleFlags & AM_SAMPLE_TIMEVALID)
			{
				pOutSample->SetTime(&pProps->tStart, &pProps->tStop);
			}
			if (pProps->dwSampleFlags & AM_SAMPLE_SPLICEPOINT)
			{
				pOutSample->SetSyncPoint(TRUE);
			}
			if (pProps->dwSampleFlags & AM_SAMPLE_DATADISCONTINUITY)
			{
				pOutSample->SetDiscontinuity(TRUE);
				m_bSampleSkipped = FALSE;
			}

			// Copy the media times
			LONGLONG MediaStart, MediaEnd;
		    if (ppSample[m_iRefStream]->GetMediaTime(&MediaStart, &MediaEnd) == NOERROR)
			{
				pOutSample->SetMediaTime(&MediaStart, &MediaEnd);
			}
		}

	}
	else
	{
		DWORD dwFlags;

		dwFlags = m_bSampleSkipped ? AM_GBF_PREVFRAMESKIPPED : 0;
		dwFlags |= AM_GBF_NOTASYNCPOINT;

		// Get Buffer
		ASSERT(m_pOutput->m_pAllocator);
		HRESULT hr = m_pOutput->m_pAllocator->GetBuffer(
			&pOutSample, &m_tStartOrg, &m_tStopOrg, dwFlags);
		*ppOutSample = pOutSample;

		if(FAILED(hr))
		{
			return hr;
		}

		// Set Properties
		ASSERT(pOutSample);
		IMediaSample2 *pOutSample2;
		if(SUCCEEDED(pOutSample->QueryInterface(IID_IMediaSample2, (void **)&pOutSample2)))
		{
			AM_SAMPLE2_PROPERTIES OutProps;

			EXECUTE_ASSERT(SUCCEEDED(pOutSample2->GetProperties(
				FIELD_OFFSET(AM_SAMPLE2_PROPERTIES, tStart), (PBYTE)&OutProps)));
			OutProps.dwTypeSpecificFlags = 0;
			OutProps.dwSampleFlags = 
				(OutProps.dwSampleFlags & AM_SAMPLE_TYPECHANGED) |
				AM_SAMPLE_SPLICEPOINT |
				(m_bTimeStampZero ? 0 : AM_SAMPLE_TIMEVALID) |
				(m_bTimeStampZero ? 0 : AM_SAMPLE_STOPVALID) |
				//(m_dwPrevTime != m_dwInterval) ? 0: AM_SAMPLE_DATADISCONTINUITY |
				AM_STREAM_MEDIA;

			OutProps.tStart = m_tStartOrg;
			OutProps.tStop  = m_tStopOrg;
			OutProps.cbData = FIELD_OFFSET(AM_SAMPLE2_PROPERTIES, dwStreamId);

			hr = pOutSample2->SetProperties(
				FIELD_OFFSET(AM_SAMPLE2_PROPERTIES, dwStreamId), (PBYTE)&OutProps);

			m_bSampleSkipped = FALSE;

			pOutSample2->Release();
		}
		else
		{
			pOutSample->SetTime(&m_tStartOrg, &m_tStopOrg);
			pOutSample->SetSyncPoint(TRUE);
			pOutSample->SetPreroll(FALSE);
			pOutSample->SetDiscontinuity(FALSE);

			m_bSampleSkipped = FALSE;

			pOutSample->SetMediaTime(NULL, NULL);
		}
	}

    return S_OK;
}

// override this to customize the transform process

HRESULT
CCpndBase2Filter::Receive(IMediaSample **ppSamples)
{
	ASSERT(m_ppInputs[0]);

    /*  Check for other streams and pass them on */
	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
		if (!ppSamples[i])
			continue;

	    AM_SAMPLE2_PROPERTIES * const pProps = m_ppInputs[i]->SampleProps();
		if (pProps->dwStreamId != AM_STREAM_MEDIA)
		{
			ppSamples[i]->Release();
			ppSamples[i] = NULL;
		}
	}

    HRESULT hr;
    IMediaSample * pOutSample;

    // If no output to deliver to then no point sending us data
    ASSERT(m_pOutput) ;

	// Set up Time stamp
	hr = SetupTimeStamp();
	if (FAILED(hr))
	{
		return hr;
	}

    // Set up the output sample
    hr = InitializeOutputSample(ppSamples, &pOutSample);
    if (FAILED(hr))
	{
        return hr;
    }

    // Start timing the transform (if PERF is defined)
    MSR_START(m_idTransform);

    // have the derived class transform the data

    hr = Transform(ppSamples, pOutSample);

    // Stop the clock and log it (if PERF is defined)
    MSR_STOP(m_idTransform);

    if (FAILED(hr))
	{
        DbgLog((LOG_TRACE,1,TEXT("Error from transform")));
    }
    else
	{
        // the Transform() function can return S_FALSE to indicate that the
        // sample should not be delivered; we only deliver the sample if it's
        // really S_OK (same as NOERROR, of course.)
        if (hr == NOERROR)
		{
            hr = m_pOutput->m_pInputPin->Receive(pOutSample);
            m_bSampleSkipped = FALSE;   // last thing no longer dropped
        }
        else
		{
            // S_FALSE returned from Transform is a PRIVATE agreement
            // We should return NOERROR from Receive() in this cause because returning S_FALSE
            // from Receive() means that this is the end of the stream and no more data should
            // be sent.
            if (S_FALSE == hr)
			{

                //  Release the sample before calling notify to avoid
                //  deadlocks if the sample holds a lock on the system
                //  such as DirectDraw buffers do
                pOutSample->Release();
                m_bSampleSkipped = TRUE;
                if (!m_bQualityChanged)
				{
                    NotifyEvent(EC_QUALITY_CHANGE,0,0);
                    m_bQualityChanged = TRUE;
                }
                return NOERROR;
            }
        }
    }

    // release the output buffer. If the connected pin still needs it,
    // it will have addrefed it itself.
    pOutSample->Release();

    return hr;
}


// Return S_FALSE to mean "pass the note on upstream"
// Return NOERROR (Same as S_OK)
// to mean "I've done something about it, don't pass it on"
HRESULT CCpndBase2Filter::AlterQuality(Quality q)
{
    UNREFERENCED_PARAMETER(q);
    return S_FALSE;
}


// EndOfStream received. Default behaviour is to deliver straight
// downstream, since we have no queued data. If you overrode Receive
// and have queue data, then you need to handle this and deliver EOS after
// all queued data is sent
HRESULT
CCpndBase2Filter::EndOfStream(unsigned int pinIdx)
{
	unsigned int i;
	for(i = 0; i < m_iInputPinCnt; i++)
	{
		if (pinIdx != i &&
			m_ppInputs[i]->IsConnected() &&
			!m_ppInputs[i]->IsEndOfStream())
			break;
	}

	HRESULT hr = NOERROR;
	if (i == m_iInputPinCnt && m_pOutput != NULL)
	{
#ifdef DEBUG
		OutputDebugString("  Deliver End Of Stream.\n");
#endif
        hr = m_pOutput->DeliverEndOfStream();
    }

    return hr;
}


// enter flush state. Receives already blocked
// must override this if you have queued data or a worker thread
HRESULT
CCpndBase2Filter::BeginFlush(unsigned int pinIdx)
{
    HRESULT hr = NOERROR;
	if(m_pOutput != NULL)
	{
        // block receives -- done by caller (CBaseInputPin::BeginFlush)

        // discard queued data -- we have no queued data

        // free anyone blocked on receive - not possible in this filter

        // call downstream
#ifdef DEBUG
		OutputDebugString("  Deliver Begin Flush.\n");
#endif
        hr = m_pOutput->DeliverBeginFlush();
    }
    return hr;
}


// leave flush state. must override this if you have queued data
// or a worker thread
HRESULT
CCpndBase2Filter::EndFlush(unsigned int pinIdx)
{
	// sync with pushing thread -- we have no worker thread
	//CAutoLock lock2(&m_pWorker->m_WorkerLock);

    // ensure no more data to go downstream -- we have no queued data

    // call EndFlush on downstream pins
    ASSERT(m_pOutput != NULL);

#ifdef DEBUG
	OutputDebugString("  Deliver End Flush.\n");
#endif
	
    return m_pOutput->DeliverEndFlush();

    // caller (the input pin's method) will unblock Receives
}


// override these so that the derived filter can catch them

STDMETHODIMP
CCpndBase2Filter::Stop()
{
	{
#ifdef DEBUG
		OutputDebugString("Stop()\n");
#endif
		CAutoLock lck(&m_pWorker->m_WorkerLock);
		m_dwPrevTime = 0;
	}

    CAutoLock lck1(&m_csFilter);
    if(m_State == State_Stopped)
	{
        return NOERROR;
    }

    // Succeed the Stop if we are not completely connected
    ASSERT(m_ppInputs[0] == NULL || m_pOutput != NULL);

    if(m_ppInputs[0] == NULL || InputIsAllDisconnected() ||
        m_pOutput->IsConnected() == FALSE)
	{
        m_State = State_Stopped;
        m_bEOSDelivered = FALSE;
        return NOERROR;
    }

    ASSERT(m_ppInputs[0]);
    ASSERT(m_pOutput);

    // decommit the input pin before locking or we can deadlock
	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	    m_ppInputs[i]->Inactive();

    // synchronize with Receive calls
    CAutoLock lck2(&m_csReceive);
    m_pOutput->Inactive();


    // allow a class derived from CCpndBase2Filter
    // to know about starting and stopping streaming
    HRESULT hr = StopStreaming();
    if(SUCCEEDED(hr)) {
        // complete the state transition
        m_State = State_Stopped;
        m_bEOSDelivered = FALSE;
    }
    return hr;
}

STDMETHODIMP
CCpndBase2Filter::Pause()
{

    CAutoLock lck(&m_csFilter);
    HRESULT hr = NOERROR;

    if (m_State == State_Paused)
	{
        // (This space left deliberately blank)
    }

    // If we have no input pin or it isn't yet connected then when we are
    // asked to pause we deliver an end of stream to the downstream filter.
    // This makes sure that it doesn't sit there forever waiting for
    // samples which we cannot ever deliver without an input connection.

    else if(m_ppInputs[0] == NULL || InputIsAllDisconnected())
	{
        if(m_pOutput && m_bEOSDelivered == FALSE)
		{
            m_pOutput->DeliverEndOfStream();
            m_bEOSDelivered = TRUE;
        }
        m_State = State_Paused;
    }

    // We may have an input connection but no output connection
    // However, if we have an input pin we do have an output pin

    else if(m_pOutput->IsConnected() == FALSE)
	{
        m_State = State_Paused;
    }
    else
	{
        if(m_State == State_Stopped)
		{
            // allow a class derived from CCpndBase2Filter
            // to know about starting and stopping streaming
            CAutoLock lck2(&m_csReceive);
            hr = StartStreaming();
        }
        if(SUCCEEDED(hr))
		{
            hr = CBaseFilter::Pause();
        }
    }

    m_bSampleSkipped = FALSE;
    m_bQualityChanged = FALSE;
    return hr;
}

HRESULT
CCpndBase2Filter::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate, unsigned int pinIdx)
{
	CAutoLock lck(&m_csProp);

    if (m_pOutput != NULL && pinIdx == m_iRefStream)
	{
#ifdef DEBUG
		char buffer[256];
		sprintf(buffer, "  Deliver NewSegment.  MediaStart:%d  MediaEnd:%d\n",
			(int)(tStart / (UNITS / MILLISECONDS)), (int)(tStop / (UNITS / MILLISECONDS)));
		OutputDebugString(buffer);
#endif

        return m_pOutput->DeliverNewSegment(tStart, tStop, dRate);
    }
    return S_OK;
}


BOOL
CCpndBase2Filter::InputIsAllDisconnected()
{
	for(unsigned int i = 0; i < m_iInputPinCnt; i++)
	{
		if (m_ppInputs[i]->IsConnected() == TRUE)
			return false;
	}
	return true;
}

HRESULT
CCpndBase2Filter::ReceiveSample(IMediaSample *pSample, unsigned int pinIdx)
{
	CAutoLock lock2(&m_pWorker->m_WorkerLock);

	if (!m_ppInputSamples[pinIdx])
	{
		m_ppInputSamples[pinIdx] = pSample;
		m_ppInputSamples[pinIdx]->AddRef();
	}

	return NOERROR;
}


///////////////////////////////////////////////////////////////

CCpndBase2Worker::CCpndBase2Worker(CCpndBase2Filter* owner) :
_owner(owner),
_oldRecTime(0)
{
}

CCpndBase2Worker::~CCpndBase2Worker()
{
}

DWORD CCpndBase2Worker::ThreadProc(void)
{
    Command com;

    do {
        com = (Command)GetRequest();
        if (com != CMD_INIT)
		{
            Reply((DWORD)E_UNEXPECTED);
        }
    } while (com != CMD_INIT);

#ifdef DEBUG
	OutputDebugString("CCpndBase2Worker Start.\n");
#endif

    // Initialisation suceeded
    Reply(NOERROR);

    Command cmd;
    do {
        cmd = (Command)GetRequest();

        switch (cmd) {

        case CMD_EXIT:
            Reply(NOERROR);
            break;

        case CMD_RUN:
        case CMD_PAUSE:
            Reply(NOERROR);
            DoBufferProcessingLoop();
            break;

        case CMD_STOP:
            Reply(NOERROR);
			ReleaseSamples();
            break;

        default:
            Reply((DWORD)E_NOTIMPL);
            break;
        }
    } while (cmd != CMD_EXIT);

#ifdef DEBUG
	OutputDebugString("CCpndBase2Worker End.\n");
#endif
    return 0;
}

HRESULT
CCpndBase2Worker::DoBufferProcessingLoop(void)
{
	Command com;

	do {
        while (!CheckRequest((DWORD*)&com)) {

			{
				CAutoLock lock2(&m_WorkerLock);

				if (_owner->IsActive())
				{
					DWORD recTime = GetTickCount();

					if (CanReceive(recTime))
					{
						_owner->Receive(_owner->m_ppInputSamples);
						ReleaseSamples();
						RegistOldTime(recTime);
					}
				}
				else 
				{
					ReleaseSamples();
					Sleep(50);
				}

			}

			Sleep(1);
        }

        if (com == CMD_RUN || com == CMD_PAUSE)
		{
            Reply(NOERROR);
        }
		else if (com != CMD_STOP)
		{
            Reply((DWORD)E_UNEXPECTED);
        }
    } while (com != CMD_STOP);

    return S_FALSE;
}

HRESULT
CCpndBase2Worker::ReleaseSamples(void)
{
	for(unsigned int i = 0; i < _owner->m_iInputPinCnt; i++)
	{
		if (_owner->m_ppInputSamples[i])
		{
			_owner->m_ppInputSamples[i]->Release();
			_owner->m_ppInputSamples[i] = NULL;
		}
	}
	return NOERROR;
}

BOOL
CCpndBase2Worker::CanReceive(DWORD recTime)
{
	CAutoLock lck(&_owner->m_csProp);

	unsigned int i;
	for(i = 0; i < _owner->m_iInputPinCnt; i++)
		if (_owner->m_ppInputSamples[i])
			break;
	if (i == _owner->m_iInputPinCnt)
		return FALSE;

	if (_oldRecTime == 0 || recTime - _oldRecTime >= _owner->m_dwInterval)
		return TRUE;
	else
		return FALSE;
}

void
CCpndBase2Worker::RegistOldTime(DWORD recTime)
{
	CAutoLock lck(&_owner->m_csProp);

#if 1
	_oldRecTime = recTime;
#else
	if (_oldRecTime == 0)
		_oldRecTime = recTime;
	else
		_oldRecTime += _owner->m_dwInterval;
#endif
}


///////////////////////////////////////////////////////////////


// Check streaming status
HRESULT
CCpndBase2InputPin::CheckStreaming() {
    ASSERT(m_pTransformFilter->m_pOutput != NULL);
    if(!m_pTransformFilter->m_pOutput->IsConnected()) {
        return VFW_E_NOT_CONNECTED;
    }
    else {
        //  Shouldn't be able to get any data if we're not connected!
        ASSERT(IsConnected());

        //  we're flushing
        if(m_bFlushing) {
            return S_FALSE;
        }
        //  Don't process stuff in Stopped state
        if(IsStopped()) {
            return VFW_E_WRONG_STATE;
        }
        if(m_bRunTimeError) {
            return VFW_E_RUNTIME_ERROR;
        }
        return S_OK;
    }
}


// =================================================================
// Implements the CCpndBase2InputPin class
// =================================================================


// constructor

CCpndBase2InputPin::CCpndBase2InputPin(TCHAR *pObjectName, CCpndBase2Filter *pTransformFilter,
									HRESULT * phr, LPCWSTR pName, unsigned int pinIdx) :
CBaseInputPin(pObjectName, pTransformFilter, &pTransformFilter->m_csFilter, phr, pName)
{
	DbgLog((LOG_TRACE,2,TEXT("CCpndBase2InputPin::CCpndBase2InputPin")));
	m_pTransformFilter = pTransformFilter;
	m_bPinIdx = pinIdx;
	m_bEOS = FALSE;
}

#ifdef UNICODE
CCpndBase2InputPin::CCpndBase2InputPin(CHAR *pObjectName, CCpndBase2Filter *pTransformFilter,
									 HRESULT * phr, LPCWSTR pName, unsigned int pinIdx) :
CBaseInputPin(pObjectName, pTransformFilter, &pTransformFilter->m_csFilter, phr, pName)
{
    DbgLog((LOG_TRACE,2,TEXT("CCpndBase2InputPin::CCpndBase2InputPin")));
    m_pTransformFilter = pTransformFilter;
	m_bPinIdx = pinIdx;
	m_bEOS = FALSE;
}
#endif


BOOL
CCpndBase2InputPin::IsEndOfStream()
{
	return m_bEOS;
}

// provides derived filter a chance to grab extra interfaces
HRESULT
CCpndBase2InputPin::CheckConnect(IPin *pPin)
{
    HRESULT hr = m_pTransformFilter->CheckConnect(PINDIR_INPUT, pPin, m_bPinIdx);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseInputPin::CheckConnect(pPin);
}


// provides derived filter a chance to release it's extra interfaces
HRESULT
CCpndBase2InputPin::BreakConnect()
{
    //  Can't disconnect unless stopped
    ASSERT(IsStopped());
    m_pTransformFilter->BreakConnect(PINDIR_INPUT, m_bPinIdx);
    return CBaseInputPin::BreakConnect();
}


// Let derived class know when the input pin is connected
HRESULT
CCpndBase2InputPin::CompleteConnect(IPin *pReceivePin)
{
    HRESULT hr = m_pTransformFilter->CompleteConnect(PINDIR_INPUT, pReceivePin, m_bPinIdx);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseInputPin::CompleteConnect(pReceivePin);
}


// check that we can support a given media type
HRESULT
CCpndBase2InputPin::CheckMediaType(const CMediaType* pmt)
{
    HRESULT hr = m_pTransformFilter->CheckInputType(pmt, m_bPinIdx);
	return hr;
}


// set the media type for this connection
HRESULT
CCpndBase2InputPin::SetMediaType(const CMediaType* mtIn)
{
    // Set the base class media type (should always succeed)
    HRESULT hr = CBasePin::SetMediaType(mtIn);
    if(FAILED(hr)) {
        return hr;
    }

    // check the transform can be done (should always succeed)
    ASSERT(SUCCEEDED(m_pTransformFilter->CheckInputType(mtIn, m_bPinIdx)));

    return m_pTransformFilter->SetMediaType(PINDIR_INPUT, mtIn, m_bPinIdx);
}


HRESULT
CCpndBase2InputPin::Active()
{
	{
	    CAutoLock lck(&m_pTransformFilter->m_csFilter);
		m_bEOS = FALSE;
	}
	return CBaseInputPin::Active();
}

// =================================================================
// Implements IMemInputPin interface
// =================================================================


// here's the next block of data from the stream.
// AddRef it yourself if you need to hold it beyond the end
// of this call.
HRESULT
CCpndBase2InputPin::Receive(IMediaSample * pSample)
{
    HRESULT hr;
    CAutoLock lck(&m_pTransformFilter->m_csReceive);
    ASSERT(pSample);

    // check all is well with the base class
	hr = CBaseInputPin::Receive(pSample);
    if(S_OK == hr)
		hr = m_pTransformFilter->ReceiveSample(pSample, m_bPinIdx);
	
    return hr;
}

STDMETHODIMP
CCpndBase2InputPin::ReceiveCanBlock()
{
	return S_OK;
}


// provide EndOfStream that passes straight downstream
// (there is no queued data)
STDMETHODIMP
CCpndBase2InputPin::EndOfStream(void)
{
	CAutoLock lck(&m_pTransformFilter->m_csReceive);

	{
	    CAutoLock lck(&m_pTransformFilter->m_csFilter);
		m_bEOS = TRUE;
	}

    HRESULT hr = CheckStreaming();
	if(S_OK == hr)
	{
		hr = m_pTransformFilter->EndOfStream(m_bPinIdx);
	}
	return hr;
}


// override to pass downstream
STDMETHODIMP
CCpndBase2InputPin::NewSegment(REFERENCE_TIME tStart, REFERENCE_TIME tStop, double dRate)
{
	//  Save the values in the pin
    CBasePin::NewSegment(tStart, tStop, dRate);
	return m_pTransformFilter->NewSegment(tStart, tStop, dRate, m_bPinIdx);
}

// enter flushing state. Call default handler to block Receives, then
// pass to overridable method in filter
STDMETHODIMP
CCpndBase2InputPin::BeginFlush(void)
{
	CAutoLock lck(&m_pTransformFilter->m_csFilter);
    //  Are we actually doing anything?
    ASSERT(m_pTransformFilter->m_pOutput != NULL);

	if (!IsConnected() || !m_pTransformFilter->m_pOutput->IsConnected())
	{
        return VFW_E_NOT_CONNECTED;
    }
    HRESULT hr = CBaseInputPin::BeginFlush();
    if(FAILED(hr)) {
        return hr;
    }

    return m_pTransformFilter->BeginFlush(m_bPinIdx);
}


// leave flushing state.
// Pass to overridable method in filter, then call base class
// to unblock receives (finally)
STDMETHODIMP
CCpndBase2InputPin::EndFlush(void)
{
	CAutoLock lck(&m_pTransformFilter->m_csFilter);
    //  Are we actually doing anything?
    ASSERT(m_pTransformFilter->m_pOutput != NULL);
    if(!IsConnected() || !m_pTransformFilter->m_pOutput->IsConnected())
	{
        return VFW_E_NOT_CONNECTED;
    }

	HRESULT hr = m_pTransformFilter->EndFlush(m_bPinIdx);
	if(FAILED(hr)) {
		return hr;
	}

	return CBaseInputPin::EndFlush();
}





// =================================================================
// Implements the CCpndBase2OutputPin class
// =================================================================


// constructor

CCpndBase2OutputPin::CCpndBase2OutputPin(TCHAR *pObjectName, CCpndBase2Filter *pTransformFilter,
									   HRESULT * phr, LPCWSTR pPinName) :
CBaseOutputPin(pObjectName, pTransformFilter, &pTransformFilter->m_csFilter, phr, pPinName),
m_pPosition(NULL)
{
    DbgLog((LOG_TRACE,2,TEXT("CCpndBase2OutputPin::CCpndBase2OutputPin")));
    m_pTransformFilter = pTransformFilter;
}

#ifdef UNICODE
CCpndBase2OutputPin::CCpndBase2OutputPin(CHAR *pObjectName, CCpndBase2Filter *pTransformFilter,
									   HRESULT * phr, LPCWSTR pPinName) :
CBaseOutputPin(pObjectName, pTransformFilter, &pTransformFilter->m_csFilter, phr, pPinName),
m_pPosition(NULL)
{
    DbgLog((LOG_TRACE,2,TEXT("CCpndBase2OutputPin::CCpndBase2OutputPin")));
    m_pTransformFilter = pTransformFilter;
}
#endif

// destructor

CCpndBase2OutputPin::~CCpndBase2OutputPin()
{
    DbgLog((LOG_TRACE,2,TEXT("CCpndBase2OutputPin::~CCpndBase2OutputPin")));
    if(m_pPosition) m_pPosition->Release();
}


// overriden to expose IMediaPosition and IMediaSeeking control interfaces

STDMETHODIMP
CCpndBase2OutputPin::NonDelegatingQueryInterface(REFIID riid, void **ppv)
{
    CheckPointer(ppv,E_POINTER);
    ValidateReadWritePtr(ppv,sizeof(PVOID));
    *ppv = NULL;

    if(riid == IID_IMediaPosition || riid == IID_IMediaSeeking) {

        // we should have an input pin by now
        ASSERT(m_pTransformFilter->m_ppInputs[0] != NULL);

        if(m_pPosition == NULL) {

            HRESULT hr = CreatePosPassThru(GetOwner(),
                FALSE,
                (IPin *)m_pTransformFilter->m_ppInputs[0],
                &m_pPosition);
            if(FAILED(hr)) {
                return hr;
            }
        }
        return m_pPosition->QueryInterface(riid, ppv);
    }
    else {
        return CBaseOutputPin::NonDelegatingQueryInterface(riid, ppv);
    }
}


// provides derived filter a chance to grab extra interfaces
HRESULT
CCpndBase2OutputPin::CheckConnect(IPin *pPin)
{
    // we should have an input connection first

    ASSERT(m_pTransformFilter->m_ppInputs[0] != NULL);
	if (m_pTransformFilter->InputIsAllDisconnected())
	{
        return E_UNEXPECTED;
    }

    HRESULT hr = m_pTransformFilter->CheckConnect(PINDIR_OUTPUT, pPin, 0);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseOutputPin::CheckConnect(pPin);
}


// provides derived filter a chance to release it's extra interfaces
HRESULT
CCpndBase2OutputPin::BreakConnect()
{
    //  Can't disconnect unless stopped
    ASSERT(IsStopped());
    m_pTransformFilter->BreakConnect(PINDIR_OUTPUT, 0);
    return CBaseOutputPin::BreakConnect();
}


// Let derived class know when the output pin is connected

HRESULT
CCpndBase2OutputPin::CompleteConnect(IPin *pReceivePin)
{
    HRESULT hr = m_pTransformFilter->CompleteConnect(PINDIR_OUTPUT, pReceivePin, 0);
    if(FAILED(hr)) {
        return hr;
    }
    return CBaseOutputPin::CompleteConnect(pReceivePin);
}


// check a given transform - must have selected input type first

HRESULT
CCpndBase2OutputPin::CheckMediaType(const CMediaType* pmtOut)
{
    HRESULT hr = m_pTransformFilter->CheckOutputType(pmtOut);
	return hr;
}


// called after we have agreed a media type to actually set it in which case
// we run the CheckTransform function to get the output format type again

HRESULT
CCpndBase2OutputPin::SetMediaType(const CMediaType* pmtOut)
{
	HRESULT hr;
    hr = CBasePin::SetMediaType(pmtOut);
    if(FAILED(hr)) {
        return hr;
    }
    return m_pTransformFilter->SetMediaType(PINDIR_OUTPUT, pmtOut, 0);
}


// pass the buffer size decision through to the main transform class

HRESULT
CCpndBase2OutputPin::DecideBufferSize(IMemAllocator * pAllocator, ALLOCATOR_PROPERTIES* pProp)
{
    return m_pTransformFilter->DecideBufferSize(pAllocator, pProp);
}



// return a specific media type indexed by iPosition

HRESULT
CCpndBase2OutputPin::GetMediaType(int iPosition, CMediaType *pMediaType)
{
	return m_pTransformFilter->GetMediaType(iPosition, pMediaType);
}


// Override this if you can do something constructive to act on the
// quality message.  Consider passing it upstream as well

// Pass the quality mesage on upstream.
STDMETHODIMP
CCpndBase2OutputPin::Notify(IBaseFilter * pSender, Quality q)
{
    UNREFERENCED_PARAMETER(pSender);
    ValidateReadPtr(pSender,sizeof(IBaseFilter));

    // First see if we want to handle this ourselves
    HRESULT hr = m_pTransformFilter->AlterQuality(q);
    if(hr!=S_FALSE) {
        return hr;        // either S_OK or a failure
    }

    // S_FALSE means we pass the message on.
    // Find the quality sink for our input pin and send it there

    ASSERT(m_pTransformFilter->m_ppInputs[0] != NULL);

	for(unsigned int i = 0; i < m_pTransformFilter->m_iInputPinCnt; i++)
	{
		if (m_pTransformFilter->m_ppInputs[i]->PassNotify(q) == S_OK)
			return S_OK;
	}
    return VFW_E_NOT_FOUND;

} // Notify


// the following removes a very large number of level 4 warnings from the microsoft
// compiler output, which are not useful at all in this case.
#pragma warning(disable:4514)

