/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/

#include <windows.h>
#include <streams.h>
#include <initguid.h>
#include <olectl.h>
#if (1100 > _MSC_VER)
#include <olectlid.h>
#endif

#include <stdio.h>
#include <limits.h>
#include <math.h>

#include <flTypes/flUIDs.h>
#include <flTypes/IVideoQuadSplitter.h>

#include "cpndbase2.h"

#include "videoquadsplitterprop.h"
#include "videoquadsplitter.h"

#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue

// setup data

const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
    &MEDIATYPE_Video,			// Major type
    &MEDIASUBTYPE_NULL			// Minor type
};

const AMOVIESETUP_PIN psudPins[] =
{
	{
		L"Input1",          // String pin name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Output",          // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Input2",          // String pin name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Output",          // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Input3",          // String pin name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Output",          // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Input4",          // String pin name
		FALSE,              // Is it rendered
		FALSE,              // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Output",          // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
	{
		L"Output",          // String pin name
		FALSE,              // Is it rendered
		TRUE,               // Is it an output
		FALSE,              // Allowed none
		FALSE,              // Allowed many
		&CLSID_NULL,        // Connects to filter
		L"Input",           // Connects to pin
		1,                  // Number of types
		&sudPinTypes        // The pin details
	},
};


const AMOVIESETUP_FILTER sudVideoQuadSplitter =
{
    &CLSID_VideoQuadSplitter,   // Filter CLSID
    L"NIME Video Quad Splitter", // Filter name
    MERIT_DO_NOT_USE,       // Its merit
    5,                      // Number of pins
    psudPins                // Pin details
};


// List of class IDs and creator functions for the class factory. This
// provides the link between the OLE entry point in the DLL and an object
// being created. The class factory will call the static CreateInstance

CFactoryTemplate g_Templates[2] = {

    { L"VideoQuadSplitter"
    , &CLSID_VideoQuadSplitter
    , CVideoQuadSplitter::CreateInstance
    , NULL
    , &sudVideoQuadSplitter }
  ,
    { L"VideoQuadSplitterProp"
    , &CLSID_VideoQuadSplitterPropertyPage
    , CVideoQuadSplitterProperties::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);



#define DEF_TIMESTAMP			TS_IN0
#define DEF_BGCOLOR				BGC_BLUE
#define DEF_BORDERLINEON		true
#define DEF_QUALITYHI			true

//
// Constructor
//
CVideoQuadSplitter::CVideoQuadSplitter(TCHAR *tszName,LPUNKNOWN punk,HRESULT *phr) :
CCpndBase2Filter(tszName, punk, CLSID_VideoQuadSplitter, 4),
CPersistStream(punk, phr),
_timeStamp(DEF_TIMESTAMP),
_bgColor(DEF_BGCOLOR),
_borderLineOn(DEF_BORDERLINEON),
_qualityHi(DEF_QUALITYHI),
_baseBufSize(0),
_baseBuf(NULL),
m_iNonCnt0(0),
m_iNonCnt1(0),
m_iNonCnt2(0),
m_iNonCnt3(0)
{
    ASSERT(tszName);
    ASSERT(phr);

} // VideoQuadSplitter

CVideoQuadSplitter::~CVideoQuadSplitter()
{
	delete _baseBuf;
}

//
// CreateInstance
//
// Provide the way for COM to create a CVideoQuadSplitter object
//
CUnknown * WINAPI CVideoQuadSplitter::CreateInstance(LPUNKNOWN punk, HRESULT *phr) {

    CVideoQuadSplitter *pNewObject = new CVideoQuadSplitter(NAME("VideoQuadSplitter"), punk, phr);
    if (pNewObject == NULL) {
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;

} // CreateInstance


//
// NonDelegatingQueryInterface
//
// Reveals IVideoQuadSplitter and ISpecifyPropertyPages
//
STDMETHODIMP CVideoQuadSplitter::NonDelegatingQueryInterface(REFIID riid, void **ppv)
{
    CheckPointer(ppv,E_POINTER);

    if (riid == IID_IVideoQuadSplitter) {
        return GetInterface((IVideoQuadSplitter *) this, ppv);
	} else if (riid == IID_IPersistStream) {
		return GetInterface((IPersistStream *) this, ppv);
    } else if (riid == IID_ISpecifyPropertyPages) {
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    } else {
        return CCpndBase2Filter::NonDelegatingQueryInterface(riid, ppv);
    }

} // NonDelegatingQueryInterface

//
// Transform
//
// Copy the input sample into the output sample
// Then transform the output sample 'in place'
//
HRESULT CVideoQuadSplitter::Transform(IMediaSample **ppInputs, IMediaSample *pOut)
{
	ASSERT(ppInputs[0] || ppInputs[1] || ppInputs[2] || ppInputs[3]);

    CAutoLock cAutoLock(&m_VideoQuadSplitterLock);

	const int BGCCNT = 15;

	if (ppInputs[0])
	{
		m_iNonCnt0 = 0;
		if (FAILED(Compound(ppInputs[0], pOut, 0)))
			return E_FAIL;
	}
	else
	{
		m_iNonCnt0++;
		if (m_iNonCnt0 == BGCCNT)
		{
			m_iNonCnt0 = 0;
			if (FAILED(Fill(pOut, 0)))
				return E_FAIL;
		}
	}

	if (ppInputs[1])
	{
		m_iNonCnt1 = 0;
		if (FAILED(Compound(ppInputs[1], pOut, 1)))
			return E_FAIL;
	}
	else
	{
		m_iNonCnt1++;
		if (m_iNonCnt1 == BGCCNT)
		{
			m_iNonCnt1 = 0;
			if (FAILED(Fill(pOut, 1)))
				return E_FAIL;
		}
	}

	if (ppInputs[2])
	{
		m_iNonCnt2 = 0;
		if (FAILED(Compound(ppInputs[2], pOut, 2)))
			return E_FAIL;
	}
	else
	{
		m_iNonCnt2++;
		if (m_iNonCnt2 == BGCCNT)
		{
			m_iNonCnt2 = 0;
			if (FAILED(Fill(pOut, 2)))
				return E_FAIL;
		}
	}

	if (ppInputs[3])
	{
		m_iNonCnt3 = 0;
		if (FAILED(Compound(ppInputs[3], pOut, 3)))
			return E_FAIL;
	}
	else
	{
		m_iNonCnt3++;
		if (m_iNonCnt3 == BGCCNT)
		{
			m_iNonCnt3 = 0;
			if (FAILED(Fill(pOut, 3)))
				return E_FAIL;
		}
	}

	if (_borderLineOn)
	{
		if (FAILED(DrawBorderLine(pOut)))
			return E_FAIL;
	}

	switch(_timeStamp)
	{
	case TS_IN0:
		SetupProp(ppInputs[0], pOut);
		break;
	case TS_IN1:
		SetupProp(ppInputs[1], pOut);
		break;
	case TS_IN2:
		SetupProp(ppInputs[2], pOut);
		break;
	case TS_IN3:
		SetupProp(ppInputs[3], pOut);
		break;
	case TS_ORIGINAL:
		SetupProp(NULL, pOut);
		break;
	}

	return NOERROR;

} // Transform


HRESULT
CVideoQuadSplitter::Fill(IMediaSample *pDest, unsigned int place) const
{
    BYTE *pDestBuffer, *pDB;
	long hSize, bcW, bcH, bcHW, bcHH, hh;

    pDest->GetPointer(&pDestBuffer);

	long lDestSize = pDest->GetActualDataLength();
	if (_baseBufSize != lDestSize)
	{
		ASSERT(FALSE);
		return E_FAIL;
	}

	CMediaType mt = m_pOutput->CurrentMediaType();
	VIDEOINFO* vinfo = (VIDEOINFO *)mt.Format();

	pDB = pDestBuffer;

	int bCnt = vinfo->bmiHeader.biBitCount / 8;
	hSize = pDest->GetActualDataLength() / 2;
	bcW = vinfo->bmiHeader.biWidth * bCnt;
	bcH = vinfo->bmiHeader.biHeight * bCnt;
	bcHW = bcW / 2;
	bcHH = bcH / 2;
	hh = vinfo->bmiHeader.biHeight / 2;

	// Seek
	switch(place)
	{
	case 0:	pDB += hSize;			break;
	case 1:	pDB += (hSize + bcHW);	break;
	case 2:	pDB += 0;				break;
	case 3:	pDB += bcHW;			break;
	}

	for(int i = 0; i < hh; i++)
	{
		CopyMemory(pDB, _baseBuf, bcHW);
		pDB += bcW;
	}


	return NOERROR;
}

HRESULT
CVideoQuadSplitter::Compound(IMediaSample *pSource, IMediaSample *pDest, unsigned int place) const
{
    BYTE *pSourceBuffer, *pDestBuffer;
	BYTE *pSB, *pDB, *pSB2;
	long size, hSize;
	long w, h, hw, hh, bcW, bcH, bcHW, bcHH;
	int i, j;

    pSource->GetPointer(&pSourceBuffer);
    pDest->GetPointer(&pDestBuffer);

	pSB = pSourceBuffer;
	pDB = pDestBuffer;

	CMediaType mt = m_pOutput->CurrentMediaType();
	VIDEOINFO* vinfo = (VIDEOINFO *)mt.Format();

	// Buffer Size
	size = pDest->GetActualDataLength();
	hSize = size / 2;

	int bCnt = vinfo->bmiHeader.biBitCount / 8;

	// Full Size
	w = vinfo->bmiHeader.biWidth;
	h = vinfo->bmiHeader.biHeight;
	bcW = w * bCnt;
	bcH = h * bCnt;

	// Half Size
	hw = w / 2;
	hh = h / 2;
	bcHW = hw * bCnt;
	bcHH = hh * bCnt;


	// Seek
	switch(place)
	{
	case 0:	pDB += hSize;			break;
	case 1:	pDB += (hSize + bcHW);	break;
	case 2:	pDB += 0;				break;
	case 3:	pDB += bcHW;			break;
	}

	pSB2 = pSB + bcW;


	if (_qualityHi)
	{
		short v1, v2, v3;
		switch(bCnt)
		{
		case 2:
			// RGB555, RGB565
			for(i = 0; i < hh; i++)
			{
				for(j = 0; j < hw; j++)
				{
					v1 = ((*((short *)pSB) & 0xf800) >> 11);
					v2 = ((*((short *)pSB) & 0x3e0) >> 5);
					v3 = (*((short *)pSB) & 0x1f);
					pSB += 2;

					v1 += ((*((short *)pSB) & 0xf800) >> 11);
					v2 += ((*((short *)pSB) & 0x3e0) >> 5);
					v3 += (*((short *)pSB) & 0x1f);
					pSB += 2;

					v1 += ((*((short *)pSB2) & 0xf800) >> 11);
					v2 += ((*((short *)pSB2) & 0x3e0) >> 5);
					v3 += (*((short *)pSB2) & 0x1f);
					pSB2 += 2;

					v1 += ((*((short *)pSB2) & 0xf800) >> 11);
					v2 += ((*((short *)pSB2) & 0x3e0) >> 5);
					v3 += (*((short *)pSB2) & 0x1f);
					pSB2 += 2;

					*((short *)pDB) = ((v1 / 4) << 11) + ((v2 / 4) << 5) + (v3 / 4);
					pDB += 2;

				}
				pDB += bcHW;
				pSB += bcW;
				pSB2 += bcW;
			}
			break;
		case 3:
			// RGB24
			for(i = 0; i < hh; i++)
			{
				for(j = 0; j < hw; j++)
				{
					v1 = *pSB;		pSB++;
					v2 = *pSB;		pSB++;
					v3 = *pSB;		pSB++;

					v1 += *pSB;		pSB++;
					v2 += *pSB;		pSB++;
					v3 += *pSB;		pSB++;

					v1 += *pSB2;	pSB2++;
					v2 += *pSB2;	pSB2++;
					v3 += *pSB2;	pSB2++;

					v1 += *pSB2;	pSB2++;
					v2 += *pSB2;	pSB2++;
					v3 += *pSB2;	pSB2++;

					*pDB = v1 / 4;	pDB++;
					*pDB = v2 / 4;	pDB++;
					*pDB = v3 / 4;	pDB++;
				}

				pDB += bcHW;
				pSB += bcW;
				pSB2 += bcW;
			}
			break;
		}
	} else 
	{
		switch(bCnt)
		{
		case 2:
			for(i = 0; i < hh; i++)
			{
				for(j = 0; j < hw; j++)
				{
					CopyMemory(pDB, pSB, 2);
					pDB += 2;
					pSB += 4;
				}
				pDB += bcHW;
				pSB += bcW;
			}
			break;
		case 3:
			for(i = 0; i < hh; i++)
			{
				for(j = 0; j < hw; j++)
				{
					CopyMemory(pDB, pSB, 3);
					pDB += 3;
					pSB += 6;
				}
				pDB += bcHW;
				pSB += bcW;
			}
			break;
		}	
	}

	return NOERROR;
}

HRESULT
CVideoQuadSplitter::DrawBorderLine(IMediaSample* pDest) const
{
    BYTE *pDestBuffer;
	BYTE *pDB;
	long size, hSize;
	long w, h, hw, hh, bcW, bcH, bcHW, bcHH;
	int i;

    pDest->GetPointer(&pDestBuffer);

	pDB = pDestBuffer;

	CMediaType mt = m_pOutput->CurrentMediaType();
	VIDEOINFO* vinfo = (VIDEOINFO *)mt.Format();

	// Buffer Size
	size = pDest->GetActualDataLength();
	hSize = size / 2;

	int bCnt = vinfo->bmiHeader.biBitCount / 8;

	// Full Size
	w = vinfo->bmiHeader.biWidth;
	h = vinfo->bmiHeader.biHeight;
	bcW = w * bCnt;
	bcH = h * bCnt;

	// Half Size
	hw = w / 2;
	hh = h / 2;
	bcHW = hw * bCnt;
	bcHH = hh * bCnt;


	switch(bCnt)
	{
	case 2:
		// Horizontal seek
		pDB += hSize - bcW;

		w *= 2;
		for(i = 0; i < w; i++)
		{
			*pDB = 0x7f; pDB++;
			*pDB = 0x0f; pDB++;
		}

		pDB = pDestBuffer;
		pDB += (bcHW - bCnt);

		for(i = 0; i < h; i++)
		{
			*pDB = 0x7f; pDB++;
			*pDB = 0x0f; pDB++;
			*pDB = 0x7f; pDB++;
			*pDB = 0x0f; pDB++;
			*pDB = 0x7f; pDB++;
			*pDB = 0x0f; pDB++;

			pDB += bcW - bCnt * 3;
		}

		break;
	case 3:
		// Horizontal seek
		pDB += hSize - bcW;

		w *= 3;
		for(i = 0; i < w; i++)
		{
			*pDB = 0xff; pDB++;
			*pDB = 0xff; pDB++;
			*pDB = 0x00; pDB++;
		}

		pDB = pDestBuffer;
		pDB += (bcHW - bCnt);

		for(i = 0; i < h; i++)
		{
			*pDB = 0xff; pDB++;
			*pDB = 0xff; pDB++;
			*pDB = 0x00; pDB++;
			*pDB = 0xff; pDB++;
			*pDB = 0xff; pDB++;
			*pDB = 0x00; pDB++;
			*pDB = 0xff; pDB++;
			*pDB = 0xff; pDB++;
			*pDB = 0x00; pDB++;

			pDB += bcW - bCnt * 3;
		}

		break;
	}

	return NOERROR;
}

HRESULT
CVideoQuadSplitter::SetupProp(IMediaSample* pSource, IMediaSample* pDest) const
{
	HRESULT timeRes;
	REFERENCE_TIME TimeStart, TimeEnd;
	HRESULT mTimeRes;
	LONGLONG MediaStart, MediaEnd;
	HRESULT syncPoint;
	HRESULT preRoll;
	HRESULT discont;

	if (!pSource)
	{
		TimeStart = m_tStartOrg;
		TimeEnd = m_tStopOrg;
		//timeRes = (!m_bTimeStampZero ? S_OK : S_FALSE);
		timeRes = S_OK;
		MediaStart = m_llMediaStart;
		MediaEnd = m_llMediaEnd;
		mTimeRes = S_OK;
		syncPoint = S_OK;
		preRoll = S_FALSE;
		//discont = (m_dwPrevTime != m_dwInterval) ? S_FALSE : S_OK;
		discont = S_FALSE;
	} else
	{
		timeRes = pSource->GetTime(&TimeStart, &TimeEnd);
		mTimeRes = pSource->GetMediaTime(&MediaStart, &MediaEnd);
		syncPoint = pSource->IsSyncPoint();
		preRoll = pSource->IsPreroll();
		discont = pSource->IsDiscontinuity();
	}

#ifdef DEBUG
	{
		static DWORD tint = 0;
		char buffer[256];
		sprintf(buffer, "TimeStart:%d  TimeEnd:%d  Sub:%d [%s] MediaStart:%d  MediaEnd:%d [%s]  SP:%s  PR:%s  DC:%s  INT:%d  TINT:%d\n",
			(int)(TimeStart / (UNITS / MILLISECONDS)),
			(int)(TimeEnd / (UNITS / MILLISECONDS)),
			(int)((TimeEnd / (UNITS / MILLISECONDS)) - TimeStart / (UNITS / MILLISECONDS)),
			(timeRes == S_OK) ? "T" : "F",
			(int)MediaStart,
			(int)MediaEnd,
			(mTimeRes == S_OK) ? "T" : "F",
			(syncPoint == S_OK) ? "T" : "F",
			(preRoll == S_OK) ? "T" : "F",
			(discont == S_OK) ? "T" : "F",
			m_dwInterval, 
			GetTickCount() - tint);
		tint = GetTickCount();
		OutputDebugString(buffer);
	}
#endif

	if (timeRes == NOERROR) {
		pDest->SetTime(&TimeStart, &TimeEnd);
	}


	if (mTimeRes == NOERROR) {
		pDest->SetMediaTime(&MediaStart,&MediaEnd);
	}


	if (syncPoint == S_OK)
	{
		pDest->SetSyncPoint(TRUE);
	}
	else if (syncPoint == S_FALSE)
	{
		pDest->SetSyncPoint(FALSE);
	}
	else {
		return E_UNEXPECTED;
	}


	if (preRoll == S_OK)
	{
		pDest->SetPreroll(TRUE);
	}
	else if (preRoll == S_FALSE)
	{
		pDest->SetPreroll(FALSE);
	}
	else
	{
		return E_UNEXPECTED;
	}


	if (discont == S_OK)
	{
		pDest->SetDiscontinuity(TRUE);
	}
	else if (discont == S_FALSE)
	{
		pDest->SetDiscontinuity(FALSE);
	}
	else
	{
		return E_UNEXPECTED;
	}


	// Copy the media type
	/*
	AM_MEDIA_TYPE *pMediaType;
	pSource->GetMediaType(&pMediaType);
	pDest->SetMediaType(pMediaType);
	DeleteMediaType(pMediaType);
	*/

	// Copy the actual data length
	/*
	long lDataLength = pSource->GetActualDataLength();
	pDest->SetActualDataLength(lDataLength);
	*/

	return NOERROR;
}


HRESULT
CVideoQuadSplitter::AllocBaseBuffer(unsigned int size)
{
	delete [] _baseBuf;

	_baseBuf = new BYTE[size];
	if (!_baseBuf)
	{
		_baseBufSize = 0;
		return E_FAIL;
	}
	_baseBufSize = size;

	return NOERROR;
}

HRESULT
CVideoQuadSplitter::DrawBaseBufferInBGColor()
{
	if (!_baseBuf || !_baseBufSize)
		return E_FAIL;

	CMediaType cmt = m_pOutput->CurrentMediaType();

	if (*(cmt.FormatType()) != FORMAT_VideoInfo || _bgColor == BGC_BLACK)
	{
		ZeroMemory((PVOID)_baseBuf, _baseBufSize);
		return NOERROR;
	}

	if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB555))
	{
		const BYTE colors[3][2] =
		{
			{ 0x00, 0xf8 },
			{ 0xe0, 0x07 },
			{ 0x1f, 0x00 }
		};

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 2; i++)
		{
			CopyMemory((PVOID)pBaseBuf, (PVOID)colors[_bgColor - 1], sizeof(BYTE) * 2);
			pBaseBuf += 2;
		}
	}
	else if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB565))
	{
		const BYTE colors[3][2] =
		{
			{ 0x00, 0x7c },
			{ 0xe0, 0x03 },
			{ 0x1f, 0x00 }
		};

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 2; i++)
		{
			CopyMemory((PVOID)pBaseBuf, (PVOID)colors[_bgColor - 1], sizeof(BYTE) * 2);
			pBaseBuf += 2;
		}
	}
	else if (IsEqualGUID(*(cmt.Subtype()), MEDIASUBTYPE_RGB24))
	{
		const BYTE colors[3][3] =
		{
			{ 0x00, 0x00, 0xff },
			{ 0x00, 0xff, 0x00 },
			{ 0xff, 0x00, 0x00 }
		};

		BYTE* pBaseBuf = _baseBuf;
		for(int i = 0; i < _baseBufSize / 3; i++)
		{
			CopyMemory((PVOID)pBaseBuf, (PVOID)colors[_bgColor - 1], sizeof(BYTE) * 3);
			pBaseBuf += 3;
		}
	}
	else
	{
		ZeroMemory(_baseBuf, _baseBufSize);
	}

	return NOERROR;
}

//
// CheckInputType
//
// Check the input type is OK, return an error otherwise
//
HRESULT CVideoQuadSplitter::CheckInputType(const CMediaType *mtIn, unsigned int pinIdx)
{
	ASSERT(m_ppInputs[0]);

	if (*mtIn->FormatType() == FORMAT_VideoInfo)
	{
		if (!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB555) &&
			!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB565) &&
			!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB24))
			return E_INVALIDARG;
		else
			return NOERROR;
	}

	return E_INVALIDARG;

} // CheckInputType

HRESULT CVideoQuadSplitter::CheckOutputType(const CMediaType *mtOut)
{
	ASSERT(m_ppInputs[0]);

	if (*mtOut->FormatType() != FORMAT_VideoInfo){
		return E_INVALIDARG;
	}

	CCpndBase2InputPin* pin = GetConnectedInputPin();
	if (!pin)
		return E_INVALIDARG;

	const CMediaType& cmt = pin->CurrentMediaType();
	if (!IsEqualGUID(*cmt.Subtype(), *mtOut->Subtype()))
		return E_INVALIDARG;

	VIDEOINFO* pivi = (VIDEOINFO *)cmt.Format();
	VIDEOINFO* povi = (VIDEOINFO *)mtOut->Format();

	if (pivi->bmiHeader.biWidth != povi->bmiHeader.biWidth ||
		pivi->bmiHeader.biHeight != povi->bmiHeader.biHeight)
		return E_INVALIDARG;

	return NOERROR;
}

//
// GetMediaType
//
// I support one type, namely the type of the input pin
// We must be connected to support the single output type
//
HRESULT CVideoQuadSplitter::GetMediaType(int iPosition, CMediaType *pMediaType)
{
	ASSERT(m_ppInputs[0]);

	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
    if (iPosition < 0) {
        return E_INVALIDARG;
    }
    if (iPosition > 0) {
        return VFW_S_NO_MORE_ITEMS;
    }

	CCpndBase2InputPin *pin = GetConnectedInputPin();
	if (!pin)
		return E_UNEXPECTED;

	*pMediaType = pin->CurrentMediaType();

	VIDEOINFO *vinfo = (VIDEOINFO *)pMediaType->Format();
	pMediaType->SetSampleSize(GetBitmapSize(&vinfo->bmiHeader));

	return NOERROR;

} // GetMediaType


//
// DecideBufferSize
//
// Tell the output pin's allocator what size buffers we
// require. Can only do this when the input is connected
//
HRESULT CVideoQuadSplitter::DecideBufferSize(IMemAllocator *pAlloc,ALLOCATOR_PROPERTIES *pProp)
{
	CCpndBase2InputPin* inputPin = GetConnectedInputPin();
	if (!inputPin)
		return E_UNEXPECTED;

	ASSERT(pAlloc);
	ASSERT(pProp);
	ASSERT(inputPin->CurrentMediaType().bFixedSizeSamples);

	VIDEOINFOHEADER *pvi = (VIDEOINFOHEADER *)m_pOutput->CurrentMediaType().pbFormat;
	pvi->bmiHeader.biSizeImage = GetBitmapSize(&pvi->bmiHeader);
	ASSERT(pvi->bmiHeader.biSizeImage != 0);

	pProp->cBuffers = 1;
	pProp->cbBuffer = pvi->bmiHeader.biSizeImage;

	ALLOCATOR_PROPERTIES Actual;
	HRESULT hr = pAlloc->SetProperties(pProp, &Actual);
	if (FAILED(hr))
		return hr;

	ASSERT(Actual.cBuffers == 1);

	if (Actual.cBuffers < pProp->cBuffers ||
		Actual.cbBuffer < pProp->cbBuffer)
		return E_FAIL;

    /// Allocate BASE BUFFER
	hr = AllocBaseBuffer(Actual.cbBuffer);
	if (FAILED(hr))
		return E_FAIL;

	/// Draw BASE BUFFER in BG COLOR
	hr = DrawBaseBufferInBGColor();
	if (FAILED(hr))
		return E_FAIL;

    return NOERROR;
}


HRESULT
CVideoQuadSplitter::SetMediaType(PIN_DIRECTION direction, const CMediaType *pmt, unsigned int pinIdx)
{
	ASSERT(m_ppInputs[0]);

	HRESULT hr = NOERROR;

	// Check tow input pin VideoResolution
	if (direction == PINDIR_INPUT)
	{
		for(unsigned int i = 0; i < m_iInputPinCnt; i++)
		{
			if (i != pinIdx && m_ppInputs[i]->IsConnected())
			{
				CMediaType cmt = m_ppInputs[i]->CurrentMediaType();

				if (!IsEqualGUID(*pmt->Subtype(), *cmt.Subtype()))
					return E_FAIL;

				VIDEOINFO *vinfo1 = (VIDEOINFO *)cmt.Format();
				VIDEOINFO *vinfo2 = (VIDEOINFO *)pmt->Format();

				if (vinfo1->bmiHeader.biWidth != vinfo2->bmiHeader.biWidth ||
					vinfo1->bmiHeader.biHeight!= vinfo2->bmiHeader.biHeight)
					return E_FAIL;

				break;
			}
		}
	}

	// Reconnect 
	CCpndBase2InputPin *pin = GetConnectedInputPin();
    if(pin && m_pOutput->IsConnected())
	{
        FILTER_INFO fInfo;
        QueryFilterInfo( &fInfo );

        if (direction == PINDIR_INPUT && *pmt != m_pOutput->CurrentMediaType())
            fInfo.pGraph->Reconnect( m_pOutput );

        QueryFilterInfoReleaseGraph( fInfo );
    }

	return hr;
}


STDMETHODIMP
CVideoQuadSplitter::get_TimeStamp(unsigned int *timeStamp)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	*timeStamp = (unsigned int)_timeStamp;

	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::put_TimeStamp(unsigned int timeStamp)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	if (_timeStamp != timeStamp)
	{
		_timeStamp = (TimeStampEnum)timeStamp;

		if (_timeStamp == TS_ORIGINAL)
			SetReferenceStream(-1);
		else
			SetReferenceStream((int)_timeStamp);

		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::get_TimeStampZero(bool *flag)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	*flag = GetTimeStampZero();
	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::put_TimeStampZero(bool flag)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);

	if (GetTimeStampZero() != flag)
	{
		SetTimeStampZero(flag);
		SetDirty(TRUE);
	}
	
	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::get_TimeOffset(int *timeOffset)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	*timeOffset = (int)GetTimeOffset();
	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::put_TimeOffset(int timeOffset)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);

	if (GetTimeOffset() != timeOffset)
	{
		SetTimeOffset(timeOffset);
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::get_BackgroundColor(unsigned int *bgColor)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	*bgColor = (unsigned int)_bgColor;

	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::put_BackgroundColor(unsigned int bgColor)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	if (_bgColor != bgColor)
	{
		_bgColor = (BackgroundColorEnum)bgColor;
		SetDirty(TRUE);

		DrawBaseBufferInBGColor();
	}

	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::get_FrameRate(int *rate)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	*rate = GetFrameRate();

	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::put_FrameRate(int rate)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	if (GetFrameRate() != rate)
	{
		SetFrameRate(rate);
		SetDirty(TRUE);
	}

	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::get_BorderLineOn(bool *flag)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	*flag = _borderLineOn;

	return NOERROR;
}

STDMETHODIMP
CVideoQuadSplitter::put_BorderLineOn(bool flag)
{
	CAutoLock cAutoLock(&m_VideoQuadSplitterLock);
	if (_borderLineOn != flag)
	{
		_borderLineOn = flag;
		SetDirty(TRUE);
	}

	return NOERROR;
}



#define WRITEOUT(var)  hr = pStream->Write(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define WRITENOUT(var,n)  hr = pStream->Write(var, n, NULL); \
if (FAILED(hr)) return hr;
#define READIN(var)    hr = pStream->Read(&var, sizeof(var), NULL); \
if (FAILED(hr)) return hr;
#define READNIN(var,n)    hr = pStream->Read(var, n, NULL); \
if (FAILED(hr)) return hr;


// IPersitStream
STDMETHODIMP
CVideoQuadSplitter::GetClassID(CLSID *pClsid)
{
	*pClsid= CLSID_VideoQuadSplitter;
	return S_OK;
} // GetClassID

HRESULT
CVideoQuadSplitter::WriteToStream(IStream *pStream)
{
	int frameRate = GetFrameRate();
	bool timeStampZero = GetTimeStampZero();
	int timeOffset = GetTimeOffset();

	HRESULT hr;
	WRITEOUT(_timeStamp);
	WRITEOUT(timeStampZero);
	WRITEOUT(timeOffset);
	WRITEOUT(_bgColor);
	WRITEOUT(frameRate);
	WRITEOUT(_borderLineOn);

	return NOERROR;
	
} // WriteToStream

HRESULT
CVideoQuadSplitter::ReadFromStream(IStream *pStream)
{
	int frameRate;
	bool timeStampZero;
	int timeOffset;

	HRESULT hr;
	READIN(_timeStamp);
	READIN(timeStampZero);
	READIN(timeOffset);
	READIN(_bgColor);
	READIN(frameRate);
	READIN(_borderLineOn);

	SetFrameRate(frameRate);
	SetTimeStampZero(timeStampZero);
	SetTimeOffset(timeOffset);

	if (_timeStamp == TS_ORIGINAL)
		SetReferenceStream(-1);
	else
		SetReferenceStream((int)_timeStamp);

	return NOERROR;
	
} // ReadFromStream

DWORD
CVideoQuadSplitter::GetSoftwareVersion(void)
{
	return 1;
} // GetSoftwareVersion

int
CVideoQuadSplitter::SizeMax()
{
	return sizeof(unsigned int) * 4 + sizeof(bool) * 2;
} // SizeMax


//
// GetPages
//
// This is the sole member of ISpecifyPropertyPages
// Returns the clsid's of the property pages we support
//
STDMETHODIMP CVideoQuadSplitter::GetPages(CAUUID *pPages)
{
    pPages->cElems = 1;
    pPages->pElems = (GUID *) CoTaskMemAlloc(sizeof(GUID));
    if (pPages->pElems == NULL) {
        return E_OUTOFMEMORY;
    }
    *(pPages->pElems) = CLSID_VideoQuadSplitterPropertyPage;
    return NOERROR;

} // GetPages


//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2( TRUE );

} // DllRegisterServer

//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2( FALSE );

} // DllUnregisterServer

