/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flVolumeGraph/VolumeGraphInputPin.h>
#include <flVolumeGraph/VolumeGraphOutputPin.h>
#include <flVolumeGraph/VolumeGraph.h>
#include <flVolumeGraph/VolumeGraphProp.h>
#include <flFilterCommon/flPCMAudio.h>
#include <flTypes/IVolumeGraph.h>
#include <flBase/flSystemTime.h>
#include <flBase/flNotify.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudVolumeGraph =
{
    &CLSID_VolumeGraph,		// Filter CLSID
    L"NIME Volume Graph",	// String name
    MERIT_DO_NOT_USE,		// Filter merit
    0,						// Number of pins
    NULL					// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[2] =
{
    { L"VolumeGraph"
    , &CLSID_VolumeGraph
    , CVolumeGraph::CreateInstance
    , NULL
    , &sudVolumeGraph }
  ,
    { L"VolumeGraphProp"
    , &CLSID_VolumeGraphPropertyPage
    , CVolumeGraphProp::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CVolumeGraph::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
    CVolumeGraph *pNewObject = new CVolumeGraph(NAME("VolumeGraph"), punk, phr);
    if (pNewObject == NULL)
	{
        *phr = E_OUTOFMEMORY;
    }
    return pNewObject;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CVolumeGraph::CVolumeGraph(TCHAR *pName, LPUNKNOWN punk, HRESULT *phr) :
CBaseFilter(pName, punk, &_filterLock, CLSID_VolumeGraph),
CPersistStream(punk, phr)
{
	// Initialize Dynamic input pins
	InitVolumeGraphInOutPins();

	// Create First input pin
	CreateInOutPin();

	_absTime			= new flSystemTime();

	_evalTime			= 20;
	_displayTime		= _VGDT_20000MS;
}
//------------------------------------------------------------------------------
CVolumeGraph::~CVolumeGraph()
{
	delete _absTime;

	InitVolumeGraphInOutPins();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
    CheckPointer(ppv, E_POINTER);

    if (riid == IID_IVolumeGraph)
	{
        return GetInterface((IVolumeGraph *) this, ppv);
	}
	else if (riid == IID__IVolumeGraph)
	{
		return GetInterface((_IVolumeGraph *) this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
        return GetInterface((ISpecifyPropertyPages *) this, ppv);
    }
	else
	{
		return CBaseFilter::NonDelegatingQueryInterface(riid, ppv);
    }
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
int
CVolumeGraph::GetPinCount()
{
	return _inputPins.getSize() + _outputPins.getSize();
}
//------------------------------------------------------------------------------
CBasePin*
CVolumeGraph::GetPin(int n)
{
	if (n < 0)
		return NULL;

	flUInt inSize = _inputPins.getSize();
	flUInt outSize = _outputPins.getSize();

	if (inSize + outSize <= (flUInt)n)
		return NULL;

	if ((flUInt)n < inSize)
		return _inputPins[n];
	else
		return _outputPins[n - inSize];
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::FindPin(LPCWSTR Id, IPin **ppPin)
{
	flInt i = WstrToInt(Id);
	*ppPin = GetPin(i);
	if (*ppPin != NULL)
	{
		(*ppPin)->AddRef();
		return NOERROR;
	}
	else
	{
		return VFW_E_NOT_FOUND;
	}
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::Stop()
{
	CAutoLock lock1(&_filterLock);
	if (m_State == State_Stopped)
		return NOERROR;

	for(flUInt i = 0; i < _inputPins.getSize(); i++)
	{
		_inputPins[i]->Inactive();

		CAutoLock lock2(_receiveLocks[i]);
		_outputPins[i]->Inactive();
	}

	m_State = State_Stopped;

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::Pause()
{
	CAutoLock lock(&_filterLock);
	return CBaseFilter::Pause();
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::GetClassID(CLSID *pClsid)
{
	*pClsid = CLSID_VolumeGraph;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraph::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	WRITEOUT(_evalTime);
	WRITEOUT(_displayTime);
	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraph::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	READIN(_evalTime);
	READIN(_displayTime);
	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CVolumeGraph::GetSoftwareVersion(void)
{
	return sizeof(flInt) * 2;
}
//------------------------------------------------------------------------------
int
CVolumeGraph::SizeMax()
{
	return 1;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// ISpecifyPropertyPages
STDMETHODIMP
CVolumeGraph::GetPages(CAUUID *pPages)
{
	pPages->cElems = 1;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 1);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_VolumeGraphPropertyPage;
	return NOERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// IVolumeGraph
STDMETHODIMP
CVolumeGraph::get_NumInputs(flUInt *numInputs)
{
	CAutoLock lock(&_filterLock);
	*numInputs = _inputPins.getSize();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::get_Volume(flUInt inputIndex, VolumeInfo *volume)
{
	if (_pcmAudios.getSize() <= inputIndex)
		return NOERROR;

	{
		CAutoLock lock(_pcmAudioLocks[inputIndex]);

		_absTime->update();
		flInt64 timeStamp = _absTime->getTime();

		flInt64 evalTime64 = flInt64(_evalTime) * 10000;
		_pcmAudios[inputIndex]->getVolume(*volume, timeStamp - evalTime64, timeStamp);
	}
	Sleep(0);

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::get_EvaluationTime(flULong* evalTime)
{
	CAutoLock lock(&_filterLock);
	*evalTime = _evalTime;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::put_EvaluationTime(flULong evalTime)
{
	CAutoLock lock(&_filterLock);
	_evalTime = evalTime;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::get_DisplayTime(flInt* displayTime)
{
	CAutoLock lock(&_filterLock);
	*displayTime = _displayTime;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraph::put_DisplayTime(flInt displayTime)
{
	CAutoLock lock(&_filterLock);
	_displayTime = displayTime;
	return NOERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CBasePin*
CVolumeGraph::GetInputPin(CBasePin* outputPin)
{
	flInt index = GetOutputPinIndex(outputPin);
	return (index == -1 ? NULL : _inputPins[index]);
}
//------------------------------------------------------------------------------
CBasePin*
CVolumeGraph::GetOutputPin(CBasePin* inputPin)
{
	flInt index = GetInputPinIndex(inputPin);
	return (index == -1 ? NULL : _outputPins[index]);
}
//------------------------------------------------------------------------------
CCritSec*
CVolumeGraph::GetReceiveLock(CBasePin* inputPin)
{
	flInt index = GetInputPinIndex(inputPin);
	return (index == -1 ? NULL : _receiveLocks[index]);
}
//------------------------------------------------------------------------------
// Called Input/Output
HRESULT
CVolumeGraph::CheckMediaType(CBasePin* pin, const CMediaType *pmt)
{
	flInt index = GetInputPinIndex(pin);
	if (index != -1)
	{
		CVolumeGraphOutputPin* outPin = _outputPins[index];
		if (outPin->IsConnected() == FALSE ||
			*pmt == outPin->CurrentMediaType())
			return NOERROR;
	}
	else
	{
		index = GetOutputPinIndex(pin);
		if (index != -1)
		{
			CVolumeGraphInputPin* inPin = _inputPins[index];
			if (*pmt == inPin->CurrentMediaType())
				return NOERROR;
		}
		else
		{
			ASSERT(FALSE);
		}
	}

	return E_INVALIDARG;
}
//------------------------------------------------------------------------------
// Called Input
HRESULT
CVolumeGraph::SetMediaType(CBasePin* inputPin, const CMediaType *pmt)
{
	CAutoLock lock(&_filterLock);

	flInt index = GetInputPinIndex(inputPin);
	ASSERT(index != -1);

	WAVEFORMATEX *format = (WAVEFORMATEX *)pmt->pbFormat;
	_pcmAudios[index]->setFormat(format);

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraph::Receive(CBasePin* inputPin, IMediaSample* pSample)
{
	// Get Receive Index
	flInt index = GetInputPinIndex(inputPin);
	ASSERT(index != -1);

	// Get MediaSample Data
	BYTE *data;
	pSample->GetPointer(&data);
	flLong dataLength = pSample->GetActualDataLength();

	{
		CAutoLock lock(_pcmAudioLocks[index]);

		// Buffering PCM Data
		_absTime->update();
		_pcmAudios[index]->setData(data, dataLength, _absTime->getTime());
	}
	Sleep(0);

	// Deliver MediaSample to Downstream
	CVolumeGraphOutputPin* outPin = _outputPins[index];
	if (outPin->IsConnected() == FALSE)
		return NOERROR;
	return outPin->Deliver(pSample);
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraph::EndOfStream(CBasePin* inputPin)
{
	flInt index = GetInputPinIndex(inputPin);
	if (index != -1)
	{
		flNotify::post(flNotify::L_INFO, "EndOfStream [%d]", index);

		CVolumeGraphOutputPin* outPin = _outputPins[index];
		return outPin->DeliverEndOfStream();
	}
	else
	{
		ASSERT(FALSE);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraph::BeginFlush(CBasePin* inputPin)
{
	flInt index = GetInputPinIndex(inputPin);
	if (index != -1)
	{
		flNotify::post(flNotify::L_INFO, "BeginFlush [%d]", index);

		CVolumeGraphOutputPin* outPin = _outputPins[index];
		return outPin->DeliverBeginFlush();
	}
	else
	{
		ASSERT(FALSE);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraph::EndFlush(CBasePin* inputPin)
{
	flInt index = GetInputPinIndex(inputPin);
	if (index != -1)
	{
		flNotify::post(flNotify::L_INFO, "EndFlush [%d]", index);

		CVolumeGraphOutputPin* outPin = _outputPins[index];
		return outPin->DeliverEndFlush();
	}
	else
	{
		ASSERT(FALSE);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraph::NewSegment(CBasePin* inputPin,
						 REFERENCE_TIME tStart,
						 REFERENCE_TIME tStop,
						 double dRate)
{
	flInt index = GetInputPinIndex(inputPin);
	if (index != -1)
	{
		flNotify::post(flNotify::L_INFO, "NewSegment [%d]", index);

		CVolumeGraphOutputPin* outPin = _outputPins[index];
		return outPin->DeliverNewSegment(tStart, tStop, dRate);
	}
	else
	{
		ASSERT(FALSE);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
// Called Output
HRESULT
CVolumeGraph::DecideBufferSize(CBasePin* outputPin, IMemAllocator * pAlloc,
							   ALLOCATOR_PROPERTIES *pProp)
{
	flInt index;
	index = GetOutputPinIndex(outputPin);
	ASSERT(index != -1);

	CVolumeGraphInputPin* inPin = _inputPins[index];
	ASSERT(inPin->IsConnected() == TRUE);

	ASSERT(pAlloc);
	ASSERT(pProp);
	ASSERT(inPin->CurrentMediaType().bFixedSizeSamples);

	WAVEFORMATEX *format = (WAVEFORMATEX *)inPin->CurrentMediaType().pbFormat;

	pProp->cBuffers = 1;
	pProp->cbBuffer = format->nAvgBytesPerSec;

	ALLOCATOR_PROPERTIES Actual;
	HRESULT hr = pAlloc->SetProperties(pProp, &Actual);
	if (FAILED(hr))
		return hr;

	ASSERT(Actual.cBuffers == 1);

	if (Actual.cBuffers < pProp->cBuffers ||
		Actual.cbBuffer < pProp->cbBuffer)
		return E_FAIL;

    return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraph::GetMediaType(CBasePin* outputPin, CMediaType *pmt)
{
	flInt index = GetOutputPinIndex(outputPin);
	if (index != -1)
	{
		*pmt = _inputPins[index]->CurrentMediaType();
	}
	else
	{
		ASSERT(FALSE);
	}

	return NOERROR;
}
//------------------------------------------------------------------------------
// Pin Creation
void
CVolumeGraph::InitVolumeGraphInOutPins()
{
	_nextPinNumber = 0;
	for(flUInt i = 0; i < _inputPins.getSize(); i++)
	{
		CVolumeGraphInputPin* inPin = _inputPins[i];
		inPin->Release();
	}

	_receiveLocks.setSize(0);
	_receiveLocks.setGrowSize(16);

	_inputPins.setSize(0);
	_inputPins.setGrowSize(16);

	_outputPins.setSize(0);
	_outputPins.setGrowSize(16);

	_pcmAudioLocks.setSize(0);
	_pcmAudioLocks.setGrowSize(16);

	_pcmAudios.setSize(0);
	_pcmAudios.setGrowSize(16);
}
//------------------------------------------------------------------------------
void
CVolumeGraph::CreateInOutPin()
{
	WCHAR szbuf[20];
	HRESULT hr = NOERROR;

	_nextPinNumber++;

	// Input Pin
	swprintf(szbuf, L"Input%d", _nextPinNumber);
	CVolumeGraphInputPin *inPin = new CVolumeGraphInputPin(
							NAME("VolumeGraph Input"), this, &hr, szbuf);

	if (FAILED(hr) || inPin == NULL)
	{
		delete inPin;
		return ;
	}

	inPin->AddRef();
	_inputPins.add(inPin);

	// Output Pin
	swprintf(szbuf, L"Output%d", _nextPinNumber);
	CVolumeGraphOutputPin *outPin = new CVolumeGraphOutputPin(
							NAME("VolumeGraph Output"), this, &hr, szbuf);

	if (FAILED(hr) || outPin == NULL)
	{
		inPin->Release();
		delete inPin;
		delete outPin;
		return ;
	}

	outPin->AddRef();
	_outputPins.add(outPin);

	// Critical Section
	_receiveLocks.add(new CCritSec());

	// PCM Audio
	_pcmAudioLocks.add(new CCritSec());
	_pcmAudios.add(new flPCMAudio());

	// Version Up !
	IncrementPinVersion();
}
//------------------------------------------------------------------------------
void
CVolumeGraph::DeleteInPin(CVolumeGraphInputPin* inputPin)
{
	for(flUInt i = 0; i < _inputPins.getSize(); i++)
	{
		CVolumeGraphInputPin* inPin = _inputPins[i];

		if (inPin == inputPin)
		{
			// Delete InputPin
			delete inPin;
			_inputPins.removeIndex(i);

			// Delete OutputPin (in CVolumeGraphOutputPin::NonDelegateRelease().. )
			if (_outputPins.getSize() <= i)
				break;
			CVolumeGraphOutputPin* outPin = _outputPins[i];

			if (outPin->IsConnected() == TRUE)
			{
				IPin *cInPin = NULL;
				HRESULT hr = outPin->ConnectedTo(&cInPin);
				if (hr == S_OK)
				{
					cInPin->Disconnect();
					cInPin->Release();
				}
				else
				{
					ASSERT(FALSE);
				}

				outPin->Disconnect();
			}

			outPin->Release();

			// Delete Critical Section
			delete _receiveLocks[i];
			_receiveLocks.removeIndex(i);

			// Delete PCM Audio
			delete _pcmAudioLocks[i];
			_pcmAudioLocks.removeIndex(i);

			delete _pcmAudios[i];
			_pcmAudios.removeIndex(i);

			// Version Up !
			IncrementPinVersion();

			break;
		}
	}
}
//------------------------------------------------------------------------------
void
CVolumeGraph::DeleteOutPin(CVolumeGraphOutputPin* outputPin)
{
	for(flUInt i = 0; i < _outputPins.getSize(); i++)
	{
		CVolumeGraphOutputPin* outPin = _outputPins[i];

		if (outPin == outputPin)
		{
			delete outPin;
			_outputPins.removeIndex(i);

			// Version Up !
			IncrementPinVersion();

			break;
		}
	}
}
//------------------------------------------------------------------------------
flUInt
CVolumeGraph::GetNumFreeInputPins()
{
	flUInt n = 0;

	for(flUInt i = 0; i < _inputPins.getSize(); i++)
	{
		CVolumeGraphInputPin* pPin = _inputPins[i];
		if (pPin->m_Connected == NULL)
			n++;
	}

    return n;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
flInt
CVolumeGraph::GetInputPinIndex(CBasePin* inputPin)
{
	if (inputPin == NULL)
		return -1;

	flInt index;
	if (_inputPins.find((CVolumeGraphInputPin *)inputPin, (flUInt &)index))
		return index;
	return -1;
}
//------------------------------------------------------------------------------
flInt
CVolumeGraph::GetOutputPinIndex(CBasePin* outputPin)
{
	if (outputPin == NULL)
		return -1;

	flInt index;
	if (_outputPins.find((CVolumeGraphOutputPin *)outputPin, (flUInt &)index))
		return index;
	return -1;
}
//------------------------------------------------------------------------------
