/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flVolumeGraph/VolumeGraphInputPin.h>
#include <flVolumeGraph/VolumeGraph.h>
#include <flBase/flNotify.h>
//------------------------------------------------------------------------------
CVolumeGraphInputPin::CVolumeGraphInputPin(TCHAR *pObjName,
											CVolumeGraph *volumeGraph,
											HRESULT *phr,
											LPCWSTR pPinName) :
CBaseInputPin(pObjName, (CBaseFilter *)volumeGraph, &volumeGraph->_filterLock, phr, pPinName)
{
    _referenceCount			= 0;
	_volumeGraph			= volumeGraph;
}
//------------------------------------------------------------------------------
CVolumeGraphInputPin::~CVolumeGraphInputPin()
{
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraphInputPin::QueryId(LPWSTR * Id)
{
	*Id = (LPWSTR)CoTaskMemAlloc(8);
	if (*Id==NULL)
		return E_OUTOFMEMORY;
	IntToWstr(_volumeGraph->GetInputPinIndex(this), *Id);
	return NOERROR;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
STDMETHODIMP_(ULONG)
CVolumeGraphInputPin::NonDelegatingAddRef()
{
	CAutoLock lock_it(&_volumeGraph->_filterLock);

    _referenceCount++;
    ASSERT(_referenceCount > 0);

	return _referenceCount;
}
//------------------------------------------------------------------------------
STDMETHODIMP_(ULONG)
CVolumeGraphInputPin::NonDelegatingRelease()
{
    CAutoLock lock_it(&_volumeGraph->_filterLock);

    _referenceCount--;
    ASSERT(_referenceCount >= 0);

    if (_referenceCount <= 1)
	{
        flInt n = 2;
        if (_referenceCount == 1)
		{
            n = _volumeGraph->GetNumFreeInputPins();
        }

        if (n >= 2) 
		{
            _referenceCount = 0;
            _volumeGraph->DeleteInPin(this);
            return (ULONG)0;
        }
    }

	return (ULONG)_referenceCount;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
HRESULT
CVolumeGraphInputPin::CheckMediaType(const CMediaType* pmt)
{
	if (!IsEqualGUID(*pmt->Type(), MEDIATYPE_Audio) ||
		!IsEqualGUID(*pmt->Subtype(), MEDIASUBTYPE_PCM) ||
		!IsEqualGUID(*pmt->FormatType(), FORMAT_WaveFormatEx))
		return E_INVALIDARG;
	
	return _volumeGraph->CheckMediaType(this, pmt);
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraphInputPin::SetMediaType(const CMediaType* pmt)
{
	HRESULT hr = _volumeGraph->SetMediaType(this, pmt);
	if (FAILED(hr))
		return hr;

    return CBasePin::SetMediaType(pmt);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
HRESULT
CVolumeGraphInputPin::CompleteConnect(IPin *pReceivePin)
{
	CAutoLock lock(&_volumeGraph->_filterLock);

	HRESULT hr = CBaseInputPin::CompleteConnect(pReceivePin);
	if (FAILED(hr))
		return hr;

	flUInt n = _volumeGraph->GetNumFreeInputPins();
	ASSERT(n <= 1);
	if (n == 1 || _volumeGraph->_inputPins.getSize() == MAXIMUM_VIDEOGRAPHPINS)
		return NOERROR;

	_volumeGraph->CreateInOutPin();

	return hr;
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraphInputPin::Receive(IMediaSample * pSample)
{
	CAutoLock lock(_volumeGraph->GetReceiveLock(this));

	HRESULT hr = CBaseInputPin::Receive(pSample);
	if (hr == S_OK)
		hr = _volumeGraph->Receive(this, pSample);

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraphInputPin::EndOfStream()
{
	CAutoLock lock(_volumeGraph->GetReceiveLock(this));

	HRESULT hr = CheckStreaming();
	if (S_OK == hr)
		hr = _volumeGraph->EndOfStream(this);

	return hr;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraphInputPin::BeginFlush()
{
	CAutoLock lock(&_volumeGraph->_filterLock);

	ASSERT(_volumeGraph->GetOutputPin(this) != NULL);

	if (!IsConnected() || !_volumeGraph->GetOutputPin(this)->IsConnected())
		return VFW_E_NOT_CONNECTED;

	HRESULT hr = CBaseInputPin::BeginFlush();
	if (FAILED(hr))
		return hr;

	return _volumeGraph->BeginFlush(this);
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraphInputPin::EndFlush()
{
	CAutoLock lock(&_volumeGraph->_filterLock);

	ASSERT(_volumeGraph->GetOutputPin(this) != NULL);

	if (!IsConnected() || !_volumeGraph->GetOutputPin(this)->IsConnected()) 
		return VFW_E_NOT_CONNECTED;

	HRESULT hr = _volumeGraph->EndFlush(this);
	if (FAILED(hr))
		return hr;

	return CBaseInputPin::EndFlush();
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeGraphInputPin::NewSegment(REFERENCE_TIME tStart,
								 REFERENCE_TIME tStop,
								 double dRate)
{
	CBasePin::NewSegment(tStart, tStop, dRate);
	return _volumeGraph->NewSegment(this, tStart, tStop, dRate);
}
//------------------------------------------------------------------------------
CMediaType&
CVolumeGraphInputPin::CurrentMediaType()
{
	return m_mt;
}
//------------------------------------------------------------------------------
