/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flVolumeGraph/VolumeGraphProp.h>
#include <flVolumeGraph/flGraphWindow.h>

#include <commctrl.h>
#include <math.h>
#include <tchar.h>
#include "resource.h"
//------------------------------------------------------------------------------
CUnknown * WINAPI
CVolumeGraphProp::CreateInstance(LPUNKNOWN lpunk, HRESULT *phr)
{
	CUnknown *punk = new CVolumeGraphProp(lpunk, phr);
	if (punk == NULL)
	{
		*phr = E_OUTOFMEMORY;
	}
	return punk;
}
//------------------------------------------------------------------------------
CVolumeGraphProp::CVolumeGraphProp(LPUNKNOWN pUnk, HRESULT *phr) :
    CBasePropertyPage(NAME("VolumeGraphProp"),pUnk,
                      IDD_VOLUMEGRAPHPROP,
                      IDS_VOLUMEGRAPHPROPTITLE),
    _volumeGraph(NULL)
{
	InitCommonControls();
}
//------------------------------------------------------------------------------
void
CVolumeGraphProp::SetDirty()
{
	m_bDirty = TRUE;
	if (m_pPageSite)
	{
		m_pPageSite->OnStatusChange(PROPPAGESTATUS_DIRTY);
	}
}
//------------------------------------------------------------------------------
BOOL
CVolumeGraphProp::OnReceiveMessage(HWND hwnd,
                                           UINT uMsg,
                                           WPARAM wParam,
                                           LPARAM lParam)
{
    switch (uMsg)
    {
        case WM_INITDIALOG:
        {
			setupControls();
			createGraphWindow(hwnd);
			setupGraph();

			return (LRESULT) 1;
        }
        case WM_COMMAND:
        {
			switch(LOWORD(wParam))
			{
			case IDC_DISPLAYTIME_COMBO:
				if (HIWORD(wParam) == CBN_SELCHANGE)
				{
					_displayTime = ComboBox_GetCurSel(
									GetDlgItem(m_Dlg, IDC_DISPLAYTIME_COMBO));
					SetDirty();
				}
				break;
			case IDC_DISPLAYPAUSE_CHECK:
				if (HIWORD(wParam) == BN_CLICKED)
				{
					_paused = (Button_GetCheck(
									GetDlgItem(m_Dlg, IDC_DISPLAYPAUSE_CHECK)) == TRUE);
				}
				break;
			}

			return (LRESULT) 1;
		}
        case WM_DESTROY:
        {
			deleteGraphWindow();
			shutdownControls();

			return (LRESULT) 1;
        }
		case WM_TIMER:
		{
			if (_paused)
				break;
			drawGraph();
			Sleep(0);

			return (LRESULT) 1;
		}
    }
    return CBasePropertyPage::OnReceiveMessage(hwnd, uMsg, wParam, lParam);
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraphProp::OnConnect(IUnknown *pUnknown)
{
	HRESULT hr;

	// IVolumeGraph
    hr = pUnknown->QueryInterface(IID_IVolumeGraph, (void **) &_volumeGraph);
    if (FAILED(hr))
	{
        return E_NOINTERFACE;
	}

	// _IVolumeGraph
	hr = _volumeGraph->QueryInterface(IID__IVolumeGraph, (void **) &__volumeGraph);
	if (FAILED(hr))
	{
		_volumeGraph->Release();
		return E_NOINTERFACE;
	}

	_timer			= NULL;
	_paused			= false;
	_graphWindowL	= NULL;
	_graphWindowR	= NULL;

	__volumeGraph->get_DisplayTime(&_displayTime);

    return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraphProp::OnDisconnect()
{
	if (__volumeGraph == NULL)
		return E_UNEXPECTED;

	__volumeGraph->Release();
	__volumeGraph = NULL;

    if (_volumeGraph == NULL)
        return E_UNEXPECTED;

    _volumeGraph->Release();
    _volumeGraph = NULL;

	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraphProp::OnDeactivate(void)
{
	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeGraphProp::OnApplyChanges()
{
	__volumeGraph->put_DisplayTime(_displayTime);

	setupGraph();

	m_bDirty = FALSE;
	return NOERROR;
}
//------------------------------------------------------------------------------
void
CVolumeGraphProp::setupControls()
{
	// Setup DisplayTime Combo
	HWND item = GetDlgItem(m_Dlg, IDC_DISPLAYTIME_COMBO);
	ComboBox_ResetContent(item);
	ComboBox_AddString(item, "60b");
	ComboBox_AddString(item, "20b");
	ComboBox_AddString(item, "10b");
	ComboBox_AddString(item, "5b");
	ComboBox_AddString(item, "2b");
	ComboBox_AddString(item, "1b");
	ComboBox_SetCurSel(GetDlgItem(m_Dlg, IDC_DISPLAYTIME_COMBO), _displayTime);

	// Setup DisplayPause Check
	Button_SetCheck(GetDlgItem(m_Dlg, IDC_DISPLAYPAUSE_CHECK), (_paused ? TRUE : FALSE));
}
//------------------------------------------------------------------------------
void
CVolumeGraphProp::shutdownControls()
{
	// Shutdown Timer
	KillTimer(m_Dlg, _timer);
}
//------------------------------------------------------------------------------
void
CVolumeGraphProp::createGraphWindow(HWND hWnd)
{
	const flUInt	GRAPHWINDOW_WIDTH		= 320;
	const flUInt	GRAPHWINDOW_HEIGHT		= 160;
	const flInt		GRAPHWINDOW_MINVOLUME	= -3000;

	_graphWindowL = new flGraphWindow(hWnd);
	_graphWindowL->setPosition(10, 55);
	_graphWindowL->setSize(GRAPHWINDOW_WIDTH, GRAPHWINDOW_HEIGHT);
	_graphWindowL->setRange(GRAPHWINDOW_MINVOLUME, 0);

	_graphWindowR = new flGraphWindow(hWnd);
	_graphWindowR->setPosition(10, 240);
	_graphWindowR->setSize(GRAPHWINDOW_WIDTH, GRAPHWINDOW_HEIGHT);
	_graphWindowR->setRange(GRAPHWINDOW_MINVOLUME, 0);
}
//------------------------------------------------------------------------------
void
CVolumeGraphProp::deleteGraphWindow()
{
	delete _graphWindowL;
	delete _graphWindowR;
}
//------------------------------------------------------------------------------
void
CVolumeGraphProp::setupGraph()
{
	const flULong	SleepTimes []	= {   250,   125,    62,   62,   30,   15 };
	const flULong	EvalTimes []	= {   120,   120,    60,   30,   30,   15 };
	const flInt		Steps []		= {   240,   160,   161,   81,   67,   64 };

	// Setup Timer
	KillTimer(m_Dlg, _timer);
	_timer = SetTimer(m_Dlg, 1, SleepTimes[_displayTime], NULL);

	// Setup VolumeGraph Evaluation Time
	_volumeGraph->put_EvaluationTime(EvalTimes[_displayTime]);

	// Setup Steps
	_graphWindowL->clearGraph();
	_graphWindowR->clearGraph();
	_graphWindowL->setStep(Steps[_displayTime]);
	_graphWindowR->setStep(Steps[_displayTime]);
}
//------------------------------------------------------------------------------
void
CVolumeGraphProp::drawGraph()
{
	const flByte COLORS [][3] = {
		{ 0x7f, 0x7f, 0xff },
		{ 0x7f, 0xff, 0x7f },
		{ 0x7f, 0xff, 0xff },
		{ 0xff, 0x7f, 0x7f },
		{ 0xff, 0x7f, 0xff },
		{ 0xff, 0xff, 0x7f },
		{ 0xff, 0xff, 0xff },
	};

	flUInt numInputs;
	_volumeGraph->get_NumInputs(&numInputs);

	_graphWindowL->setNumGraph(numInputs - 1);
	_graphWindowR->setNumGraph(numInputs - 1);

	for(flUInt i = 0; i < numInputs - 1; i++)
	{
		VolumeInfo volume;
		_volumeGraph->get_Volume(i, &volume);

		_graphWindowL->setGraphColor(i, COLORS[i % 7]);
		_graphWindowR->setGraphColor(i, COLORS[i % 7]);

		_graphWindowL->pointGraph(i, volume.left);
		_graphWindowR->pointGraph(i, volume.right);
	}

	_graphWindowL->draw();
	_graphWindowR->draw();
}
//------------------------------------------------------------------------------
