/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#include <flVolumeGraph/flGraphWindow.h>
#include <flBase/flNotify.h>
#include <limits.h>
//------------------------------------------------------------------------------
flGraphWindow::flGraphWindow(HWND parentWindow)
{
	// Create GraphWindow
	HINSTANCE hInstance = GetModuleHandle(NULL);

	WNDCLASSEX winc;
	winc.cbSize			= sizeof(WNDCLASSEX);
	winc.style			= CS_HREDRAW | CS_VREDRAW;
	winc.lpfnWndProc	= windowProc;
	winc.cbClsExtra		= 0;
	winc.cbWndExtra		= 0;
	winc.hInstance		= hInstance;
	winc.hIcon			= LoadIcon(NULL, IDI_APPLICATION);
	winc.hCursor		= LoadCursor(NULL, IDC_ARROW);
	winc.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
	winc.lpszMenuName	= NULL;
	winc.lpszClassName	= TEXT("GRAPHWINDOW");
	winc.hIconSm		= LoadIcon(NULL, IDI_APPLICATION);

	RegisterClassEx(&winc);

	_window = CreateWindow("GRAPHWINDOW",
						"",
						WS_CHILD | WS_VISIBLE,
						0, 0, 100, 100,
						parentWindow,
						NULL,
						hInstance,
						this);

	// Initialize Variables
	_graphs.setGrowSize(16);
	_step				= 100;
	_rangeMin			= 0;
	_rangeMax			= 100;

	_memDC				= NULL;
	_bitmap				= NULL;
	createMemDC(100, 100);
}
//------------------------------------------------------------------------------
flGraphWindow::~flGraphWindow()
{
	DeleteObject(_bitmap);
	DeleteDC(_memDC);

	SendMessage(_window, WM_CLOSE, 0, 0);
}
//------------------------------------------------------------------------------
void
flGraphWindow::setPosition(flInt x, flInt y)
{
	assert(_window != NULL);

	SetWindowPos(_window, HWND_TOP, x, y, 0, 0,
						SWP_NOACTIVATE | SWP_NOSIZE);
}
//------------------------------------------------------------------------------
void
flGraphWindow::getPosition(flInt& x, flInt& y) const
{
	assert(_window != NULL);

	RECT	rect;
	POINT	pt;

	GetWindowRect(_window, &rect);
	pt.x = rect.left;
	pt.y = rect.top;

	HWND parentWindow = GetParent(_window);

	ScreenToClient(parentWindow, &pt);

	x = pt.x;
	y = pt.y;
}
//------------------------------------------------------------------------------
void
flGraphWindow::setSize(flUInt width, flUInt height)
{
	assert(_window != NULL);

	SetWindowPos(_window, HWND_TOP, 0, 0, width, height,
						SWP_NOACTIVATE | SWP_NOMOVE);

	createMemDC(width, height);
}
//------------------------------------------------------------------------------
void
flGraphWindow::getSize(flUInt& width, flUInt& height) const
{
	assert(_window != NULL);

	RECT rect;
	GetWindowRect(_window, &rect);

	width = rect.right - rect.left;
	height = rect.bottom - rect.top;
}
//------------------------------------------------------------------------------
void
flGraphWindow::setStep(flUInt step)
{
	_step = step;
}
//------------------------------------------------------------------------------
flUInt
flGraphWindow::getStep() const
{
	return _step;
}
//------------------------------------------------------------------------------
void
flGraphWindow::setRange(flInt min, flInt max)
{
	_rangeMin = min;
	_rangeMax = max;
}
//------------------------------------------------------------------------------
void
flGraphWindow::getRange(flInt& min, flInt& max) const
{
	min = _rangeMin;
	max = _rangeMax;
}
//------------------------------------------------------------------------------
void
flGraphWindow::setNumGraph(flUInt count)
{
	_graphs.setSize(count);
}
//------------------------------------------------------------------------------
flUInt
flGraphWindow::getNumGraph() const
{
	return _graphs.getSize();
}
//------------------------------------------------------------------------------
void
flGraphWindow::setGraphColor(flUInt index, const flByte* color)
{
	if (_graphs.getSize() <= index)
		return ;
	_graphs[index].setColor(color);
}
//------------------------------------------------------------------------------
const flByte*
flGraphWindow::getGraphColor(flUInt index) const
{
	if (_graphs.getSize() <= index)
		return NULL;
	return _graphs[index].getColor();
}
//------------------------------------------------------------------------------
void
flGraphWindow::pointGraph(flUInt index, flInt value)
{
	if (_graphs.getSize() <= index)
		return ;
	_graphs[index].point(value);
}
//------------------------------------------------------------------------------
void
flGraphWindow::clearGraph(flUInt index)
{
	if (_graphs.getSize() <= index)
		return ;
	_graphs[index].clear();
}
//------------------------------------------------------------------------------
void
flGraphWindow::clearGraph()
{
	for(flUInt i = 0; i < _graphs.getSize(); i++)
		_graphs[i].clear();
}
//------------------------------------------------------------------------------
void
flGraphWindow::draw()
{
	flUInt w, h;
	getSize(w, h);
	RECT rect;
	SetRect(&rect, 0, 0, w, h);

	HBRUSH brush;
	HPEN pen;
	HGDIOBJ oldBrush;
	HGDIOBJ	oldPen;

	//
	// Draw Background
	//
	brush = CreateSolidBrush(RGB(0, 0, 0));
	oldBrush = SelectObject(_memDC, brush);

	FillRect(_memDC, &rect, brush);

	SelectObject(_memDC, oldBrush);
	DeleteObject(brush);

	//
	// Draw Grid
	//

	// Vertical
	const flInt NUMVERT = 20;
	flUInt incH = w / NUMVERT;
	for(flUInt i = 0; i < w; i += incH)
	{
		pen = CreatePen(PS_SOLID, 1, RGB(0, (i % (incH * 2) == 0 ? 100 : 70), 0));
		oldPen = SelectObject(_memDC, pen);

		MoveToEx(_memDC, i, 0, NULL);
		LineTo(_memDC, i, h);

		SelectObject(_memDC, oldPen);
		DeleteObject(pen);
	}

	// Horizontal
	const flInt NUMHORZ = 10;
	flUInt incV = h / NUMHORZ;
	for(flUInt i = 0; i < h; i += incV)
	{
		pen = CreatePen(PS_SOLID, 1, RGB(0, (i % (incV * 2) == 0 ? 100 : 70), 0));
		oldPen = SelectObject(_memDC, pen);

		MoveToEx(_memDC, 0, i, NULL);
		LineTo(_memDC, w, i);

		SelectObject(_memDC, oldPen);
		DeleteObject(pen);
	}

	//
	// Draw Graph
	//
	for(i = 0; i < _graphs.getSize(); i++)
		_graphs[i].draw(_memDC, w, h, _step, _rangeMin, _rangeMax);

	// 
	InvalidateRect(_window, &rect, FALSE);
}
//------------------------------------------------------------------------------
void
flGraphWindow::draw(HWND hwnd)
{
	assert(_window == hwnd);

	PAINTSTRUCT ps;
	HDC hdc = BeginPaint(hwnd, &ps);

	flUInt w, h;
	getSize(w, h);
	BitBlt(hdc, 0, 0, w, h, _memDC, 0, 0, SRCCOPY);

	EndPaint(hwnd, &ps);
}
//------------------------------------------------------------------------------
void
flGraphWindow::createMemDC(flUInt width, flUInt height)
{
	if (_memDC != NULL)
	{
		DeleteObject(_bitmap);
		DeleteDC(_memDC);
	}

    HDC hDC = GetDC(_window);

    _memDC	= CreateCompatibleDC(hDC);
    _bitmap	= CreateCompatibleBitmap(hDC, width, height);
    SelectObject(_memDC, _bitmap);

    ReleaseDC(_window, hDC);
}
//------------------------------------------------------------------------------
LRESULT
flGraphWindow::wm_proc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg)
	{
	case WM_CLOSE:
		DestroyWindow(_window);
		return 0;

	case WM_DESTROY:
		SetWindowLong(_window, GWL_USERDATA, (LONG)NULL);
		_window = NULL;
		return 0;
		
	case WM_PAINT:
		draw(hWnd);
		return 0;
	}

	return DefWindowProc(hWnd, msg, wParam, lParam);
}
//------------------------------------------------------------------------------
LRESULT CALLBACK
flGraphWindow::windowProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	flGraphWindow *win = (flGraphWindow *)GetWindowLong(hWnd, GWL_USERDATA);

	if (msg == WM_CREATE)
	{
		CREATESTRUCT* cs = (CREATESTRUCT *)lParam;
		win	= (flGraphWindow *)(cs->lpCreateParams);
		SetWindowLong(hWnd, GWL_USERDATA, (LONG)(win));
		if (win != NULL)
			win->_window = hWnd;
	}

	if (win != NULL)
		return win->wm_proc(hWnd, msg, wParam, lParam);

	return DefWindowProc(hWnd, msg, wParam, lParam);
}
//------------------------------------------------------------------------------
