/*
**	Copyright (c) 2003-2004 National Institute of Multimedia Education. All rights reserved.
**	2-12 Wakaba, Mihama, Chiba, 261-0014 JAPAN
**	http://www.nime.ac.jp/
*/
//------------------------------------------------------------------------------
#define WIN32_LEAN_AND_MEAN
#include <streams.h>

#include <flVolumeIndicator/VolumeIndicator.h>
#include <flVolumeIndicator/VolumeIndicatorProp.h>
#include <flBase/flSystemTime.h>
#include <flBase/flNotify.h>
#include <math.h>
//------------------------------------------------------------------------------
#pragma warning(disable:4238)  // nonstd extension used: class rvalue used as lvalue
//------------------------------------------------------------------------------
const AMOVIESETUP_MEDIATYPE sudPinTypes =
{
	&MEDIATYPE_Audio,
	&MEDIASUBTYPE_PCM
};
//------------------------------------------------------------------------------
const AMOVIESETUP_PIN psudPins[] =
{
	{
		L"Input",			// String pin name
		FALSE,				// Is it rendered
		FALSE,				// Is it an output
		FALSE,				// Allowed none
		FALSE,				// Allowed many
		&CLSID_NULL,		// Connects to filter
		L"Output",			// Connects to pin
		1,					// Number of types
		&sudPinTypes		// The pin details
	},
	{
		L"Output",			// String pin name
		FALSE,				// Is it rendered
		TRUE,				// Is it an output
		FALSE,				// Allowed none
		FALSE,				// Allowed many
		&CLSID_NULL,		// Connects to filter
		L"Input",			// Connects to pin
		1,					// Number of types
		&sudPinTypes		// The pin details
	}
};
//------------------------------------------------------------------------------
const AMOVIESETUP_FILTER sudVolumeIndicator =
{
	&CLSID_VolumeIndicator,		// Filter CLSID
	L"NIME Volume Indicator",	// Filter name
	MERIT_DO_NOT_USE,			// Its merit
	2,							// Number of pins
	psudPins					// Pin details
};
//------------------------------------------------------------------------------
CFactoryTemplate g_Templates[2] = {
    { L"VolumeIndicator"
    , &CLSID_VolumeIndicator
	, CVolumeIndicator::CreateInstance
    , NULL
    , &sudVolumeIndicator }
  ,
    { L"VolumeIndicatorProp"
    , &CLSID_VolumeIndicatorPropertyPage
    , CVolumeIndicatorProp::CreateInstance }
};
int g_cTemplates = sizeof(g_Templates) / sizeof(g_Templates[0]);
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllMain
//
//------------------------------------------------------------------------------
extern "C" BOOL WINAPI DllEntryPoint(HINSTANCE, ULONG, LPVOID);
BOOL APIENTRY DllMain(HANDLE hModule, DWORD dwReason, LPVOID lpReserved)
{
    return DllEntryPoint((HINSTANCE)(hModule), dwReason, lpReserved);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//
// DllRegisterServer
//
//------------------------------------------------------------------------------
STDAPI DllRegisterServer()
{
    return AMovieDllRegisterServer2(TRUE);
}
//------------------------------------------------------------------------------
//
// DllUnregisterServer
//
//------------------------------------------------------------------------------
STDAPI DllUnregisterServer()
{
    return AMovieDllRegisterServer2(FALSE);
}
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
CUnknown * WINAPI
CVolumeIndicator::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{
	CVolumeIndicator *pNewObject = new CVolumeIndicator(NAME("VolumeIndicator"), punk, phr);
	if (pNewObject == NULL)
	{
		*phr = E_OUTOFMEMORY;
	}
	return pNewObject;
}
//------------------------------------------------------------------------------
CVolumeIndicator::CVolumeIndicator(TCHAR *tszName, LPUNKNOWN punk, HRESULT *phr) :
CTransInPlaceFilter(tszName, punk, CLSID_VolumeIndicator, phr),
CPersistStream(punk, phr)
{
	_absTime		= new flSystemTime();

	_pcmAudio		= new flPCMAudio();
	_evalTime		= 20;
}
//------------------------------------------------------------------------------
CVolumeIndicator::~CVolumeIndicator()
{
	delete _pcmAudio;
	delete _absTime;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeIndicator::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
	CheckPointer(ppv, E_POINTER);

	if (riid == IID_IVolumeIndicator)
	{
		return GetInterface((IVolumeIndicator *)this, ppv);
	}
	else if (riid == IID_IPersistStream)
	{
		return GetInterface((IPersistStream *) this, ppv);
    }
	else if (riid == IID_ISpecifyPropertyPages)
	{
		return GetInterface((ISpecifyPropertyPages *)this, ppv);
	}
	else
	{
		return CTransInPlaceFilter::NonDelegatingQueryInterface(riid, ppv);
	}
}
//------------------------------------------------------------------------------
HRESULT
CVolumeIndicator::SetMediaType(PIN_DIRECTION direction, const CMediaType* pmt)
{
	WAVEFORMATEXTENSIBLE *formatExt = (WAVEFORMATEXTENSIBLE *)pmt->Format();
	_pcmAudio->setFormat(&formatExt->Format);

	return CTransInPlaceFilter::SetMediaType(direction, pmt);
}
//------------------------------------------------------------------------------
HRESULT
CVolumeIndicator::CheckInputType(const CMediaType* mtIn)
{
	if (IsEqualGUID(mtIn->majortype, MEDIATYPE_Audio) &&
		IsEqualGUID(mtIn->subtype, MEDIASUBTYPE_PCM))
		return S_OK;

	return VFW_E_TYPE_NOT_ACCEPTED;
}
//------------------------------------------------------------------------------
// IPersistStream
#define WRITEOUT(var)								\
	hr = pStream->Write(&var, sizeof(var), NULL);	\
	if (FAILED(hr))	return hr;
#define WRITENOUT(var, n)							\
	hr = pStream->Write(var, n, NULL);				\
	if (FAILED(hr)) return hr;
#define READIN(var)									\
	hr = pStream->Read(&var, sizeof(var), NULL);	\
	if (FAILED(hr)) return hr;
#define READNIN(var, n)								\
	hr = pStream->Read(var, n, NULL);				\
	if (FAILED(hr)) return hr;
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeIndicator::GetClassID(CLSID *pClsid)
{
	*pClsid = CLSID_VolumeIndicator;
	return S_OK;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeIndicator::WriteToStream(IStream *pStream)
{
	HRESULT hr;
	WRITEOUT(_evalTime);
	return NOERROR;
}
//------------------------------------------------------------------------------
HRESULT
CVolumeIndicator::ReadFromStream(IStream *pStream)
{
	HRESULT hr;
	READIN(_evalTime);
	return NOERROR;
}
//------------------------------------------------------------------------------
DWORD
CVolumeIndicator::GetSoftwareVersion(void)
{
	return 1;
}
//------------------------------------------------------------------------------
int
CVolumeIndicator::SizeMax()
{
	return 4;
}
//------------------------------------------------------------------------------
// IVolumeIndicator
STDMETHODIMP
CVolumeIndicator::get_Channels(flUShort* channels)
{
	CAutoLock lock(&_filterLock);
	*channels = _pcmAudio->getChannels();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeIndicator::get_BitsPerSample(flUShort* bitsPerSample)
{
	CAutoLock lock(&_filterLock);
	*bitsPerSample = _pcmAudio->getBitsPerSample();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeIndicator::get_SamplesPerSec(flUInt* samplesPerSec)
{
	CAutoLock lock(&_filterLock);
	*samplesPerSec = _pcmAudio->getSamplesPerSec();
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeIndicator::get_Volume(VolumeInfo *volume)
{
	CAutoLock lock(&_filterLock);

	_absTime->update();
	flInt64 timeStamp = _absTime->getTime();

	flInt64 evalTime64 = flInt64(_evalTime) * 10000;
	_pcmAudio->getVolume(*volume, timeStamp - evalTime64, timeStamp);

	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeIndicator::get_EvaluationTime(flULong *time)
{
	CAutoLock lock(&_filterLock);
	*time = _evalTime;
	return NOERROR;
}
//------------------------------------------------------------------------------
STDMETHODIMP
CVolumeIndicator::put_EvaluationTime(flULong time)
{
	CAutoLock lock(&_filterLock);
	_evalTime = time;
	return NOERROR;
}
//------------------------------------------------------------------------------
// ISpecifyPropertyPages
STDMETHODIMP
CVolumeIndicator::GetPages(CAUUID *pPages)
{
	pPages->cElems = 1;
	pPages->pElems = (GUID *)CoTaskMemAlloc(sizeof(GUID) * 1);
	if (pPages->pElems == NULL)
	{
		return E_OUTOFMEMORY;
	}
	pPages->pElems[0] = CLSID_VolumeIndicatorPropertyPage;
	return NOERROR;
}
//------------------------------------------------------------------------------
// CTransformInPlaceFilter Overrides
HRESULT
CVolumeIndicator::Transform(IMediaSample *pSample)
{
	flByte* data;
	pSample->GetPointer(&data);
	flLong dataLength = pSample->GetActualDataLength();

	CAutoLock lock(&_filterLock);

	_absTime->update();
	_pcmAudio->setData(data, dataLength, _absTime->getTime());

	return NOERROR;
}
//------------------------------------------------------------------------------
