//------------------------------------------------------------------------------
#include <cstdio>
#include <cstring>
//------------------------------------------------------------------------------
#define	TAGSIZE			12
#define	SWAP_SH(_x)		((((_x)&0xff)<<8)|(((_x)&0xff00)>>8))
#define	SWAP_LG(_x)		(((((_x)&0xff)<<24)|(((_x)&0xff00)<<8))|\
						((((_x)&0xff0000)>>8)|(((_x)&0xff000000)>>24)))
//------------------------------------------------------------------------------
struct tiff_rec
{
	tiff_rec()
	{
		memset(this, 0, sizeof(tiff_rec));
	}

	int			big_endian;
	short		version;
	long		ifd;
	long		position;
	short		width;
	short		height;
	short		components;
	short		compress;
	FILE*		fp;
};
//------------------------------------------------------------------------------
static void
read_tag(tiff_rec& rec,
         unsigned short& id,
         unsigned short& type,
         unsigned long& length)
{
	unsigned long	value = 0;

	if (rec.position != 0)
		fseek(rec.fp, rec.position, SEEK_SET);
	rec.position = ftell(rec.fp) + TAGSIZE;

	fread(&id, 2, 1, rec.fp);
	if (rec.big_endian)
		id = SWAP_SH(id);

	fread(&type, 2, 1, rec.fp);
	if (rec.big_endian)
		type = SWAP_SH(type);

	fread(&length, 4, 1, rec.fp);
	if (rec.big_endian)
		length = SWAP_LG(length);

	switch (type)
	{
	case 3:
		if (length > 2)
		{
			fread(&value, 4, 1, rec.fp);
			if (rec.big_endian)
				value = SWAP_LG(value);
		}
		break;

	case 4:
		if (length > 1)
		{
			fread(&value, 4, 1, rec.fp);
			if (rec.big_endian)
				value = SWAP_LG(value);
		}
		break;

	case 5:
		fread(&value, 4, 1, rec.fp);
		if (rec.big_endian)
			value = SWAP_LG(value);
		break;
	}

	if (value != 0)
		fseek(rec.fp, value, SEEK_SET);
}
//------------------------------------------------------------------------------
bool
read_tiff(FILE* fp,
          unsigned int& width,
          unsigned int& height,
          unsigned int& components,
          unsigned char*& image)
{
	int				i;
	unsigned short	j;
	unsigned short	s;
	unsigned long	l;
	char			code[2];
	unsigned short	tagno, id, type;
	unsigned long	length, offset;
	unsigned char*	dst;
	unsigned int	h, c;

	if (fp == NULL)
		return false;

	tiff_rec	rec;
	rec.fp		= fp;

	width		= 0;
	height		= 0;
	components	= 0;
	image		= NULL;

	// read header
	// byte order
	fread(code, 1, 2, rec.fp);
	if ((code[0] == 0x4d) && (code[1] == 0x4d))
		rec.big_endian = true;		// big endian
	else
		rec.big_endian = false;		// little endian

	// veraion
	fread(&rec.version, 2, 1, rec.fp);
	if (rec.big_endian)
		rec.version = SWAP_SH(rec.version);

	// first IFD pointer
	fread(&rec.ifd, 4, 1, rec.fp);
	if (rec.big_endian)
		rec.ifd = SWAP_LG(rec.ifd);

	// to IFD position
	fseek(rec.fp, rec.ifd, SEEK_SET);

	// TAG

	fread(&tagno, 2, 1, rec.fp);
	if (rec.big_endian)
		tagno = SWAP_SH(tagno);

	for (i = 0; i < tagno; i++)
	{
		read_tag(rec, id, type, length);

		switch (id) {
		case 0x100:	// image width
			fread(&rec.width, 2, 1, rec.fp);
			if (rec.big_endian)
				rec.width = SWAP_SH(rec.width);
			break;

		case 0x101:	// image length
			fread(&rec.height, 2, 1, rec.fp);
			if (rec.big_endian)
				rec.height = SWAP_SH(rec.height);
			break;

		case 0x102:	// bits per sample
			for (j = 0; j < length; j++)
				fread(&s, 2, 1, rec.fp);
			break;

		case 0x103:	// data compression
			fread(&rec.compress, 2, 1, rec.fp);
			if (rec.big_endian)
				rec.compress = SWAP_SH(rec.compress);
			if (rec.compress != 1)
				throw 0;
			break;

		case 0x111:	// strip offset
			fread(&offset, 4, 1, rec.fp);
			if (rec.big_endian)
				offset = SWAP_LG(offset);
			break;

		case 0x115:	// sample per pixel
			fread(&rec.components, 2, 1, rec.fp);
			if (rec.big_endian)
				rec.components = SWAP_SH(rec.components);
			break;

		case 0x106:	// photmetricinterp
			fread(&s, 2, 1, rec.fp);
			break;

		case 0x117:	// strip byte counts
			fread(&l, 4, 1, rec.fp);
			break;

		case 0x11A:	// X resolution
		case 0x11B:	// Y resolution
			fread(&l, 4, 1, rec.fp);
			fread(&l, 4, 1, rec.fp);
			break;

		case 0x11C:	// planar configuration
			fread(&s, 2, 1, rec.fp);
			break;

		default:	// another unknown tags
			switch (type)
			{
		  	case 1:
		  	case 2:
		  	case 3:
				fread(&s, 2, 1, rec.fp);
				break;
			case 4:
			case 5:
				fread(&l, 4, 1, rec.fp);
				break;
			}
			break;
		}
	}

	fseek(rec.fp, 2 + TAGSIZE * tagno, SEEK_SET);
	fread(&l, 4, 1, rec.fp);

	fseek(rec.fp, offset, SEEK_SET);

	// image
	width		= rec.width;
	height		= rec.height;
	components	= rec.components;
	image		= new unsigned char[width * height * components];

	dst			= image;

	for (h = 0; h < height; ++h)
	{
		dst = &image[(height-h-1) * width * components];
		for (c = 0; c < width * components; ++c)
			*dst++	= (unsigned char)getc(rec.fp);
	}

	return true;
}
//------------------------------------------------------------------------------
